/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.agent.language;

import com.newrelic.agent.Agent;
import com.newrelic.agent.deps.com.google.common.annotations.VisibleForTesting;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.stats.StatsWorks;
import java.lang.reflect.Field;
import java.text.MessageFormat;
import java.util.logging.Level;

public class SourceLibraryDetector
implements Runnable {
    private static final String SCALA_VERSION_CLASS = "scala.util.Properties";
    private static final String SCALA_VERSION_METHOD = "versionNumberString";
    private static final String KOTLIN_VERSION_CLASS = "com.newrelic.agent.deps.kotlin.KotlinVersion";
    private static final String KOTLIN_VERSION_FIELD = "CURRENT";
    private static final String CLOJURE_VERSION_CLASS = "clojure.core$clojure_version";
    private static final String CLOJURE_VERSION_METHOD = "invoke";
    private boolean done = false;

    @Override
    public void run() {
        if (this.done) {
            return;
        }
        if (Agent.LOG.isFinerEnabled()) {
            Agent.LOG.log(Level.FINER, "Obtaining source library language supportability metrics");
        }
        try {
            this.detectSourceLanguages();
        }
        catch (Throwable e) {
            Agent.LOG.log(Level.FINER, e, "Unexpected error attempting to find source languages");
        }
        finally {
            this.done = true;
        }
    }

    public boolean isDone() {
        return this.done;
    }

    @VisibleForTesting
    void detectSourceLanguages() {
        ClassLoader systemClassLoader = ClassLoader.getSystemClassLoader();
        this.detectJava();
        this.detectScala(systemClassLoader);
        this.detectKotlin(systemClassLoader);
        this.detectClojure(systemClassLoader);
    }

    private void detectJava() {
        String javaVersion = System.getProperty("java.specification.version");
        this.recordSourceLanguageMetric("java", javaVersion);
        String jvmVendor = System.getProperty("java.vm.name").toLowerCase();
        if (jvmVendor.contains("hotspot")) {
            this.recordJvmVendorMetric("Oracle");
        } else if (jvmVendor.contains("ibm")) {
            this.recordJvmVendorMetric("IBM");
        } else if (jvmVendor.contains("azul") || jvmVendor.contains("zing") || jvmVendor.contains("zulu")) {
            this.recordJvmVendorMetric("Azul");
        } else if (jvmVendor.contains("eclipse")) {
            this.recordJvmVendorMetric("Eclipse");
        } else if (jvmVendor.contains("openjdk")) {
            this.recordJvmVendorMetric("OpenJDK");
        } else if (jvmVendor.contains("corretto")) {
            this.recordJvmVendorMetric("Corretto");
        } else {
            this.recordJvmVendorMetric("Other");
        }
    }

    private void detectScala(ClassLoader systemClassLoader) {
        try {
            Class<?> aClass = Class.forName(SCALA_VERSION_CLASS, true, systemClassLoader);
            if (aClass != null) {
                String version = (String)aClass.getMethod(SCALA_VERSION_METHOD, new Class[0]).invoke(null, new Object[0]);
                this.recordSourceLanguageMetric("scala", version);
            }
        }
        catch (ClassNotFoundException cfne) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Class ''{0}'' was not found; scala is not present in the classpath", SCALA_VERSION_CLASS));
        }
        catch (NoSuchMethodException nfme) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Method ''{0}'' in class ''{1}'' was not found; cannot determine scala version", SCALA_VERSION_METHOD, SCALA_VERSION_CLASS));
        }
        catch (Exception e) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Exception occurred while trying to determine scala version", e.getMessage()));
        }
    }

    private void detectKotlin(ClassLoader systemClassLoader) {
        try {
            Class<?> aClass = Class.forName(KOTLIN_VERSION_CLASS, true, systemClassLoader);
            if (aClass != null) {
                Field field = aClass.getField(KOTLIN_VERSION_FIELD);
                String version = field.get(null).toString();
                this.recordSourceLanguageMetric("com.newrelic.agent.deps.kotlin", version);
            }
        }
        catch (ClassNotFoundException cfne) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Class ''{0}'' was not found; kotlin is not present in the classpath", KOTLIN_VERSION_CLASS));
        }
        catch (NoSuchFieldException nfme) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Field ''{0}'' in class ''{1}'' was not found; cannot determine kotlin version", KOTLIN_VERSION_FIELD, KOTLIN_VERSION_CLASS));
        }
        catch (Exception e) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Exception occurred while trying to determine kotlin version", e.getMessage()));
        }
    }

    private void detectClojure(ClassLoader systemClassLoader) {
        try {
            Class<?> aClass = Class.forName(CLOJURE_VERSION_CLASS, true, systemClassLoader);
            if (aClass != null) {
                Object clojureVersion = aClass.newInstance();
                String version = aClass.getMethod(CLOJURE_VERSION_METHOD, new Class[0]).invoke(clojureVersion, new Object[0]).toString();
                this.recordSourceLanguageMetric("clojure", version);
            }
        }
        catch (ClassNotFoundException cfne) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Class ''{0}'' was not found; clojure is not present in the classpath", CLOJURE_VERSION_CLASS));
        }
        catch (NoSuchMethodException nfme) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Method ''{0}'' in class ''{1}'' was not found; cannot determine clojure version", CLOJURE_VERSION_METHOD, CLOJURE_VERSION_CLASS));
        }
        catch (Exception e) {
            Agent.LOG.log(Level.FINEST, MessageFormat.format("Exception occurred while trying to determine clojure version", e.getMessage()));
        }
    }

    private void recordSourceLanguageMetric(String language, String version) {
        ServiceFactory.getStatsService().doStatsWork(StatsWorks.getRecordMetricWork(MessageFormat.format("Supportability/SourceLanguage/{0}/{1}", language, version), 1.0f));
    }

    private void recordJvmVendorMetric(String vendor) {
        ServiceFactory.getStatsService().doStatsWork(StatsWorks.getRecordMetricWork(MessageFormat.format("Supportability/Jvm/Vendor/{0}", vendor), 1.0f));
    }
}

