/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject;

import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.sys.businessobject.FinancialSystemDocumentHeader;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;

public class NonAppliedHolding extends PersistableBusinessObjectBase {

    private String referenceFinancialDocumentNumber;
    private KualiDecimal financialDocumentLineAmount = KualiDecimal.ZERO;
    private String customerNumber;
    private Customer customer;
    private KualiDecimal availableUnappliedAmount = KualiDecimal.ZERO;
    private KualiDecimal appliedUnappliedAmount = KualiDecimal.ZERO;
    private Collection<NonInvoicedDistribution> nonInvoicedDistributions;
    private Collection<NonAppliedDistribution> nonAppliedDistributions;
    private FinancialSystemDocumentHeader documentHeader;

    /**
     * Default constructor.
     */
    public NonAppliedHolding() {
        nonAppliedDistributions = new ArrayList<NonAppliedDistribution>();
    }

    /**
     * Gets the referenceFinancialDocumentNumber attribute.
     *
     * @return Returns the referenceFinancialDocumentNumber
     */
    public String getReferenceFinancialDocumentNumber() {
        return referenceFinancialDocumentNumber;
    }

    /**
     * Sets the referenceFinancialDocumentNumber attribute.
     *
     * @param referenceFinancialDocumentNumber The referenceFinancialDocumentNumber to set.
     */
    public void setReferenceFinancialDocumentNumber(String referenceFinancialDocumentNumber) {
        this.referenceFinancialDocumentNumber = referenceFinancialDocumentNumber;
    }


    /**
     * Gets the financialDocumentLineAmount attribute.
     *
     * @return Returns the financialDocumentLineAmount
     */
    public KualiDecimal getFinancialDocumentLineAmount() {
        return financialDocumentLineAmount;
    }

    /**
     * Sets the financialDocumentLineAmount attribute.
     *
     * @param financialDocumentLineAmount The financialDocumentLineAmount to set.
     */
    public void setFinancialDocumentLineAmount(KualiDecimal financialDocumentLineAmount) {
        this.financialDocumentLineAmount = financialDocumentLineAmount;
    }


    /**
     * Gets the customerNumber attribute.
     *
     * @return Returns the customerNumber
     */
    public String getCustomerNumber() {
        return customerNumber;
    }

    /**
     * Sets the customerNumber attribute.
     *
     * @param customerNumber The customerNumber to set.
     */
    public void setCustomerNumber(String customerNumber) {
        this.customerNumber = customerNumber;
    }


    /**
     * Gets the customer attribute.
     *
     * @return Returns the customer
     */
    public Customer getCustomer() {
        return customer;
    }

    /**
     * Sets the customer attribute.
     *
     * @param customer The customer to set.
     * @deprecated
     */
    public void setCustomer(Customer customer) {
        this.customer = customer;
    }

    /**
     * Gets the nonInvoicedDistributions attribute.
     *
     * @return Returns the nonInvoicedDistributions.
     */
    public Collection<NonInvoicedDistribution> getNonInvoicedDistributions() {
        return nonInvoicedDistributions;
    }

    /**
     * Sets the nonInvoicedDistributions attribute value.
     *
     * @param nonInvoicedDistributions The nonInvoicedDistributions to set.
     */
    public void setNonInvoicedDistributions(Collection<NonInvoicedDistribution> nonInvoicedDistributions) {
        this.nonInvoicedDistributions = nonInvoicedDistributions;
    }

    /**
     * Gets the nonAppliedDistributions attribute.
     *
     * @return Returns the nonAppliedDistributions.
     */
    public Collection<NonAppliedDistribution> getNonAppliedDistributions() {
        return nonAppliedDistributions;
    }

    /**
     * Sets the nonAppliedDistributions attribute value.
     *
     * @param nonAppliedDistributions The nonAppliedDistributions to set.
     */
    public void setNonAppliedDistributions(List<NonAppliedDistribution> nonAppliedDistributions) {
        this.nonAppliedDistributions = nonAppliedDistributions;
    }

    /**
     * @see org.kuali.rice.krad.bo.BusinessObjectBase#toStringMapper()
     */
    protected LinkedHashMap toStringMapper_RICE20_REFACTORME() {
        LinkedHashMap m = new LinkedHashMap();
        m.put("referenceFinancialDocumentNumber", this.referenceFinancialDocumentNumber);
        return m;
    }

    /**
     * Gets the availableUnappliedAmount attribute.
     *
     * @return Returns the availableUnappliedAmount.
     */
    public KualiDecimal getAvailableUnappliedAmount() {
        //  start with the original unapplied amount
        KualiDecimal amount = financialDocumentLineAmount;

        //  subtract any non-invoiced distributions made against it
        for (NonInvoicedDistribution nonInvoicedDistribution : nonInvoicedDistributions) {
            amount = amount.subtract(nonInvoicedDistribution.getFinancialDocumentLineAmount());
        }

        //  subtract any non-applied distributions made against it
        for (NonAppliedDistribution nonAppliedDistribution : nonAppliedDistributions) {
            amount = amount.subtract(nonAppliedDistribution.getFinancialDocumentLineAmount());
        }
        return amount;
    }

    /**
     * Gets the appliedUnappliedAmount attribute.
     *
     * @return Returns the appliedUnappliedAmount.
     */
    public KualiDecimal getAppliedUnappliedAmount() {
        //  start with zero
        KualiDecimal amount = KualiDecimal.ZERO;

        //  add any non-invoiced distributions made against it
        for (NonInvoicedDistribution nonInvoicedDistribution : nonInvoicedDistributions) {
            amount = amount.add(nonInvoicedDistribution.getFinancialDocumentLineAmount());
        }

        //  add any non-applied distributions made against it
        for (NonAppliedDistribution nonAppliedDistribution : nonAppliedDistributions) {
            amount = amount.add(nonAppliedDistribution.getFinancialDocumentLineAmount());
        }
        return amount;
    }

    /**
     * Sets the documentHeader attribute value.
     *
     * @param documentHeader.
     */
    public FinancialSystemDocumentHeader getDocumentHeader() {
        return documentHeader;
    }

    /**
     * Sets the documentHeader attribute value.
     *
     * @param documentHeader.
     */
    public void setDocumentHeader(FinancialSystemDocumentHeader documentHeader) {
        this.documentHeader = documentHeader;
    }

}
