/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import org.apache.commons.beanutils.converters.BooleanConverter;
import org.apache.commons.beanutils.converters.SqlDateConverter;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.util.KNSGlobalVariables;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.util.ErrorMessage;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.Bill;
import org.kuali.kfs.module.ar.businessobject.PredeterminedBillingSchedule;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.kew.api.exception.WorkflowException;
import org.springframework.util.AutoPopulatingList;

import java.sql.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class PredeterminedBillingScheduleCsvInputFileType extends ScheduleCsvInputFileType {

    @Override
    protected Object convertParsedObjectToVO(Object parsedContent) {
        List<PredeterminedBillingSchedule> predeterminedBillingSchedules = new LinkedList<>();
        List<Map<String, String>> parseDataList = (List<Map<String, String>>) parsedContent;
        for (Map<String, String> row : parseDataList) {
            String proposalNumber = row.get(KFSPropertyConstants.PROPOSAL_NUMBER);
            String chartOfAccountsCode = row.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
            String accountNumber = row.get(KFSPropertyConstants.ACCOUNT_NUMBER);

            if (StringUtils.isEmpty(chartOfAccountsCode)) {
                chartOfAccountsCode = deriveChartOfAccountsCodeFromAccount(accountNumber);
            }

            PredeterminedBillingSchedule currentSchedule;

            try {
                currentSchedule = new PredeterminedBillingSchedule.PredeterminedBillingScheduleBuilder()
                        .setProposalNumber(proposalNumber)
                        .setChartOfAccountsCode(chartOfAccountsCode)
                        .setAccountNumber(accountNumber)
                        .addBill(createBill(proposalNumber, chartOfAccountsCode, accountNumber, row))
                        .build();
            } catch (IllegalStateException ise) {
                GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                        KFSKeyConstants.ERROR_BATCH_UPLOAD_PARSING, ise.getMessage());
                return null;
            }

            if (predeterminedBillingSchedules.contains(currentSchedule)) {
                PredeterminedBillingSchedule existingSchedule = predeterminedBillingSchedules
                        .get(predeterminedBillingSchedules.indexOf(currentSchedule));
                existingSchedule.addBill(currentSchedule.getBills().get(0));
            } else {
                predeterminedBillingSchedules.add(currentSchedule);
            }
        }

        return predeterminedBillingSchedules;
    }

    private Bill createBill(String proposalNumber, String chartOfAccountsCode, String accountNumber,
            Map<String, String> row) {
        String billNumber = row.get(ArPropertyConstants.BillFields.BILL_NUMBER);
        String billDescription = row.get(ArPropertyConstants.BillFields.BILL_DESCRIPTION);

        final String estimatedAmountString = row.get(ArPropertyConstants.BillFields.ESTIMATED_AMOUNT);
        KualiDecimal estimatedAmount = null;
        if (StringUtils.isNotBlank(estimatedAmountString)) {
            estimatedAmount = new KualiDecimal(estimatedAmountString);
        }

        SqlDateConverter sqlDateConverter = new SqlDateConverter();
        String billDateString = row.get(ArPropertyConstants.BillFields.BILL_DATE);
        Date billDate = null;
        if (billDateString.length() > 0) {
            billDate = sqlDateConverter.convert(Date.class, billDateString);
        }

        BooleanConverter converter = new BooleanConverter(Boolean.FALSE);
        boolean active = converter.convert(Boolean.TYPE, row.get(KFSPropertyConstants.ACTIVE));

        return new Bill.BillBuilder()
                .setProposalNumber(proposalNumber)
                .setChartOfAccountsCode(chartOfAccountsCode)
                .setAccountNumber(accountNumber)
                .setActive(active)
                .setBillNumber(billNumber)
                .setBillDescription(billDescription)
                .setBillDate(billDate)
                .setEstimatedAmount(estimatedAmount)
                .build();
    }

    @Override
    public String getFileTypeIdentifier() {
        return ArConstants.PredeterminedBillingScheduleImport.FILE_TYPE_IDENTIFIER;
    }

    @Override
    public void process(String fileName, Object parsedFileContents) {
        List<PredeterminedBillingSchedule> predeterminedBillingSchedules =
                (List<PredeterminedBillingSchedule>) parsedFileContents;

        for (PredeterminedBillingSchedule predeterminedBillingSchedule: predeterminedBillingSchedules) {
            MaintenanceDocument predeterminedBillingScheduleDocument;
            try {
                predeterminedBillingScheduleDocument = (MaintenanceDocument) documentService.getNewDocument(
                        maintenanceDocumentDictionaryService.getDocumentTypeName(PredeterminedBillingSchedule.class));

            } catch (WorkflowException we) {
                LOG.error("process(...) - encountered WorkflowException while attempting to get a new " +
                        "predetermined billing schedule document.", we);
                GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                        ArKeyConstants.ERROR_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_NONRECOVERABLE);
                break;
            }

            predeterminedBillingScheduleDocument.getDocumentHeader()
                    .setDocumentDescription("Predetermined Billing Schedule Import");

            PredeterminedBillingSchedule existingPredeterminedBillingSchedule =
                    getPredeterminedBillingSchedule(predeterminedBillingSchedule);
            if (ObjectUtils.isNull(existingPredeterminedBillingSchedule)) {
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setMaintenanceAction(KRADConstants.MAINTENANCE_NEW_ACTION);
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setBusinessObject(predeterminedBillingSchedule);
                predeterminedBillingScheduleDocument.getOldMaintainableObject()
                        .setBusinessObject(new PredeterminedBillingSchedule());
            } else {
                PredeterminedBillingSchedule copyOfExistingPredeterminedBillingSchedule =
                        (PredeterminedBillingSchedule)ObjectUtils.deepCopy(existingPredeterminedBillingSchedule);
                predeterminedBillingSchedule.getBills().forEach(bill -> {
                    copyOfExistingPredeterminedBillingSchedule.addBill(bill);
                    // add empty bill to collection so that bill collection sizes on old and new
                    // will match. yes this is dumb. see FieldUtils.meshFields (src of stupid)
                    existingPredeterminedBillingSchedule.addBill(new Bill());
                });
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setBusinessObject(copyOfExistingPredeterminedBillingSchedule);
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setMaintenanceAction(KRADConstants.MAINTENANCE_EDIT_ACTION);
                predeterminedBillingScheduleDocument.getOldMaintainableObject()
                        .setBusinessObject(existingPredeterminedBillingSchedule);
            }

            try {
                documentService.saveDocument(predeterminedBillingScheduleDocument);
                documentService.routeDocument(predeterminedBillingScheduleDocument,
                        "Routed New/Edit Predetermined Billing Schedule Maintenance Document from " +
                                "Predetermined Billing Schedule Import Process",
                        null);
            } catch (ValidationException ve) {
                // ValidationException(s) that are mapped to GLOBAL_ERRORS may prevent the document from being saved
                // (e.g. when there is another maintenance document that has a lock on the same key as the given
                // document). In those cases we want to display an appropriate message to the user.
                final Map<String, AutoPopulatingList<ErrorMessage>> errorMessages = GlobalVariables.getMessageMap()
                        .getErrorMessages();
                errorMessages.keySet().stream()
                        .filter(key -> key.equals(KFSConstants.GLOBAL_ERRORS))
                        .forEach(key -> KNSGlobalVariables.getMessageList()
                                .add(ArKeyConstants.ERROR_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_SAVE_FAILURE,
                                predeterminedBillingSchedule.getProposalNumber(),
                                        predeterminedBillingSchedule.getChartOfAccountsCode(),
                                        predeterminedBillingSchedule.getAccountNumber(),
                                        flattenErrorMessages(errorMessages.get(key))));
                // We have elevated all the error messages we need to relay to the user, so we can clear them out.
                GlobalVariables.getMessageMap().clearErrorMessages();
                // no additional action is necessary if routing caused a ValidationException; in this case we want to
                // continue potentially processing additionally predetermined billing schedules and leave the in error
                // document in saved state
            } catch (WorkflowException we) {
                LOG.error("process(...) - encountered WorkflowException while attempting to save or route a " +
                        "predetermined billing schedule document.", we);
                GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                        ArKeyConstants.ERROR_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_NONRECOVERABLE);
                break;
            }
        }

        // used this method so that messaging would go to same place as the messaging coming from batch save process
        KNSGlobalVariables.getMessageList().add(
                ArKeyConstants.MESSAGE_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_SUCCESSFUL);
        removeFiles(fileName);
    }

    private PredeterminedBillingSchedule getPredeterminedBillingSchedule(
            PredeterminedBillingSchedule predeterminedBillingSchedule) {
        Map<String, Object> primaryKeys = new HashMap<>();
        primaryKeys.put(KFSPropertyConstants.PROPOSAL_NUMBER, predeterminedBillingSchedule.getProposalNumber());
        primaryKeys.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE,
                predeterminedBillingSchedule.getChartOfAccountsCode());
        primaryKeys.put(KFSPropertyConstants.ACCOUNT_NUMBER, predeterminedBillingSchedule.getAccountNumber());
        return businessObjectService.findByPrimaryKey(PredeterminedBillingSchedule.class, primaryKeys);
    }

    @Override
    public String getTitleKey() {
        return ArKeyConstants.MESSAGE_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_TITLE;
    }

}
