/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.service.AccountingPeriodService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAwardAccount;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.batch.service.VerifyBillingFrequencyService;
import org.kuali.kfs.module.ar.businessobject.Bill;
import org.kuali.kfs.module.ar.businessobject.BillingFrequency;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.businessobject.InvoiceAccountDetail;
import org.kuali.kfs.module.ar.businessobject.Milestone;
import org.kuali.kfs.module.ar.businessobject.OrganizationAccountingDefault;
import org.kuali.kfs.module.ar.businessobject.SystemInformation;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsBillingAwardVerificationService;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.ChartOrgHolder;
import org.kuali.kfs.sys.document.service.FinancialSystemDocumentService;
import org.kuali.kfs.sys.service.FinancialSystemUserService;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.sql.Date;
import java.time.LocalDate;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class ContractsGrantsBillingAwardVerificationServiceImpl implements ContractsGrantsBillingAwardVerificationService {
    protected AccountingPeriodService accountingPeriodService;
    protected BusinessObjectService businessObjectService;
    protected ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService;
    protected CustomerService customerService;
    protected FinancialSystemDocumentService financialSystemDocumentService;
    protected KualiModuleService kualiModuleService;
    protected ParameterService parameterService;
    protected VerifyBillingFrequencyService verifyBillingFrequencyService;
    protected UniversityDateService universityDateService;
    protected OptionsService optionsService;
    protected FinancialSystemUserService financialSystemUserService;

    /**
     * Check if the value of the billing frequency code is in the BillingFrequency value set.
     *
     * @param award
     * @return
     */
    @Override
    public boolean isValueOfBillingFrequencyValid(ContractsAndGrantsBillingAward award) {
        if (!StringUtils.isBlank(award.getBillingFrequencyCode())) {
            Map<String, Object> criteria = new HashMap<>();
            criteria.put(KFSPropertyConstants.FREQUENCY, award.getBillingFrequencyCode());
            criteria.put(KFSPropertyConstants.ACTIVE, true);
            final int billingFrequencyCount = businessObjectService.countMatching(BillingFrequency.class, criteria);

            return (billingFrequencyCount > 0);
        }

        return false;
    }

    @Override
    public boolean isAwardFinalInvoiceAlreadyBuilt(ContractsAndGrantsBillingAward award) {
        for (ContractsAndGrantsBillingAwardAccount awardAccount : award.getActiveAwardAccounts()) {
            if (awardAccount.isFinalBilledIndicator()) {
                return true;
            }
        }

        return false;
    }

    /**
     * this method checks If all accounts of award has invoices in progress.
     *
     * @param award
     * @return
     */
    @Override
    public boolean isInvoiceInProgress(ContractsAndGrantsBillingAward award) {
        Map<String, Object> fieldValues = new HashMap<>();
        fieldValues.put(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields.PROPOSAL_NUMBER, award.getProposalNumber());
        fieldValues.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_DOCUMENT_STATUS_CODE, getFinancialSystemDocumentService().getPendingDocumentStatuses());

        Collection<ContractsGrantsInvoiceDocument> invoiceDocuments =
                businessObjectService.findMatching(ContractsGrantsInvoiceDocument.class, fieldValues);

        for (ContractsGrantsInvoiceDocument invoiceDocument: invoiceDocuments) {
            for (InvoiceAccountDetail accountDetail: invoiceDocument.getAccountDetails()) {
                if (hasMatchingAccount(award, accountDetail)) {
                    return true;
                }
            }
        }

        return false;
    }

    private boolean hasMatchingAccount(ContractsAndGrantsBillingAward award, InvoiceAccountDetail accountDetail) {
        for (ContractsAndGrantsBillingAwardAccount awardAccount: award.getActiveAwardAccounts()) {
            if (StringUtils.equals(accountDetail.getChartOfAccountsCode(), awardAccount.getChartOfAccountsCode())
                    && StringUtils.equals(accountDetail.getAccountNumber(), awardAccount.getAccountNumber())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean isAwardAccountValidToInvoiceBasedOnSchedule(ContractsAndGrantsBillingAwardAccount awardAccount) {
        if (ArConstants.BillingFrequencyValues.isMilestone(awardAccount.getAward())) {
            return hasMilestonesToInvoice(awardAccount);
        }

        if (ArConstants.BillingFrequencyValues.isPredeterminedBilling(awardAccount.getAward())) {
            return hasBillsToInvoice(awardAccount);
        }

        return true;
    }

    @Override
    public boolean hasMilestonesToInvoice(ContractsAndGrantsBillingAward award) {
        if (ArConstants.BillingFrequencyValues.isMilestone(award)) {
            for (ContractsAndGrantsBillingAwardAccount awardAccount: award.getActiveAwardAccounts()) {
                if (hasMilestonesToInvoice(awardAccount)) {
                    return true;
                }
            }
        }

        return false;
    }

    private boolean hasMilestonesToInvoice(ContractsAndGrantsBillingAwardAccount awardAccount) {
        Map<String, Object> map = new HashMap<>();
        map.put(KFSPropertyConstants.PROPOSAL_NUMBER, awardAccount.getProposalNumber());
        map.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, awardAccount.getChartOfAccountsCode());
        map.put(KFSPropertyConstants.ACCOUNT_NUMBER, awardAccount.getAccountNumber());
        map.put(KFSPropertyConstants.ACTIVE, true);

        Collection<Milestone> milestones = businessObjectService.findMatching(Milestone.class, map);

        List<Milestone> milestonesToInvoice = determineMilestonesToInvoice(milestones);

        return CollectionUtils.isNotEmpty(milestonesToInvoice);
    }

    public List<Milestone> determineMilestonesToInvoice(Collection<Milestone> milestones) {
        Date today = Date.valueOf(LocalDate.now());

        return milestones.stream()
                .filter(milestone -> canBeInvoiced(milestone, today))
                .collect(Collectors.toList());
    }

    private boolean canBeInvoiced(Milestone milestone, Date today) {
        return milestone.getMilestoneActualCompletionDate() != null
                && !milestone.getMilestoneActualCompletionDate().after(today) && !milestone.isBilled()
                && ObjectUtils.isNotNull(milestone.getMilestoneAmount())
                && milestone.getMilestoneAmount().isGreaterThan(KualiDecimal.ZERO);
    }

    @Override
    public boolean hasBillsToInvoice(ContractsAndGrantsBillingAward award) {
        if (ArConstants.BillingFrequencyValues.isPredeterminedBilling(award)) {
            for (ContractsAndGrantsBillingAwardAccount awardAccount: award.getActiveAwardAccounts()) {
                if (hasBillsToInvoice(awardAccount)) {
                    return true;
                }
            }
        }

        return false;
    }

    private boolean hasBillsToInvoice(ContractsAndGrantsBillingAwardAccount awardAccount) {
        Map<String, Object> map = new HashMap<>();
        map.put(KFSPropertyConstants.PROPOSAL_NUMBER, awardAccount.getProposalNumber());
        map.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, awardAccount.getChartOfAccountsCode());
        map.put(KFSPropertyConstants.ACCOUNT_NUMBER, awardAccount.getAccountNumber());
        map.put(KFSPropertyConstants.ACTIVE, true);

        Collection<Bill> bills = businessObjectService.findMatching(Bill.class, map);

        List<Bill> billsToInvoice = determineBillsToInvoice(bills);

        return CollectionUtils.isNotEmpty(billsToInvoice);
    }

    public List<Bill> determineBillsToInvoice(Collection<Bill> bills) {
        Date today = Date.valueOf(LocalDate.now());

        return bills.stream()
                .filter(bill -> canBeInvoiced(bill, today))
                .collect(Collectors.toList());
    }

    private boolean canBeInvoiced(Bill bill, Date today) {
        return bill.getBillDate() != null && !bill.getBillDate().after(today) && !bill.isBilled()
                && bill.getEstimatedAmount().isGreaterThan(KualiDecimal.ZERO);
    }

    @Override
    public boolean owningAgencyHasCustomerRecord(ContractsAndGrantsBillingAward award) {
        if (ObjectUtils.isNotNull(award.getAgency().getCustomerNumber())) {
            Customer customer = customerService.getByPrimaryKey(award.getAgency().getCustomerNumber());
            return !ObjectUtils.isNull(customer);
        }

        return false;
    }

    /**
     * This method checks if the System Information and Organization Accounting Default are setup for the Chart Code
     * and Org Code from the award accounts.
     *
     * @param award
     * @return
     */
    @Override
    public boolean isChartAndOrgSetupForInvoicing(ContractsAndGrantsBillingAward award) {
        if (ObjectUtils.isNull(award.getAwardPrimaryFundManager()) || ObjectUtils.isNull(award.getAwardPrimaryFundManager().getFundManager())) {
            return false;
        }
        ChartOrgHolder chartOrgHolder = financialSystemUserService.getPrimaryOrganization(award.getAwardPrimaryFundManager().getFundManager().getPrincipalId(), KFSConstants.OptionalModuleNamespaces.ACCOUNTS_RECEIVABLE);
        if (ObjectUtils.isNull(chartOrgHolder)) {
            return false;
        }
        String coaCode = chartOrgHolder.getChartOfAccountsCode();
        String orgCode = chartOrgHolder.getOrganizationCode();
        Integer currentYear = universityDateService.getCurrentFiscalYear();

        Map<String, Object> criteria = new HashMap<>();
        Map<String, Object> sysCriteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, currentYear);
        sysCriteria.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, currentYear);
        criteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, coaCode);
        criteria.put(KFSPropertyConstants.ORGANIZATION_CODE, orgCode);

        // To retrieve processing codes based on billing codes using organization options
        List<String> procCodes = getContractsGrantsInvoiceDocumentService().getProcessingFromBillingCodes(coaCode, orgCode);
        if (!CollectionUtils.isEmpty(procCodes) && procCodes.size() > 1) {
            sysCriteria.put(ArPropertyConstants.OrganizationAccountingDefaultFields.PROCESSING_CHART_OF_ACCOUNTS_CODE, procCodes.get(0));
            sysCriteria.put(ArPropertyConstants.OrganizationAccountingDefaultFields.PROCESSING_ORGANIZATION_CODE, procCodes.get(1));
            OrganizationAccountingDefault organizationAccountingDefault = businessObjectService.findByPrimaryKey(OrganizationAccountingDefault.class, criteria);

            SystemInformation systemInformation = businessObjectService.findByPrimaryKey(SystemInformation.class, sysCriteria);
            return ObjectUtils.isNotNull(organizationAccountingDefault) && ObjectUtils.isNotNull(systemInformation);
        }
        return false;

    }

    public AccountingPeriodService getAccountingPeriodService() {
        return accountingPeriodService;
    }

    public void setAccountingPeriodService(AccountingPeriodService accountingPeriodService) {
        this.accountingPeriodService = accountingPeriodService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public ContractsGrantsInvoiceDocumentService getContractsGrantsInvoiceDocumentService() {
        return contractsGrantsInvoiceDocumentService;
    }

    public void setContractsGrantsInvoiceDocumentService(ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService) {
        this.contractsGrantsInvoiceDocumentService = contractsGrantsInvoiceDocumentService;
    }

    public CustomerService getCustomerService() {
        return customerService;
    }

    public void setCustomerService(CustomerService customerService) {
        this.customerService = customerService;
    }

    public FinancialSystemDocumentService getFinancialSystemDocumentService() {
        return financialSystemDocumentService;
    }

    public void setFinancialSystemDocumentService(FinancialSystemDocumentService financialSystemDocumentService) {
        this.financialSystemDocumentService = financialSystemDocumentService;
    }

    public KualiModuleService getKualiModuleService() {
        return kualiModuleService;
    }

    public void setKualiModuleService(KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public VerifyBillingFrequencyService getVerifyBillingFrequencyService() {
        return verifyBillingFrequencyService;
    }

    public void setVerifyBillingFrequencyService(VerifyBillingFrequencyService verifyBillingFrequencyService) {
        this.verifyBillingFrequencyService = verifyBillingFrequencyService;
    }

    public UniversityDateService getUniversityDateService() {
        return universityDateService;
    }

    public void setUniversityDateService(UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public OptionsService getOptionsService() {
        return optionsService;
    }

    public void setOptionsService(OptionsService optionsService) {
        this.optionsService = optionsService;
    }

    public void setFinancialSystemUserService(FinancialSystemUserService financialSystemUserService) {
        this.financialSystemUserService = financialSystemUserService;
    }
}
