/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.dataaccess.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.Query;
import org.apache.ojb.broker.query.QueryByCriteria;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.broker.query.ReportQueryByCriteria;
import org.kuali.kfs.coa.businessobject.ObjectCode;
import org.kuali.kfs.coa.businessobject.ObjectCodeCurrent;
import org.kuali.kfs.coa.businessobject.ObjectLevel;
import org.kuali.kfs.gl.businessobject.Balance;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CostCategory;
import org.kuali.kfs.module.ar.businessobject.CostCategoryDetail;
import org.kuali.kfs.module.ar.businessobject.CostCategoryObjectCode;
import org.kuali.kfs.module.ar.businessobject.CostCategoryObjectConsolidation;
import org.kuali.kfs.module.ar.businessobject.CostCategoryObjectLevel;
import org.kuali.kfs.module.ar.dataaccess.CostCategoryDao;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.util.TransactionalServiceUtils;
import org.kuali.rice.core.framework.persistence.ojb.dao.PlatformAwareDaoBaseOjb;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Default OJB implementation of the CostCategoryDao
 */
public class CostCategoryDaoOjb extends PlatformAwareDaoBaseOjb implements CostCategoryDao {

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryConsolidationAmongConsolidations(CostCategoryObjectConsolidation objectConsolidation) {
        Criteria crit = new Criteria();
        if (!StringUtils.isBlank(objectConsolidation.getCategoryCode())) {
            crit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, objectConsolidation.getCategoryCode());
        }
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, objectConsolidation.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.FIN_CONSOLIDATION_OBJECT_CODE, objectConsolidation.getFinConsolidationObjectCode());
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        crit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query q = new QueryByCriteria(CostCategoryObjectConsolidation.class, crit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(q));
    }

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryConsolidationAmongLevels(CostCategoryObjectConsolidation consolidation) {
        Criteria crit = new Criteria();
        if (!StringUtils.isBlank(consolidation.getCategoryCode())) {
            crit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, consolidation.getCategoryCode());
        }
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, consolidation.getChartOfAccountsCode());
        crit.addEqualTo(ArPropertyConstants.OBJECT_LEVEL + "." + KFSPropertyConstants.FINANCIAL_CONSOLIDATION_OBJECT_CODE, consolidation.getFinConsolidationObjectCode());
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        crit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query q = new QueryByCriteria(CostCategoryObjectLevel.class, crit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(q));
    }

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryConsolidationAmongCodes(CostCategoryObjectConsolidation consolidation) {
        Criteria crit = new Criteria();
        if (!StringUtils.isBlank(consolidation.getCategoryCode())) {
            crit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, consolidation.getCategoryCode());
        }
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, consolidation.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.OBJECT_CODE_CURRENT + "." + KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL + "." + KFSPropertyConstants.FINANCIAL_CONSOLIDATION_OBJECT_CODE, consolidation.getFinConsolidationObjectCode());
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        crit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query q = new QueryByCriteria(CostCategoryObjectCode.class, crit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(q));
    }

    /**
     * @return a newly generated subquery which guarantees that we will only check active cost categories
     */
    protected ReportQueryByCriteria buildActiveCostCategorySubQuery() {
        Criteria crit = new Criteria();
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(CostCategory.class, crit);
        subQuery.setAttributes(new String[]{ArPropertyConstants.CATEGORY_CODE});
        return subQuery;
    }

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryLevelAmongLevels(CostCategoryObjectLevel objectLevel) {
        Criteria crit = new Criteria();
        if (!StringUtils.isBlank(objectLevel.getCategoryCode())) {
            crit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, objectLevel.getCategoryCode());
        }
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, objectLevel.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE, objectLevel.getFinancialObjectLevelCode());
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        crit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query q = new QueryByCriteria(CostCategoryObjectLevel.class, crit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(q));
    }

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryLevelAmongConsolidations(CostCategoryObjectLevel level) {
        Criteria consolidationCrit = new Criteria();
        if (!StringUtils.isBlank(level.getCategoryCode())) {
            consolidationCrit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, level.getCategoryCode());
        }
        consolidationCrit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, level.getChartOfAccountsCode());
        consolidationCrit.addEqualTo(KFSPropertyConstants.FIN_CONSOLIDATION_OBJECT_CODE, buildLevelConsolidationCodeSubQuery(level));
        consolidationCrit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        consolidationCrit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query consolidationQuery = new QueryByCriteria(CostCategoryObjectConsolidation.class, consolidationCrit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(consolidationQuery));
    }

    /**
     * Builds a subquery - so we can perform the whole operation as a single query - which finds the consolidation code of an object level
     *
     * @param level the cost category object level to find the consolidation code for
     * @return a subQuery to use in a larger query
     */
    protected ReportQueryByCriteria buildLevelConsolidationCodeSubQuery(CostCategoryObjectLevel level) {
        Criteria crit = new Criteria();
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, level.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE, level.getFinancialObjectLevelCode());
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(ObjectLevel.class, crit);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.FINANCIAL_CONSOLIDATION_OBJECT_CODE});
        return subQuery;
    }

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryLevelAmongCodes(CostCategoryObjectLevel level) {
        Criteria crit = new Criteria();
        if (!StringUtils.isBlank(level.getCategoryCode())) {
            crit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, level.getCategoryCode());
        }
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, level.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.OBJECT_CODE_CURRENT + "." + KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL + "." + KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE, level.getFinancialObjectLevelCode());
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        crit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query q = new QueryByCriteria(CostCategoryObjectCode.class, crit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(q));
    }

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryObjectCodeAmongCodes(CostCategoryObjectCode objectCode) {
        Criteria crit = new Criteria();
        if (!StringUtils.isBlank(objectCode.getCategoryCode())) {
            crit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, objectCode.getCategoryCode());
        }
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, objectCode.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, objectCode.getFinancialObjectCode());
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        crit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query uniqueQuery = new QueryByCriteria(CostCategoryObjectCode.class, crit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(uniqueQuery));
    }

    @Override
    public CostCategoryDetail retrieveMatchingCostCategoryObjectCodeAmongLevels(CostCategoryObjectCode objectCode) {
        Criteria objectLevelCrit = new Criteria();
        if (!StringUtils.isBlank(objectCode.getCategoryCode())) {
            objectLevelCrit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, objectCode.getCategoryCode());
        }
        objectLevelCrit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, objectCode.getChartOfAccountsCode());
        objectLevelCrit.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE, buildObjectCodeLevelCodeSubQuery(objectCode));
        objectLevelCrit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        objectLevelCrit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query objectLevelQuery = new QueryByCriteria(CostCategoryObjectLevel.class, objectLevelCrit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(getPersistenceBrokerTemplate().getIteratorByQuery(objectLevelQuery));
    }

    /**
     * Builds a subquery - so we can perform the whole operation as a single query - which finds the object level
     * code for an object code
     *
     * @param objectCode the cost category object code to find the object level code for
     * @return subQuery to use in a larger query
     */
    protected ReportQueryByCriteria buildObjectCodeLevelCodeSubQuery(CostCategoryObjectCode objectCode) {
        Criteria crit = new Criteria();
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, objectCode.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, objectCode.getFinancialObjectCode());
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(ObjectCodeCurrent.class, crit);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE});
        return subQuery;
    }

    @Override
    public CostCategoryDetail retrieveCostCategoryObjectCodeAmongConsolidations(CostCategoryObjectCode objectCode) {
        Criteria objectConsolidationCrit = new Criteria();
        if (!StringUtils.isBlank(objectCode.getCategoryCode())) {
            objectConsolidationCrit.addNotEqualTo(ArPropertyConstants.CATEGORY_CODE, objectCode.getCategoryCode());
        }
        objectConsolidationCrit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, objectCode.getChartOfAccountsCode());
        objectConsolidationCrit.addEqualTo(KFSPropertyConstants.FIN_CONSOLIDATION_OBJECT_CODE, buildObjectCodeConsolidationCodeSubQuery(objectCode));
        objectConsolidationCrit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        objectConsolidationCrit.addIn(ArPropertyConstants.CATEGORY_CODE, buildActiveCostCategorySubQuery());
        Query objectConsolidationQuery = new QueryByCriteria(CostCategoryObjectConsolidation.class, objectConsolidationCrit);
        return (CostCategoryDetail) TransactionalServiceUtils.retrieveFirstAndExhaustIterator((getPersistenceBrokerTemplate().getIteratorByQuery(objectConsolidationQuery)));
    }

    /**
     * Builds a subquery - so we can perform the whole operation as a single query - which finds the object
     * consolidation code for an object code
     *
     * @param objectCode the cost category object code to find the object level code for
     * @return subQuery to use in a larger query
     */
    protected ReportQueryByCriteria buildObjectCodeConsolidationCodeSubQuery(CostCategoryObjectCode objectCode) {
        Criteria crit = new Criteria();
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, objectCode.getChartOfAccountsCode());
        crit.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, objectCode.getFinancialObjectCode());
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(ObjectCodeCurrent.class, crit);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL + "." + KFSPropertyConstants.FINANCIAL_CONSOLIDATION_OBJECT_CODE});
        return subQuery;
    }

    @Override
    public List<Balance> getBalancesForCostCategory(Integer fiscalYear, String chartOfAccountsCode,
            String accountNumber, String balanceType, Collection<String> objectTypeCodes, CostCategory costCategory) {
        if (!costCategory.isActive()) {
            return new ArrayList<>(); // cost category isn't active?  then dont' find any matching balances
        }

        Criteria crit = new Criteria();
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        crit.addEqualTo(KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);
        crit.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        crit.addEqualTo(KFSPropertyConstants.BALANCE_TYPE_CODE, balanceType);

        crit.addIn(KFSPropertyConstants.OBJECT_TYPE_CODE, objectTypeCodes);

        Criteria objectCodesCollectiveCriteria = new Criteria();

        Criteria objectCodeCriteria = new Criteria();
        objectCodeCriteria.addIn(KFSPropertyConstants.OBJECT_CODE, buildObjectCodesOfCostCategorySubQuery(costCategory, chartOfAccountsCode));
        objectCodesCollectiveCriteria.addOrCriteria(objectCodeCriteria);

        Criteria objectLevelCriteria = new Criteria();
        objectLevelCriteria.addIn(KFSPropertyConstants.FINANCIAL_OBJECT + "." + KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE, buildObjectLevelsOfCostCategorySubQuery(costCategory, chartOfAccountsCode));
        objectCodesCollectiveCriteria.addOrCriteria(objectLevelCriteria);

        Criteria consolidationCriteria = new Criteria();
        consolidationCriteria.addIn(KFSPropertyConstants.FINANCIAL_OBJECT + "." + KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL + "." + KFSPropertyConstants.FINANCIAL_CONSOLIDATION_OBJECT_CODE, buildObjectConsolidationsOfCostCategorySubQuery(costCategory, chartOfAccountsCode));
        objectCodesCollectiveCriteria.addOrCriteria(consolidationCriteria);

        crit.addAndCriteria(objectCodesCollectiveCriteria);
        Query q = new QueryByCriteria(Balance.class, crit);
        return (List<Balance>) getPersistenceBrokerTemplate().getCollectionByQuery(q);
    }

    /**
     * Builds a subquery to find the object codes which match the given cost category within the given chart of accounts
     *
     * @param costCategory        the costCategory to find object codes for
     * @param chartOfAccountsCode the chart that object codes should be limited to
     * @return subQuery the subquery to find matching object codes
     */
    protected ReportQueryByCriteria buildObjectCodesOfCostCategorySubQuery(CostCategory costCategory,
            String chartOfAccountsCode) {
        Criteria crit = new Criteria();
        crit.addEqualTo(ArPropertyConstants.CATEGORY_CODE, costCategory.getCategoryCode());
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(CostCategoryObjectCode.class, crit);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.FINANCIAL_OBJECT_CODE});
        return subQuery;
    }

    /**
     * Builds a subquery to find the object levels which match the given cost category within the given chart of accounts
     *
     * @param costCategory        the costCategory to find object lvels for
     * @param chartOfAccountsCode the chart that object codes should be limited to
     * @return subQuery the subquery to find matching object levels
     */
    protected ReportQueryByCriteria buildObjectLevelsOfCostCategorySubQuery(CostCategory costCategory,
            String chartOfAccountsCode) {
        Criteria crit = new Criteria();
        crit.addEqualTo(ArPropertyConstants.CATEGORY_CODE, costCategory.getCategoryCode());
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(CostCategoryObjectLevel.class, crit);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE});
        return subQuery;
    }

    /**
     * Builds a subquery to find the object consolidations which match the given cost category within the given chart of accounts
     *
     * @param costCategory        the costCategory to find object consolidations for
     * @param chartOfAccountsCode the chart that object consolidations should be limited to
     * @return subQuery the subquery to find matching object consolidations
     */
    protected ReportQueryByCriteria buildObjectConsolidationsOfCostCategorySubQuery(CostCategory costCategory,
            String chartOfAccountsCode) {
        Criteria crit = new Criteria();
        crit.addEqualTo(ArPropertyConstants.CATEGORY_CODE, costCategory.getCategoryCode());
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        crit.addEqualTo(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(CostCategoryObjectConsolidation.class, crit);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.FIN_CONSOLIDATION_OBJECT_CODE});
        return subQuery;
    }

    /**
     * Attempts to look up a cost category which matches the balance
     */
    @Override
    public CostCategory getCostCategoryForObjectCode(Integer universityFiscalYear, String chartOfAccountsCode,
            String financialObjectCode) {
        Criteria crit = new Criteria();

        Criteria objectCodesCriteria = new Criteria();
        objectCodesCriteria.addEqualTo(ArPropertyConstants.OBJECT_CODES + "." + KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        objectCodesCriteria.addEqualTo(ArPropertyConstants.OBJECT_CODES + "." + KFSPropertyConstants.FINANCIAL_OBJECT_CODE, financialObjectCode);
        objectCodesCriteria.addEqualTo(ArPropertyConstants.OBJECT_CODES + "." + KFSPropertyConstants.ACTIVE, Boolean.TRUE);

        Criteria objectLevelsCriteria = new Criteria();
        objectLevelsCriteria.addEqualTo(ArPropertyConstants.OBJECT_LEVELS + "." + KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        objectLevelsCriteria.addIn(ArPropertyConstants.OBJECT_LEVELS + "." + KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE, buildObjectCodeOfBalanceSubQuery(universityFiscalYear, chartOfAccountsCode, financialObjectCode, KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL_CODE));
        objectLevelsCriteria.addEqualTo(ArPropertyConstants.OBJECT_LEVELS + "." + KFSPropertyConstants.ACTIVE, Boolean.TRUE);

        Criteria objectConsolidationsCriteria = new Criteria();
        objectConsolidationsCriteria.addEqualTo(ArPropertyConstants.OBJECT_CONSOLIDATIONS + "." + KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        objectConsolidationsCriteria.addIn(ArPropertyConstants.OBJECT_CONSOLIDATIONS + "." + KFSPropertyConstants.FIN_CONSOLIDATION_OBJECT_CODE, buildObjectCodeOfBalanceSubQuery(universityFiscalYear, chartOfAccountsCode, financialObjectCode, KFSPropertyConstants.FINANCIAL_OBJECT_LEVEL + "." + KFSPropertyConstants.FINANCIAL_CONSOLIDATION_OBJECT_CODE));
        objectConsolidationsCriteria.addEqualTo(ArPropertyConstants.OBJECT_CONSOLIDATIONS + "." + KFSPropertyConstants.ACTIVE, Boolean.TRUE);

        crit.addOrCriteria(objectCodesCriteria);
        crit.addOrCriteria(objectLevelsCriteria);
        crit.addOrCriteria(objectConsolidationsCriteria);

        Query q = new QueryByCriteria(CostCategory.class, crit);
        Collection<CostCategory> costCategories = getPersistenceBrokerTemplate().getCollectionByQuery(q);
        if (CollectionUtils.isEmpty(costCategories)) {
            return null;
        }
        return TransactionalServiceUtils.retrieveFirstAndExhaustIterator(costCategories.iterator());
    }

    /**
     * Builds a sub-query to look up a value from the object code table
     *
     * @param universityFiscalYear
     * @param chartOfAccountsCode
     * @param financialObjectCode
     * @param retrievalProperty the property to retrieve from the object code
     * @return the subquery
     */
    protected ReportQueryByCriteria buildObjectCodeOfBalanceSubQuery(Integer universityFiscalYear,
            String chartOfAccountsCode, String financialObjectCode, String retrievalProperty) {
        Criteria crit = new Criteria();
        crit.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, universityFiscalYear);
        crit.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        crit.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, financialObjectCode);
        ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(ObjectCode.class, crit);
        subQuery.setAttributes(new String[]{retrievalProperty});
        return subQuery;
    }
}
