/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.event;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.document.TransactionalDocument;
import org.kuali.kfs.krad.rules.rule.BusinessRule;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEventBase;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.CustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.document.validation.RecalculateCustomerCreditMemoDetailRule;

public class RecalculateCustomerCreditMemoDetailEvent extends KualiDocumentEventBase {

    private static final Logger LOG = LogManager.getLogger();
    protected CustomerCreditMemoDetail customerCreditMemoDetail;

    public RecalculateCustomerCreditMemoDetailEvent(String errorPathPrefix, Document document,
            CustomerCreditMemoDetail customerCreditMemoDetail) {
        super("Recalculating customer credit memo detail for document " + getDocumentId(document),
                errorPathPrefix, document);

        // by doing a deep copy, we are ensuring that the business rule class can't update
        // the original object by reference
        this.customerCreditMemoDetail = (CustomerCreditMemoDetail) ObjectUtils.deepCopy(customerCreditMemoDetail);

        logEvent();
    }

    @SuppressWarnings("unchecked")
    public Class getRuleInterfaceClass() {
        return RecalculateCustomerCreditMemoDetailRule.class;
    }

    @SuppressWarnings("unchecked")
    public boolean invokeRuleMethod(BusinessRule rule) {
        return ((RecalculateCustomerCreditMemoDetailRule) rule)
                .processRecalculateCustomerCreditMemoDetailRules((TransactionalDocument) document,
                        customerCreditMemoDetail);
    }

    public CustomerCreditMemoDetail getCustomerCreditMemoDetail() {
        return customerCreditMemoDetail;
    }

    public void validate() {
        super.validate();
        if (customerCreditMemoDetail == null) {
            throw new IllegalArgumentException("invalid (null) customer credit memo detail");
        }
    }

    /**
     * Logs the event type and some information about the associated customer credit memo detail
     */
    private void logEvent() {
        if (LOG.isDebugEnabled()) {
            StringBuffer logMessage = new StringBuffer(StringUtils.substringAfterLast(this.getClass().getName(), "."));
            logMessage.append(" with ");

            // vary logging detail as needed
            if (customerCreditMemoDetail == null) {
                logMessage.append("null customerCreditMemoDetail");
            } else {
                logMessage.append(" customer credit memo detail# ");
                logMessage.append(customerCreditMemoDetail.getReferenceInvoiceItemNumber());
            }

            LOG.debug(logMessage);
        }
    }
}
