/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.web.struts;

import org.kuali.kfs.kns.service.DocumentHelperService;
import org.kuali.kfs.kns.web.ui.ExtraButton;
import org.kuali.kfs.kns.web.ui.HeaderField;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.module.ar.ArAuthorizationConstants;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceItemCode;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.authorization.CustomerInvoiceDocumentPresentationController;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.businessobject.UnitOfMeasure;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.AccountingDocument;
import org.kuali.kfs.sys.web.struts.KualiAccountingDocumentFormBase;
import org.kuali.rice.core.api.config.property.ConfigurationService;
import org.kuali.rice.core.web.format.CurrencyFormatter;
import org.kuali.rice.kew.api.WorkflowDocument;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class CustomerInvoiceForm extends KualiAccountingDocumentFormBase {

    protected transient ConfigurationService configService;

    protected CustomerInvoiceDetail newCustomerInvoiceDetail;

    /**
     * Constructs a CustomerInvoiceForm.java. Also sets new customer invoice document detail to a newly constructed customer
     * invoice detail.
     */
    public CustomerInvoiceForm() {
        super();
    }

    @Override
    protected String getDefaultDocumentTypeName() {
        return KFSConstants.FinancialDocumentTypeCodes.CUSTOMER_INVOICE;
    }

    /**
     * This method...
     *
     * @return
     */
    public CustomerInvoiceDocument getCustomerInvoiceDocument() {
        return (CustomerInvoiceDocument) getDocument();
    }

    /**
     * @see org.kuali.kfs.sys.web.struts.KualiAccountingDocumentFormBase#populate(javax.servlet.http.HttpServletRequest)
     */
    @Override
    public void populate(HttpServletRequest request) {
        super.populate(request);
        SpringContext.getBean(CustomerInvoiceDocumentService.class).loadCustomerAddressesForCustomerInvoiceDocument(getCustomerInvoiceDocument());
    }

    /**
     * Reused to create new source accounting line (i.e customer invoice detail line) with defaulted values.
     *
     * @see org.kuali.kfs.sys.web.struts.KualiAccountingDocumentFormBase#createNewSourceAccountingLine(org.kuali.kfs.sys.document.AccountingDocument)
     */
    @Override
    protected SourceAccountingLine createNewSourceAccountingLine(AccountingDocument financialDocument) {
        if (financialDocument == null) {
            throw new IllegalArgumentException("invalid (null) document");
        }
        return SpringContext.getBean(CustomerInvoiceDetailService.class).getCustomerInvoiceDetailFromOrganizationAccountingDefaultForCurrentYear();
    }

    /**
     * By overriding this method, we can add the invoice total and open amount to the document header.
     *
     * @see org.kuali.rice.kns.web.struts.form.KualiForm#getDocInfo()
     * <p>
     * KRAD Conversion: Performs the customization of the header fields.
     * No data dictionary is involved here.
     */
    @Override
    public void populateHeaderFields(WorkflowDocument workflowDocument) {
        super.populateHeaderFields(workflowDocument);
        getDocInfo().add(new HeaderField("DataDictionary.CustomerInvoiceDocument.attributes.sourceTotal", (String) new CurrencyFormatter().format(getCustomerInvoiceDocument().getSourceTotal())));
        getDocInfo().add(new HeaderField("DataDictionary.CustomerInvoiceDocument.attributes.openAmount", (String) new CurrencyFormatter().format(getCustomerInvoiceDocument().getOpenAmount())));
    }

    /**
     * Configure lookup for Invoice Item Code source accounting line
     *
     * @see org.kuali.kfs.sys.web.struts.KualiAccountingDocumentFormBase#getForcedLookupOptionalFields()
     */
    @Override
    public Map getForcedLookupOptionalFields() {
        Map forcedLookupOptionalFields = super.getForcedLookupOptionalFields();

        forcedLookupOptionalFields.put(ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_ITEM_CODE, ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_ITEM_CODE + ";" + CustomerInvoiceItemCode.class.getName());
        forcedLookupOptionalFields.put(ArPropertyConstants.CustomerInvoiceDocumentFields.UNIT_OF_MEASURE_CODE, ArPropertyConstants.CustomerInvoiceDocumentFields.UNIT_OF_MEASURE_CODE + ";" + UnitOfMeasure.class.getName());

        return forcedLookupOptionalFields;
    }

    /**
     * Make amount and sales tax read only
     *
     * @see org.kuali.rice.kns.web.struts.form.KualiTransactionalDocumentFormBase#getForcedReadOnlyFields()
     */
    @Override
    public Map getForcedReadOnlyFields() {
        Map map = super.getForcedReadOnlyFields();
        map.put(KFSPropertyConstants.AMOUNT, Boolean.TRUE);
        map.put("invoiceItemTaxAmount", Boolean.TRUE);
        map.put("openAmount", Boolean.TRUE);
        return map;
    }

    /**
     * Build additional customer invoice specific buttons and set extraButtons list.
     *
     * @return - list of extra buttons to be displayed to the user
     * <p>
     * KRAD Conversion: Performs the creation of extra buttons.
     * No data dictionary is involved here.
     */
    @Override
    public List<ExtraButton> getExtraButtons() {

        // clear out the extra buttons array
        extraButtons.clear();

        //  get the edit modes from the preso controller
        CustomerInvoiceDocument invoiceDocument = (CustomerInvoiceDocument) getDocument();
        DocumentHelperService docHelperService = SpringContext.getBean(DocumentHelperService.class);
        CustomerInvoiceDocumentPresentationController presoController =
            (CustomerInvoiceDocumentPresentationController) docHelperService.getDocumentPresentationController(invoiceDocument);
        Set<String> editModes = presoController.getEditModes(invoiceDocument);

        //  draw the Print File button if appropriate
        if (editModes.contains(ArAuthorizationConstants.CustomerInvoiceDocumentEditMode.DISPLAY_PRINT_BUTTON)) {
            String printButtonURL = getConfigService().getPropertyValueAsString(KFSConstants.EXTERNALIZABLE_IMAGES_URL_KEY);

            ExtraButton newButton = new ExtraButton();

            newButton.setExtraButtonProperty("methodToCall.print");
            newButton.setExtraButtonSource(printButtonURL + "buttonsmall_genprintfile.gif");
            newButton.setExtraButtonAltText("Print");
            newButton.setExtraButtonOnclick("excludeSubmitRestriction=true");

            extraButtons.add(newButton);
        }

        //  draw the Error Correction button if appropriate
        if (presoController.canErrorCorrect(invoiceDocument)) {
            String printButtonURL = getConfigService().getPropertyValueAsString(KFSConstants.EXTERNALIZABLE_IMAGES_URL_KEY);
            extraButtons.add(generateErrorCorrectionButton());
        }

        return extraButtons;
    }

    /**
     * Adds a new button to the extra buttons collection.
     *
     * @param property - property for button
     * @param source   - location of image
     * @param altText  - alternate text for button if images don't appear
     *                 <p>
     *                 KRAD Conversion: Performs extra button customization.
     *                 No data dictionary is involved here.
     */
    protected void addExtraButton(String property, String source, String altText) {

        ExtraButton newButton = new ExtraButton();

        newButton.setExtraButtonProperty(property);
        newButton.setExtraButtonSource(source);
        newButton.setExtraButtonAltText(altText);

        extraButtons.add(newButton);
    }

    protected ConfigurationService getConfigService() {
        if (configService == null) {
            configService = SpringContext.getBean(ConfigurationService.class);
        }
        return configService;
    }

    /**
     * @see org.kuali.rice.kns.web.struts.form.KualiDocumentFormBase#shouldMethodToCallParameterBeUsed(java.lang.String, java.lang.String, javax.servlet.http.HttpServletRequest)
     */
    @Override
    public boolean shouldMethodToCallParameterBeUsed(String methodToCallParameterName, String methodToCallParameterValue, HttpServletRequest request) {
        if (KRADConstants.DISPATCH_REQUEST_PARAMETER.equals(methodToCallParameterName) && ArConstants.PRINT_INVOICE_PDF_METHOD.equals(methodToCallParameterValue)) {
            return true;
        }
        return super.shouldMethodToCallParameterBeUsed(methodToCallParameterName, methodToCallParameterValue, request);
    }
}
