/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAwardAccount;
import org.kuali.kfs.kns.document.authorization.BusinessObjectRestrictions;
import org.kuali.kfs.kns.web.comparator.CellComparatorHelper;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsMilestoneReport;
import org.kuali.kfs.module.ar.businessobject.Milestone;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.krad.bo.BusinessObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Defines a custom lookup for a Milestone Reports.
 */
public class ContractsGrantsMilestoneReportLookupableHelperServiceImpl extends
        ContractsGrantsReportLookupableHelperServiceImplBase {

    protected DateTimeService dateTimeService;

    @Override
    public Collection performLookup(LookupForm lookupForm, Collection resultTable, boolean bounded) {
        Map lookupFormFields = lookupForm.getFieldsForLookup();

        setBackLocation(lookupForm.getFieldsForLookup().get(KRADConstants.BACK_LOCATION));
        setDocFormKey(lookupForm.getFieldsForLookup().get(KRADConstants.DOC_FORM_KEY));

        Collection<ContractsGrantsMilestoneReport> displayList = new ArrayList<>();
        Map<String, String> lookupCriteria = buildCriteriaForLookup(lookupFormFields);
        Collection<Milestone> milestones = getLookupService().findCollectionBySearchHelper(Milestone.class,
                lookupCriteria, true);

        final String awardChartOfAccountsCode = (String) lookupFormFields.get(
                KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        final String awardAccountNumber = (String) lookupFormFields.get(KFSPropertyConstants.ACCOUNT_NUMBER);

        // build search result fields
        for (Milestone milestone : milestones) {
            if (isMilestoneMatchesChart(milestone, awardChartOfAccountsCode)
                    && isMilestoneMatchesAccount(milestone, awardAccountNumber)) {
                ContractsGrantsMilestoneReport cgMilestoneReport = new ContractsGrantsMilestoneReport();
                cgMilestoneReport.setProposalNumber(milestone.getProposalNumber());

                ContractsAndGrantsBillingAward award = milestone.getAward();
                List<ContractsAndGrantsBillingAwardAccount> awardAccounts = (ObjectUtils.isNull(award)) ?
                        new ArrayList<>() : award.getActiveAwardAccounts();
                String accountNumber = (awardAccounts.size() > 0) ? awardAccounts.get(0).getAccountNumber() : "";

                cgMilestoneReport.setAccountNumber(accountNumber);
                cgMilestoneReport.setChartOfAccountsCode(milestone.getChartOfAccountsCode());
                cgMilestoneReport.setMilestoneNumber(milestone.getMilestoneNumber());
                cgMilestoneReport.setMilestoneExpectedCompletionDate(milestone.getMilestoneExpectedCompletionDate());
                cgMilestoneReport.setMilestoneAmount(milestone.getMilestoneAmount());
                cgMilestoneReport.setBilled(milestone.isBilled());
                cgMilestoneReport.setActive(milestone.isActive());

                displayList.add(cgMilestoneReport);
            }
        }

        buildResultTable(lookupForm, displayList, resultTable);

        return displayList;
    }

    /**
     * Builds criteria which the Milestone lookup will be a bit happier with, as it uses names from Milestone, not
     * the report BO
     *
     * @param lookupFormFields the fields from the lookup itself
     * @return a converted Map of fields which can be used to search for Milestones
     */
    protected Map<String, String> buildCriteriaForLookup(Map lookupFormFields) {
        Map<String, String> lookupCriteria = new HashMap<>();

        final String proposalNumber = (String) lookupFormFields.get(KFSPropertyConstants.PROPOSAL_NUMBER);
        if (!StringUtils.isBlank(proposalNumber)) {
            lookupCriteria.put(KFSPropertyConstants.PROPOSAL_NUMBER, proposalNumber);
        }

        final String lowerBoundMilestoneExpectedCompletionDate = (String) lookupFormFields.get(
                KRADConstants.LOOKUP_RANGE_LOWER_BOUND_PROPERTY_PREFIX +
                        ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE);
        final String upperBoundMilestoneExpectedCompletionDate = (String) lookupFormFields.get(
                ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE);
        final String milestoneExpectedCompletionDate = getContractsGrantsReportHelperService()
                .fixDateCriteria(lowerBoundMilestoneExpectedCompletionDate, upperBoundMilestoneExpectedCompletionDate,
                        false);
        if (!StringUtils.isBlank(milestoneExpectedCompletionDate)) {
            lookupCriteria.put(ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE,
                    milestoneExpectedCompletionDate);
        }

        final String billed = (String) lookupFormFields.get(ArPropertyConstants.BILLED);
        if (!StringUtils.isBlank(billed)) {
            lookupCriteria.put(ArPropertyConstants.BILLED, billed);
        }

        final String active = (String) lookupFormFields.get(KFSPropertyConstants.ACTIVE);
        if (!StringUtils.isBlank(active)) {
            lookupCriteria.put(KFSPropertyConstants.ACTIVE, active);
        }

        return lookupCriteria;
    }

    /**
     * Determines if the given milestone matches the given chart
     *
     * @param milestone           the milestone to check
     * @param chartOfAccountsCode the chart to check
     * @return true if there is no chart to check, or if there are no award accounts to check against, or if it finds
     *         a match; false otherwise
     */
    protected boolean isMilestoneMatchesChart(Milestone milestone, String chartOfAccountsCode) {
        if (StringUtils.isBlank(chartOfAccountsCode)) {
            // no chart to check
            return true;
        }
        if (ObjectUtils.isNull(milestone.getAward())
                || ObjectUtils.isNull(milestone.getAward().getActiveAwardAccounts())
                || milestone.getAward().getActiveAwardAccounts().isEmpty()) {
            // oddly, no award accounts to check against
            return true;
        }
        Pattern chartRegex = convertWildcardsToPattern(chartOfAccountsCode);
        for (ContractsAndGrantsBillingAwardAccount awardAccount : milestone.getAward().getActiveAwardAccounts()) {
            Matcher chartMatch = chartRegex.matcher(awardAccount.getChartOfAccountsCode());
            if (chartMatch.matches()) {
                // we found a match
                return true;
            }
        }
        return false;
    }

    /**
     * Determines if the given milestone matches the given account
     *
     * @param milestone     the milestone to check against
     * @param accountNumber the account number to check
     * @return true if there is no account number to check, if there are no award accounts to check against, or if it
     *         finds a match; false otherwise
     */
    protected boolean isMilestoneMatchesAccount(Milestone milestone, String accountNumber) {
        if (StringUtils.isBlank(accountNumber)) {
            return true;
        }
        if (ObjectUtils.isNull(milestone.getAward())
                || ObjectUtils.isNull(milestone.getAward().getActiveAwardAccounts())
                || milestone.getAward().getActiveAwardAccounts().isEmpty()) {
            // oddly, no award accounts to check against
            return true;
        }
        Pattern accountRegex = convertWildcardsToPattern(accountNumber);
        for (ContractsAndGrantsBillingAwardAccount awardAccount : milestone.getAward().getActiveAwardAccounts()) {
            Matcher accountMatch = accountRegex.matcher(awardAccount.getAccountNumber());
            if (accountMatch.matches()) {
                // we found a match
                return true;
            }
        }
        return false;
    }

    /**
     * Converts the given String into a regex pattern, replacing "*" with ".+" for the regex
     *
     * @param lookupFieldValue the field value to convert into a regex
     * @return the regex pattern
     */
    protected Pattern convertWildcardsToPattern(String lookupFieldValue) {
        final String lookupFieldValueRegex = lookupFieldValue.replaceAll("\\*", ".+");
        return Pattern.compile(lookupFieldValueRegex);
    }

    @Override
    protected void buildResultTable(LookupForm lookupForm, Collection displayList, Collection resultTable) {
        Person user = GlobalVariables.getUserSession().getPerson();
        boolean hasReturnableRow = false;
        // iterate through result list and wrap rows with return url and action url
        for (Object entry : displayList) {
            BusinessObject element = (BusinessObject) entry;

            BusinessObjectRestrictions businessObjectRestrictions = getBusinessObjectAuthorizationService()
                    .getLookupResultRestrictions(element, user);

            List<Column> columns = getColumns();
            for (Column col : columns) {
                String propValue = ObjectUtils.getFormattedPropertyValue(element, col.getPropertyName(),
                        col.getFormatter());
                Class propClass = getPropertyClass(element, col.getPropertyName());

                col.setComparator(CellComparatorHelper.getAppropriateComparatorForPropertyClass(propClass));
                col.setValueComparator(CellComparatorHelper.getAppropriateValueComparatorForPropertyClass(propClass));

                propValue = maskValueIfNecessary(element, col.getPropertyName(), propValue,
                        businessObjectRestrictions);
                col.setPropertyValue(propValue);
            }

            ResultRow row = new ResultRow(columns, "", ACTION_URLS_EMPTY);

            if (getBusinessObjectDictionaryService().isExportable(getBusinessObjectClass())) {
                row.setBusinessObject(element);
            }
            boolean isRowReturnable = isResultReturnable(element);
            row.setRowReturnable(isRowReturnable);
            if (isRowReturnable) {
                hasReturnableRow = true;
            }
            resultTable.add(row);
        }
        lookupForm.setHasReturnableRow(hasReturnableRow);
    }

    /**
     * Validate the milestone expected completion date field as a date
     */
    @Override
    public void validateSearchParameters(Map<String, String> fieldValues) {
        super.validateSearchParameters(fieldValues);

        final String lowerBoundMilestoneExpectedCompletionDate = fieldValues.get(
                KRADConstants.LOOKUP_RANGE_LOWER_BOUND_PROPERTY_PREFIX +
                        ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE);
        validateDateField(lowerBoundMilestoneExpectedCompletionDate,
                KRADConstants.LOOKUP_RANGE_LOWER_BOUND_PROPERTY_PREFIX +
                        ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE,
                getDateTimeService());

        final String upperBoundMilestoneExpectedCompletionDate = fieldValues.get(
                ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE);
        validateDateField(upperBoundMilestoneExpectedCompletionDate,
                ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE, getDateTimeService());
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }
}
