/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.ar.AccountsReceivableSchedule;
import org.kuali.kfs.kns.lookup.KualiLookupableHelperServiceImpl;
import org.kuali.kfs.kns.lookup.LookupUtils;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.rice.core.api.datetime.DateTimeService;
import org.kuali.rice.krad.bo.BusinessObject;

import java.text.ParseException;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Lookupable Helper Service for Milestone and Predetermined Billing Schedule Lookups.
 */
public class ScheduleLookupableHelperServiceImpl extends KualiLookupableHelperServiceImpl {

    private static final String AWARD_ENDING_DATE_KEY = KFSPropertyConstants.AWARD + KFSConstants.DELIMITER
            + KFSPropertyConstants.AWARD_ENDING_DATE;
    private static final String AWARD_ENDING_DATE_FROM_KEY = KRADConstants.LOOKUP_RANGE_LOWER_BOUND_PROPERTY_PREFIX
            + AWARD_ENDING_DATE_KEY;
    private static final List<String> AWARD_KEYS = Arrays.asList(KFSPropertyConstants.AGENCY_NUMBER, AWARD_ENDING_DATE_KEY);

    private DateTimeService dateTimeService;

    /**
     * Overridden so we can tweak how the search is performed.
     *
     * Since some of the search criteria fields are based on the Award linked to the MS/PDBS Schedule, we can't
     * search for them directly, we need to find all the MS/PDBS Schedules that match the criteria specific to
     * the schedule, then filter out Awards that don't match the award specific criteria (if any).
     *
     * Since this is a bounded search, we also need to potentially truncate the results once these previous steps have
     * been done.
     */
    public List<? extends BusinessObject> getSearchResults(Map<String, String> fieldValues) {
        return truncateResultsListIfNecessary(getSearchResultsUnbounded(fieldValues));
    }

    /**
     * Overridden so we can tweak how the search is performed.
     *
     * Since some of the search criteria fields are based on the Award linked to the MS/PDBS Schedule, we can't
     * search for them directly, we need to find all the MS/PDBS Schedules that match the criteria specific to
     * the schedule, then filter out Awards that don't match the award specific criteria (if any).
     */
    public List<? extends BusinessObject> getSearchResultsUnbounded(Map<String, String> fieldValues) {
        Map<String, String> awardFieldValues = fieldValues.entrySet().stream()
                .filter(map -> AWARD_KEYS.contains(map.getKey()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        fieldValues.entrySet().removeAll(awardFieldValues.entrySet());

        List<? extends BusinessObject> schedules = getSearchResultsHelper(
                LookupUtils.forceUppercase(getBusinessObjectClass(), fieldValues), true);

        schedules = filterOutSchedulesWithoutMatchingAgencies(awardFieldValues, schedules);
        schedules = filterOutSchedulesWithoutMatchingAwardEndDate(awardFieldValues, schedules);

        return schedules;
    }

    private List<? extends BusinessObject> filterOutSchedulesWithoutMatchingAgencies(Map<String, String> awardFieldValues,
            List<? extends BusinessObject> schedules) {
        String agencyNumber = awardFieldValues.get(KFSPropertyConstants.AGENCY_NUMBER);
        if (StringUtils.isNotBlank(agencyNumber)) {
            String updatedAgencyNumber = agencyNumber
                    .replace(KFSConstants.WILDCARD_CHARACTER, KFSConstants.DELIMITER + KFSConstants.WILDCARD_CHARACTER)
                    .replace(KFSConstants.PERCENTAGE_SIGN, KFSConstants.DELIMITER + KFSConstants.WILDCARD_CHARACTER)
                    .replace(KFSConstants.QUESTION_MARK, KFSConstants.DELIMITER + KFSConstants.QUESTION_MARK);
            schedules = schedules.stream()
                    .filter(schedule ->
                            ((AccountsReceivableSchedule) schedule).getAward().getAgencyNumber().matches(updatedAgencyNumber))
                    .collect(Collectors.toList());
        }
        return schedules;
    }

    private List<? extends BusinessObject> filterOutSchedulesWithoutMatchingAwardEndDate(Map<String, String> awardFieldValues,
            List<? extends BusinessObject> schedules) {
        String awardEndingDateSearchString = awardFieldValues.get(AWARD_ENDING_DATE_KEY);
        final String awardEndingDateLabel = getDataDictionaryService().getAttributeLabel(getBusinessObjectClass(),
                AWARD_ENDING_DATE_KEY);
        if (StringUtils.isNotBlank(awardEndingDateSearchString)) {
            if (awardEndingDateSearchString.startsWith("<=")) {
                Date awardEndingDateEnd = parseDate(awardEndingDateSearchString.substring(2), AWARD_ENDING_DATE_KEY,
                        awardEndingDateLabel + ArConstants.TO_SUFFIX);
                schedules = schedules.stream()
                    .filter(schedule -> {
                        Date awardEndingDate = ((AccountsReceivableSchedule) schedule).getAward().getAwardEndingDate();
                        return awardEndingDate.before(awardEndingDateEnd) || awardEndingDate.equals(awardEndingDateEnd);
                    })
                    .collect(Collectors.toList());
            } else if (awardEndingDateSearchString.startsWith(">=")) {
                Date awardEndingDateStart = parseDate(awardEndingDateSearchString.substring(2), AWARD_ENDING_DATE_FROM_KEY,
                        awardEndingDateLabel + ArConstants.FROM_SUFFIX);
                schedules = schedules.stream()
                    .filter(schedule -> {
                        Date awardEndingDate = ((AccountsReceivableSchedule) schedule).getAward().getAwardEndingDate();
                        return awardEndingDate.after(awardEndingDateStart) || awardEndingDate.equals(awardEndingDateStart);
                    })
                    .collect(Collectors.toList());
            } else if (awardEndingDateSearchString.contains("..")) {
                String[] awardEndingDates = awardEndingDateSearchString.split("\\.\\.");
                Date awardEndingDateStart = parseDate(awardEndingDates[0], AWARD_ENDING_DATE_FROM_KEY,
                        awardEndingDateLabel + ArConstants.FROM_SUFFIX);
                Date awardEndingDateEnd = parseDate(awardEndingDates[1], AWARD_ENDING_DATE_KEY,
                        awardEndingDateLabel + ArConstants.TO_SUFFIX);
                schedules = schedules.stream()
                    .filter(schedule -> {
                        Date awardEndingDate = ((AccountsReceivableSchedule) schedule).getAward().getAwardEndingDate();
                        return (awardEndingDate.after(awardEndingDateStart) || awardEndingDate.equals(awardEndingDateStart))
                                && (awardEndingDate.before(awardEndingDateEnd) || awardEndingDate.equals(awardEndingDateEnd));
                    })
                    .collect(Collectors.toList());
            }
        }
        return schedules;
    }

    private Date parseDate(String awardEndingDate, String propertyName, String awardEndingDateLabel) {
        Date awardEndingDateStart;
        try {
            awardEndingDateStart = dateTimeService.convertToDate(awardEndingDate);
        } catch (ParseException e) {
            GlobalVariables.getMessageMap().putError(propertyName, KFSKeyConstants.ERROR_DATE_TIME,
                    awardEndingDateLabel);
            throw new ValidationException("errors in search criteria");
        }
        return awardEndingDateStart;
    }

    @Override
    public void validateSearchParameters(Map<String, String> fieldValues) {
        String awardEndingDateFromString = fieldValues.get(AWARD_ENDING_DATE_FROM_KEY);
        String awardEndingDateToString = fieldValues.get(AWARD_ENDING_DATE_KEY);

        final String awardEndingDateLabel =
                getDataDictionaryService().getAttributeLabel(getBusinessObjectClass(), AWARD_ENDING_DATE_KEY);
        validateDateField(awardEndingDateFromString, AWARD_ENDING_DATE_FROM_KEY,
                awardEndingDateLabel + ArConstants.FROM_SUFFIX);
        validateDateField(awardEndingDateToString, AWARD_ENDING_DATE_KEY,
                awardEndingDateLabel + ArConstants.TO_SUFFIX);

        super.validateSearchParameters(fieldValues);
    }

    private void validateDateField(String dateFieldValue, String dateFieldPropertyName, String fieldLabel) {
        if (StringUtils.isNotBlank(dateFieldValue)) {
            try {
                dateTimeService.convertToDate(dateFieldValue);
            } catch (ParseException pe) {
                GlobalVariables.getMessageMap().putError(dateFieldPropertyName, KFSKeyConstants.ERROR_DATE_TIME,
                        fieldLabel);
            }
        }
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }
}
