/**
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2019 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service;

import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.NonInvoicedDistribution;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.report.util.CustomerStatementResultHolder;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.kew.api.WorkflowDocument;

import java.sql.Date;
import java.util.Collection;
import java.util.List;

public interface CustomerInvoiceDocumentService {

    /**
     * Converts discount lines on the customer invoice document to paidapplieds. This method is only intended to be used
     * once the document is at least in the Processed state, and will throw an error if used on a document in an earlier
     * state. This method is intended to be called from the CustomerInvoiceDocument.handleRouteStatusChange
     *
     * @param invoice A populated Invoice document that is at least PROCESSED.
     */
    void convertDiscountsToPaidApplieds(CustomerInvoiceDocument invoice);

    /**
     * Retrieves all invoice documents that are Open with outstanding balances, including workflow headers.
     *
     * @return A collection of CustomerInvoiceDocument documents, or an empty list of no matches.
     */
    Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocuments();

    /**
     * Retrieves all invoice documents that are Open with outstanding balances. Will NOT retrieve workflow headers, so
     * results of this are not suitable for using to route, save, or otherwise perform workflow operations upon.
     *
     * @return
     */
    Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocumentsWithoutWorkflow();

    /**
     * Gets invoices without workflow headers, retrieves the workflow headers and returns invoices with workflow headers.
     *
     * @return
     */
    Collection<CustomerInvoiceDocument> attachWorkflowHeadersToTheInvoices(Collection<CustomerInvoiceDocument> invoices);

    /**
     * Retrieves all Open Invoices for this given Customer Number. IMPORTANT - Workflow headers and status are not
     * retrieved by this method, only the raw Customer Invoice Document from the Database. If you need a full workflow
     * document, you can do use DocumentService to retrieve each by document number.
     *
     * @param customerNumber
     * @return
     */
    Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerNumber(String customerNumber);

    /**
     * Retrieves all Open Invoices for the given Customer Name and Customer Type Code Note that the customerName field
     * is turned into a 'LIKE customerName*' query. IMPORTANT - Workflow headers and status are not retrieved by this
     * method, only the raw Customer Invoice Document from the Database. If you need a full workflow document, you can
     * do use DocumentService to retrieve each by document number.
     *
     * @param customerName
     * @param customerTypeCode
     * @return
     */
    Collection getOpenInvoiceDocumentsByCustomerNameByCustomerType(String customerName, String customerTypeCode);

    /**
     * Retrieves all Open Invoices for the given Customer Name. Note that this is a leading substring search, so whatever
     * is entered into the customerName field is turned into a 'LIKE customerName*' query. IMPORTANT - Workflow headers
     * and status are not retrieved by this method, only the raw Customer Invoice Document from the Database. If you
     * need a full workflow document, you can do use DocumentService to retrieve each by document number.
     *
     * @param customerName
     * @return
     */
    Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerName(String customerName);

    /**
     * Retrieves all Open Invoices for the given Customer Type Code. IMPORTANT - Workflow headers and status are not
     * retrieved by this method, only the raw Customer Invoice Document from the Database. If you need a full workflow
     * document, you can do use DocumentService to retrieve each by document number.
     *
     * @param customerTypeCode
     * @return
     */
    Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerType(String customerTypeCode);

    /**
     * This method sets up default values for customer invoice document on initiation.
     *
     * @param document
     */
    void setupDefaultValuesForNewCustomerInvoiceDocument(CustomerInvoiceDocument document);

    /**
     * This method sets up default values for customer invoice document when copied.
     *
     * @param customerInvoiceDocument
     */
    void setupDefaultValuesForCopiedCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * If the customer number and address identifiers are present, display customer information
     *
     * @param customerInvoiceDocument
     */
    void loadCustomerAddressesForCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * This method returns a Collection of CustomerInvoiceDocuments corresponding to a customerNumber.
     *
     * @param customerNumber used to find invoices
     * @return Collection<CustomerInvoiceDocument> invoice documents
     */
    Collection<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByCustomerNumber(String customerNumber);

    /**
     * This method returns a Collection of CustomerInvoiceDetails for a customer invoice document number.
     *
     * @param customerInvoiceDocumentNumber used to find customer invoice details for a customer invoice document
     * @return Collection<CustomerInvoiceDetail> customer invoice details
     */
    Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocument(
            String customerInvoiceDocumentNumber);

    /**
     * This method returns a Collection of CustomerInvoiceDetails for a customer invoice document.
     *
     * @param customerInvoiceDocument used to find customer invoice details
     * @return Collection<CustomerInvoiceDetail> customer invoice details
     */
    Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocument(
            CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * This method returns a Collection of CustomerInvoiceDetails for a customer invoice document.
     * Cached for better performance
     *
     * @param customerInvoiceDocument used to find customer invoice details
     * @return Collection<CustomerInvoiceDetail> customer invoice details
     */
    Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocumentWithCaching(
            CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * This method returns a Customer for an invoice based on organizationInvoiceNumber.
     *
     * @param invoiceNumber organizationInvoiceNumber used to find invoice and get the customer
     * @return Customer for the invoice corresponding to the organizationInvoiceNumber
     */
    Customer getCustomerByOrganizationInvoiceNumber(String invoiceNumber);

    /**
     * This method returns a CustomerInvoiceDocument based on organizationInvoiceNumber.
     *
     * @param organizationInvoiceNumber used to find invoice and get the customer
     * @return CustomerInvoiceDocument corresponding to the organizationInvoiceNumber
     */
    CustomerInvoiceDocument getInvoiceByOrganizationInvoiceNumber(String organizationInvoiceNumber);

    /**
     * This method returns a Customer for an invoice based on invoice document number.
     *
     * @param documentNumber used to find the invoice and get the customer
     * @return Customer for the invoice corresponding to the documentNumber
     */
    Customer getCustomerByInvoiceDocumentNumber(String documentNumber);

    /**
     * This method returns a CustomerInvoiceDocument based on invoice document number.
     *
     * @param invoiceDocumentNumber used to find the invoice
     * @return CustomerInvoiceDocument corresponding to the invoiceDocumentNumber
     */
    CustomerInvoiceDocument getInvoiceByInvoiceDocumentNumber(String invoiceDocumentNumber);

    /**
     * Gets the printable customer invoice for the specified initiatorPrincipalName.
     *
     * @param initiatorPrincipalName the specified initiatorPrincipalName
     * @return the printable customer invoice for the specified initiatorPrincipalName
     */
    List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByInitiatorPrincipalName(String initiatorPrincipalName);

    /**
     * Gets the printable customer invoice for the specified chartOfAccountsCode and organizationCode.
     *
     * @param chartOfAccountsCode
     * @param organizationCode
     * @return the printable customer invoice for the specified chartOfAccountsCode and organizationCode.
     */
    List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByBillingChartAndOrg(String chartOfAccountsCode,
            String organizationCode);

    /**
     * @param chartOfAccountsCode
     * @param organizationCode
     * @return the printable customer invoice for the BillingStatement by chartOfAccountsCode and organizationCode.
     */
    List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsForBillingStatementByBillingChartAndOrg(
            String chartOfAccountsCode, String organizationCode);

    /**
     * @param chartOfAccountsCode
     * @param organizationCode
     * @return the printable customer invoice for the specified chartOfAccountsCode and organizationCode.
     */
    List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByProcessingChartAndOrg(
            String chartOfAccountsCode, String organizationCode);

    /**
     * @param chartOfAccountsCode
     * @param organizationCode
     * @return customer invoice for the specified chartOfAccountsCode and organizationCode.
     */
    List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByBillingChartAndOrg(String chartOfAccountsCode,
            String organizationCode);

    /**
     * Get list of customer invoice by processing chartOfAccountsCode and organizationCode.
     *
     * @param chartOfAccountsCode
     * @param organizationCode
     * @return list of customer invoice
     */
    List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByProcessingChartAndOrg(String chartOfAccountsCode,
            String organizationCode);

    /**
     * Gets customer invoice by account number.
     *
     * @param accountNumber
     * @return collection of customer invoice.
     */
    Collection<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByAccountNumber(String accountNumber);

    /**
     * This method returns the NonInvoicedDistributions for an invoice based on documentNumber.
     *
     * @param documentNumber used to find the invoice
     * @return Collection<NonInvoicedDistribution> non invoiced distributions for the invoice corresponding to the
     *         documentNumber
     */
    Collection<NonInvoicedDistribution> getNonInvoicedDistributionsForInvoice(String documentNumber);

    /**
     * This method returns the total NonInvoicedDistribution amount for an invoice based on documentNumber.
     *
     * @param documentNumber used to find the invoice
     * @return KualiDecimal non invoiced distribution total for the invoice
     */
    KualiDecimal getNonInvoicedTotalForInvoice(String documentNumber);

    /**
     * This method returns the total NonInvoicedDistribution amount for an invoice.
     *
     * @param invoice used to get the non invoiced distributions to total
     * @return KualiDecimal non invoiced distribution total for the invoice
     */
    KualiDecimal getNonInvoicedTotalForInvoice(CustomerInvoiceDocument invoice);

    /**
     * This method returns the total InvoicePaidApplied amount for an invoice based on documentNumber.
     *
     * @param documentNumber used to find the invoice
     * @return KualiDecimal paid applied total for the invoice
     */
    KualiDecimal getPaidAppliedTotalForInvoice(String documentNumber);

    /**
     * This method returns the total InvoicePaidApplied amount for an invoice.
     *
     * @param invoice used to get the invoice paid applied objects to total
     * @return KualiDecimal paid applied total for the invoice
     */
    KualiDecimal getPaidAppliedTotalForInvoice(CustomerInvoiceDocument invoice);

    /**
     * This method updates the open invoice indicator if amounts have been completely paid off
     *
     * @param customerInvoiceDocument
     */
    void closeCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * This method returns the total open amount for an invoice based on customerInvoiceDocumentNumber.
     *
     * @param customerInvoiceDocumentNumber used to find the invoice
     * @return KualiDecimal total open amount for the invoice
     */
    KualiDecimal getOpenAmountForCustomerInvoiceDocument(String customerInvoiceDocumentNumber);


    /**
     * This method returns the total open amount for an invoice based on customerInvoiceDocumentNumber.
     *
     * @param customerInvoiceDocument invoice used to calculate total open amount
     * @return KualiDecimal total open amount for the invoice
     */
    KualiDecimal getOpenAmountForCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * get total amount of customer invoice by customer invoice.
     *
     * @param customerInvoiceDocument
     * @return
     */
    KualiDecimal getOriginalTotalAmountForCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument);

    /**
     * This method checks if the invoice is final.
     *
     * @param invDocumentNumber invoice to check
     * @return true if invoice is final, false otherwise
     */
    boolean checkIfInvoiceNumberIsFinal(String invDocumentNumber);

    /**
     * Updates report date
     *
     * @param docNumber
     */
    void updateReportedDate(String docNumber);

    /**
     * Updates statement info
     *
     * @param data
     */
    void updateReportedInvoiceInfo(CustomerStatementResultHolder data);

    /**
     * get all customer invoice documents that are open and with the given age and customer types
     *
     * @param customerTypes      the given customer types
     * @param invoiceAge         the given invoice document age
     * @param invoiceDueDateFrom the given invoice billing from date
     * @return all customer invoice documents that are open and with the given age
     */
    Collection<CustomerInvoiceDocument> getAllAgingInvoiceDocumentsByCustomerTypes(List<String> customerTypes,
            Integer invoiceAge, Date invoiceDueDateFrom);

    /**
     * Adds a note to the CustomerInvoiceDocument about what person and document numbers closed it. This only occurs if
     * openInvoiceIndicator=true at the time of calling this method.
     *
     * @param documentToClose the document about to be closed which we want to add the note too
     * @param closingDocument the closing document
     */
    void addCloseNote(CustomerInvoiceDocument documentToClose, WorkflowDocument closingDocument);

    /**
     * Calculates the total amount of applied payments - either applied via payment application or customer credit memo
     * documents - against the given invoice.
     *
     * @param invoice the invoice to find a payment amount for
     * @return the applied payment amount
     */
    KualiDecimal calculateAppliedPaymentAmount(CustomerInvoiceDocument invoice);
}
