/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.ar.AccountsReceivableModuleBillingService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAwardAccount;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.datadictionary.validation.result.DictionaryValidationResult;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.Bill;
import org.kuali.kfs.module.ar.businessobject.PredeterminedBillingSchedule;
import org.kuali.kfs.module.ar.document.service.PredeterminedBillingScheduleMaintenanceService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.rice.core.api.util.type.AbstractKualiDecimal;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class PredeterminedBillingScheduleRule extends MaintenanceDocumentRuleBase {

    private static final Logger LOG = LogManager.getLogger();

    PredeterminedBillingSchedule newPredeterminedBillingScheduleCopy;

    private static volatile AccountsReceivableModuleBillingService accountsReceivableModuleBillingService;
    private static volatile PredeterminedBillingScheduleMaintenanceService predeterminedBillingScheduleMaintenanceService;

    @Override
    public boolean processCustomAddCollectionLineBusinessRules(MaintenanceDocument document, String collectionName,
            PersistableBusinessObject line) {
        LOG.debug("Entering PredeterminedBillingScheduleRule.processCustomAddCollectionLineBusinessRules");

        boolean isValid = checkForDuplicateBillNumber(collectionName, line);

        LOG.debug("Leaving PredeterminedBillingScheduleRule.processCustomAddCollectionLineBusinessRules");

        return isValid;
    }

    /**
     * Check to see if a Bill with the same bill number already exists.
     *
     * @param collectionName name of the collection being added to
     * @param line           PersistableBusinessObject being added to the collection
     * @return true if there isn't already a bill with the same bill number, false otherwise
     */
    private boolean checkForDuplicateBillNumber(String collectionName, PersistableBusinessObject line) {
        boolean isValid = true;

        if (StringUtils.equalsIgnoreCase(collectionName, ArConstants.BILL_SECTION)) {
            Bill bill = (Bill) line;
            String newBillNumber = bill.getBillNumber();

            for (Bill existingBill : newPredeterminedBillingScheduleCopy.getBills()) {
                if (existingBill.getBillNumber().equals(newBillNumber)) {
                    isValid = false;
                    putFieldError(collectionName, ArKeyConstants.ERROR_DUPLICATE_BILL_NUMBER);
                    break;
                }
            }
        }

        return isValid;
    }

    @Override
    protected boolean processCustomSaveDocumentBusinessRules(MaintenanceDocument document) {
        LOG.debug("Entering PredeterminedBillingScheduleRule.processCustomSaveDocumentBusinessRules");

        processCustomRouteDocumentBusinessRules(document);

        LOG.debug("Leaving PredeterminedBillingScheduleRule.processCustomSaveDocumentBusinessRules");

        // save despite error messages
        return true;
    }

    @Override
    protected boolean processCustomRouteDocumentBusinessRules(MaintenanceDocument document) {
        LOG.debug("Entering PredeterminedBillingScheduleRule.processCustomRouteDocumentBusinessRules");

        boolean success = checkAward();
        success &= checkPredeterminedBillingScheduleAccount();
        success &= checkAwardBillingFrequency();
        success &= checkForDuplicateBillNumbers();
        success &= checkBillTotalAgainstAwardTotal();
        success &= performDataDictionaryValidationForBills();

        LOG.debug("Leaving PredeterminedBillingScheduleRule.processCustomRouteDocumentBusinessRules");

        return success;
    }

    private boolean checkAward() {
        if (ObjectUtils.isNull(newPredeterminedBillingScheduleCopy.getAward())
                || StringUtils.isBlank(newPredeterminedBillingScheduleCopy.getAward().getProposalNumber())) {
            putFieldError(ArPropertyConstants.PROPOSAL_NUMBER_FOR_AWARD_ACCOUNT_LOOKUP,
                    ArKeyConstants.ERROR_AWARD_DOES_NOT_EXIST, newPredeterminedBillingScheduleCopy.getProposalNumber());
            return false;
        }

        return true;
    }

    private boolean checkPredeterminedBillingScheduleAccount() {
        if (ObjectUtils.isNull(newPredeterminedBillingScheduleCopy.getAward())) {
            return true;
        }

        List<ContractsAndGrantsBillingAwardAccount> activeAwardAccounts = newPredeterminedBillingScheduleCopy.getAward()
                .getActiveAwardAccounts();
        for (ContractsAndGrantsBillingAwardAccount account : activeAwardAccounts) {
            if (StringUtils.equals(account.getChartOfAccountsCode(),
                    newPredeterminedBillingScheduleCopy.getChartOfAccountsCode())
                    && StringUtils.equals(account.getAccountNumber(),
                    newPredeterminedBillingScheduleCopy.getAccountNumber())) {
                return true;
            }
        }
        putFieldError(KFSPropertyConstants.ACCOUNT_NUMBER,
                ArKeyConstants.ERROR_PREDETERMINED_BILLING_SCHEDULE_ACCOUNT_DOES_NOT_EXIST_ON_AWARD);
        return false;
    }

    /**
     * checks to see if the billing frequency on the award is Predetermined Billing
     *
     * @return true if if the billing frequency on the award is Predetermined Billing, false otherwise
     */
    private boolean checkAwardBillingFrequency() {
        boolean success = false;
        if (ObjectUtils.isNull(newPredeterminedBillingScheduleCopy.getAward()) ||
                StringUtils.isBlank(newPredeterminedBillingScheduleCopy.getProposalNumber())) {
            return true;
        }
        if (ObjectUtils.isNotNull(newPredeterminedBillingScheduleCopy.getAward().getBillingFrequencyCode())) {
            if (ArConstants.BillingFrequencyValues.isPredeterminedBilling(
                    newPredeterminedBillingScheduleCopy.getAward())) {
                success = true;
            }
        }

        if (!success) {
            putFieldError(ArPropertyConstants.PROPOSAL_NUMBER_FOR_AWARD_ACCOUNT_LOOKUP,
                    ArKeyConstants.ERROR_AWARD_PREDETERMINED_BILLING_SCHEDULE_INCORRECT_BILLING_FREQUENCY,
                    new String[]{newPredeterminedBillingScheduleCopy.getProposalNumber()});
        }

        return success;
    }

    /**
     * Check to see if there is more than one Bill with the same bill number.
     *
     * @return true if there is more than one bill with the same bill number, false otherwise
     */
    private boolean checkForDuplicateBillNumbers() {
        boolean isValid = true;

        Set<String> billNumbers = new HashSet<>();
        Set<String> duplicateBillNumbers = new HashSet<>();

        for (Bill bill : newPredeterminedBillingScheduleCopy.getBills()) {
            if (!billNumbers.add(bill.getBillNumber())) {
                duplicateBillNumbers.add(bill.getBillNumber());
            }
        }

        if (duplicateBillNumbers.size() > 0) {
            isValid = false;
            int lineNum = 0;
            for (Bill bill : newPredeterminedBillingScheduleCopy.getBills()) {
                // If the Bill has already been copied to the Invoice, it will be readonly, the user won't have been
                // able to change it and thus we don't need to highlight it as an error if it's a dupe. There will be
                // another dupe in the list that we will highlight.
                boolean copiedToInvoice = false;
                if (ObjectUtils.isNotNull(bill.getBillIdentifier())) {
                    if (getPredeterminedBillingScheduleMaintenanceService().hasBillBeenCopiedToInvoice(
                            bill.getProposalNumber(), bill.getBillIdentifier().toString())) {
                        copiedToInvoice = true;
                    }
                }

                if (!copiedToInvoice) {
                    if (duplicateBillNumbers.contains(bill.getBillNumber())) {
                        String errorPath = ArPropertyConstants.PredeterminedBillingScheduleFields.BILLS + "["
                                + lineNum + "]." + ArPropertyConstants.BillFields.BILL_NUMBER;
                        putFieldError(errorPath, ArKeyConstants.ERROR_DUPLICATE_BILL_NUMBER);
                    }
                }
                lineNum++;
            }
        }
        return isValid;
    }

    /**
     * Check to see if the bill total for this schedule and other schedules associated with this award
     * is less than the award total amount.
     *
     * @return true if the bill total is less than the award total, false otherwise
     */
    private boolean checkBillTotalAgainstAwardTotal() {
        if (ObjectUtils.isNull(newPredeterminedBillingScheduleCopy.getAward())
                || ObjectUtils.isNull(newPredeterminedBillingScheduleCopy.getAward().getAwardTotalAmount())) {
            return true;
        }
        final KualiDecimal awardTotalAmount = newPredeterminedBillingScheduleCopy.getAward().getAwardTotalAmount();
        KualiDecimal billTotalAmount = newPredeterminedBillingScheduleCopy.getBills().stream()
                .filter(Bill::isActive)
                .filter(bill -> ObjectUtils.isNotNull(bill.getEstimatedAmount()))
                .reduce(KualiDecimal.ZERO, (sum, bill) -> bill.getEstimatedAmount().add(sum), AbstractKualiDecimal::add);

        KualiDecimal totalAmountFromOtherPredeterminedBillingSchedules = getAccountsReceivableModuleBillingService()
                .getBillsTotalAmountForOtherSchedules(newPredeterminedBillingScheduleCopy.getProposalNumber(),
                        newPredeterminedBillingScheduleCopy.getChartOfAccountsCode(),
                        newPredeterminedBillingScheduleCopy.getAccountNumber());

        final KualiDecimal totalForAllBills = billTotalAmount.add(totalAmountFromOtherPredeterminedBillingSchedules);
        if (totalForAllBills.isGreaterThan(awardTotalAmount)) {
            putFieldError(ArPropertyConstants.ScheduleFields.TOTAL_SCHEDULED_ACCOUNT,
                    ArKeyConstants.ERROR_BILL_TOTAL_EXCEEDS_AWARD_TOTAL);
            return false;
        } else {
            return true;
        }
    }

    /**
     * Performs Data Dictionary validation on each bill for this schedule. This is most useful for bills that have
     * been created as part of a Predetermined Billing Schedule Upload in case the data uploaded parses ok, but
     * violates Data Dictionary constraints.
     *
     * @return true if bills are valid, false otherwise
     */
    private boolean performDataDictionaryValidationForBills() {
        boolean valid = true;
        int lineNum = 0;
        for (Bill bill : newPredeterminedBillingScheduleCopy.getBills()) {
            GlobalVariables.getMessageMap().addToErrorPath(KFSConstants.MAINTENANCE_NEW_MAINTAINABLE +
                    ArPropertyConstants.PredeterminedBillingScheduleFields.BILLS + "[" + lineNum + "]");
            DictionaryValidationResult dictionaryValidationResult = getDictionaryValidationService().validate(bill);
            if (dictionaryValidationResult.getNumberOfErrors() > 0) {
                valid = false;
            }
            GlobalVariables.getMessageMap().removeFromErrorPath(KFSConstants.MAINTENANCE_NEW_MAINTAINABLE +
                    ArPropertyConstants.PredeterminedBillingScheduleFields.BILLS + "[" + lineNum + "]");
            lineNum++;
        }
        return valid;
    }

    @Override
    public void setupConvenienceObjects() {
        newPredeterminedBillingScheduleCopy = (PredeterminedBillingSchedule) super.getNewBo();
    }

    AccountsReceivableModuleBillingService getAccountsReceivableModuleBillingService() {
        if (accountsReceivableModuleBillingService == null) {
            accountsReceivableModuleBillingService =
                    SpringContext.getBean(AccountsReceivableModuleBillingService.class);
        }
        return accountsReceivableModuleBillingService;
    }

    PredeterminedBillingScheduleMaintenanceService getPredeterminedBillingScheduleMaintenanceService() {
        if (predeterminedBillingScheduleMaintenanceService == null) {
            predeterminedBillingScheduleMaintenanceService =
                    SpringContext.getBean(PredeterminedBillingScheduleMaintenanceService.class);
        }
        return predeterminedBillingScheduleMaintenanceService;
    }
}
