/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.web.struts;

import com.fasterxml.jackson.annotation.JsonIgnore;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.sys.rest.presentation.Link;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class PaymentApplicationInvoiceApply implements Serializable {

    private CustomerInvoiceDocument invoice;
    private List<PaymentApplicationInvoiceDetailApply> detailApplications;

    private String payAppDocNumber;
    private boolean quickApply;
    private boolean quickApplyOldValue;
    private String documentUrl;
    private String customerUrl;

    public PaymentApplicationInvoiceApply(String payAppDocNumber, CustomerInvoiceDocument invoice) {
        this.invoice = invoice;
        this.detailApplications = new ArrayList<>();
        for (CustomerInvoiceDetail invoiceDetail : invoice.getCustomerInvoiceDetailsWithoutDiscounts()) {
            this.detailApplications.add(new PaymentApplicationInvoiceDetailApply(payAppDocNumber, invoiceDetail));
        }
        this.quickApply = false;
        this.quickApplyOldValue = false;
        this.payAppDocNumber = payAppDocNumber;
    }

    public KualiDecimal getAmountToApply() {
        KualiDecimal applyAmount = KualiDecimal.ZERO;
        for (PaymentApplicationInvoiceDetailApply detailApplication : detailApplications) {
            applyAmount = applyAmount.add(detailApplication.getAmountApplied());
        }
        return applyAmount;
    }

    // yes this method name is awkward.  Blame JSP that expects an is* or get*
    @JsonIgnore
    public boolean isAnyAppliedAmounts() {
        for (PaymentApplicationInvoiceDetailApply detailApplication : detailApplications) {
            if (detailApplication.getAmountApplied().isGreaterThan(KualiDecimal.ZERO)) {
                return true;
            }
        }
        return false;
    }

    public List<PaymentApplicationInvoiceDetailApply> getDetailApplications() {
        return detailApplications;
    }

    public String getDocumentNumber() {
        return invoice.getDocumentNumber();
    }

    public KualiDecimal getTotalAmount() {
        return invoice.getTotalDollarAmount();
    }

    public KualiDecimal getOpenAmount() {
        return invoice.getOpenAmount();
    }

    public Date getInvoiceBillingDate() {
        return invoice.getBillingDate();
    }

    public String getCustomerName() {
        return invoice.getCustomer().getCustomerName();
    }

    public String getCustomerNumber() {
        return invoice.getCustomer().getCustomerNumber();
    }

    public String getHeaderText() {
        return invoice.getInvoiceHeaderText();
    }

    @JsonIgnore
    public boolean isQuickApply() {
        return quickApply;
    }

    public void setQuickApply(boolean quickApply) {
        this.quickApplyOldValue = this.quickApply;
        this.quickApply = quickApply;
        for (PaymentApplicationInvoiceDetailApply detailApplication : detailApplications) {
            detailApplication.setInvoiceQuickApplied(quickApply);
        }
    }

    @JsonIgnore
    public boolean isQuickApplyChanged() {
        return quickApply != quickApplyOldValue;
    }

    @JsonIgnore
    public CustomerInvoiceDocument getInvoice() {
        return invoice;
    }

    public String getDocumentUrl() {
        return documentUrl;
    }

    public void setDocumentUrl(String documentUrl) {
        this.documentUrl = documentUrl;
    }

    public String getCustomerUrl() {
        return customerUrl;
    }

    public void setCustomerUrl(String customerUrl) {
        this.customerUrl = customerUrl;
    }

    public Link getCustomerLink() {
        return new Link(getCustomerNumber(), getCustomerUrl());
    }

    public Link getDocumentLink() {
        return new Link(getDocumentNumber(), getDocumentUrl());
    }
}
