/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.rest.resource.requests;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;

@JsonIgnoreProperties(ignoreUnknown = true)
public class PaymentApplicationAdjustmentRequest extends DocumentRequest {
    private final List<InvoiceApplication> invoiceApplications;
    private final List<NonAppliedHolding> nonAppliedHoldings;

    @JsonCreator
    public PaymentApplicationAdjustmentRequest(
            @JsonProperty("description") final String description,
            @JsonProperty("explanation") final String explanation,
            @JsonProperty("orgDocNumber") final String orgDocNumber,
            @JsonProperty("adHocRouting") final AdHocRoutingRequest adHocRoutingRequest,
            @JsonProperty("invoiceApplications") final List<InvoiceApplication> invoiceApplications,
            @JsonProperty("nonAppliedHoldings") final List<NonAppliedHolding> nonAppliedHoldings) {
        super(description, explanation, orgDocNumber, adHocRoutingRequest);
        Validate.isTrue(invoiceApplications != null, "invoiceApplications must be supplied");
        Validate.isTrue(nonAppliedHoldings != null, "nonAppliedHoldings must be supplied");
        this.invoiceApplications = invoiceApplications;
        this.nonAppliedHoldings = nonAppliedHoldings;
    }

    public List<NonAppliedHolding> getNonAppliedHoldings() {
        return nonAppliedHoldings;
    }

    public List<InvoiceApplication> getInvoiceApplications() {
        return invoiceApplications;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final PaymentApplicationAdjustmentRequest that = (PaymentApplicationAdjustmentRequest) o;
        return invoiceApplications.equals(that.invoiceApplications) &&
                nonAppliedHoldings.equals(that.nonAppliedHoldings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(invoiceApplications, nonAppliedHoldings);
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", PaymentApplicationAdjustmentRequest.class.getSimpleName() + "[", "]")
                .add("invoiceApplications=" + invoiceApplications)
                .add("nonAppliedHoldings=" + nonAppliedHoldings)
                .toString();
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class InvoiceApplication {
        private final String customerNumber;
        private final String documentNumber;
        private final List<InvoiceApplicationDetail> detailApplications;

        @JsonCreator
        public InvoiceApplication(
                @JsonProperty("customerNumber") final String customerNumber,
                @JsonProperty("documentNumber") final String documentNumber,
                @JsonProperty("detailApplications") final List<InvoiceApplicationDetail> detailApplications
        ) {
            Validate.isTrue(StringUtils.isNotBlank(customerNumber), "customerNumber must be supplied");
            Validate.isTrue(StringUtils.isNotBlank(documentNumber), "documentNumber must be supplied");
            Validate.isTrue(CollectionUtils.isNotEmpty(detailApplications), "detailApplications must be supplied");
            this.customerNumber = customerNumber;
            this.documentNumber = documentNumber;
            this.detailApplications = detailApplications;
        }

        public String getCustomerNumber() {
            return customerNumber;
        }

        public String getDocumentNumber() {
            return documentNumber;
        }

        public List<InvoiceApplicationDetail> getDetailApplications() {
            return detailApplications;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final InvoiceApplication that = (InvoiceApplication) o;
            return customerNumber.equals(that.customerNumber) &&
                    documentNumber.equals(that.documentNumber) &&
                    detailApplications.equals(that.detailApplications);
        }

        @Override
        public int hashCode() {
            return Objects.hash(customerNumber, documentNumber, detailApplications);
        }

        @Override
        public String toString() {
            return new StringJoiner(", ",
                    InvoiceApplication.class.getSimpleName() + "[", "]")
                    .add("customerNumber='" + customerNumber + "'")
                    .add("detailApplications=" + detailApplications)
                    .add("documentNumber='" + documentNumber + "'")
                    .toString();
        }

    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class InvoiceApplicationDetail {
        private final KualiDecimal amountApplied;
        private final Integer sequenceNumber;

        @JsonCreator
        public InvoiceApplicationDetail(
                @JsonProperty("amountApplied") final KualiDecimal amountApplied,
                @JsonProperty("sequenceNumber") final Integer sequenceNumber
        ) {
            Validate.isTrue(amountApplied != null, "amountApplied must be supplied");
            Validate.isTrue(sequenceNumber != null, "sequenceNumber must be supplied");
            Validate.isTrue(sequenceNumber >= 0, "sequenceNumber must not be negative");
            this.amountApplied = amountApplied;
            this.sequenceNumber = sequenceNumber;
        }

        public KualiDecimal getAmountApplied() {
            return amountApplied;
        }

        public Integer getSequenceNumber() {
            return sequenceNumber;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final InvoiceApplicationDetail that =
                    (InvoiceApplicationDetail) o;
            return amountApplied.equals(that.amountApplied) &&
                    sequenceNumber.equals(that.sequenceNumber);
        }

        @Override
        public int hashCode() {
            return Objects.hash(amountApplied, sequenceNumber);
        }

        @Override
        public String toString() {
            return new StringJoiner(", ",
                    InvoiceApplicationDetail.class.getSimpleName() + "[", "]")
                    .add("amountApplied=" + amountApplied)
                    .add("sequenceNumber=" + sequenceNumber)
                    .toString();
        }

    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class NonAppliedHolding {
        private final KualiDecimal amount;
        private final String customerNumber;

        @JsonCreator
        public NonAppliedHolding(
                @JsonProperty("amount") final KualiDecimal amount,
                @JsonProperty("customerNumber") final String customerNumber
        ) {
            Validate.isTrue(amount != null, "amount must be supplied");
            Validate.isTrue(StringUtils.isNotBlank(customerNumber), "customerNumber must be supplied");
            this.amount = amount;
            this.customerNumber = customerNumber;
        }

        public KualiDecimal getAmount() {
            return amount;
        }

        public String getCustomerNumber() {
            return customerNumber;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final NonAppliedHolding that = (NonAppliedHolding) o;
            return amount.equals(that.amount) &&
                    customerNumber.equals(that.customerNumber);
        }

        @Override
        public int hashCode() {
            return Objects.hash(amount, customerNumber);
        }

        @Override
        public String toString() {
            return new StringJoiner(", ", NonAppliedHolding.class.getSimpleName() + "[", "]")
                    .add("amount=" + amount)
                    .add("customerNumber='" + customerNumber + "'")
                    .toString();
        }

    }
}

