/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject;

import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

/**
 * This class represents a invoice detail on the customer invoice document.
 */
public class ContractsGrantsInvoiceDetail extends PersistableBusinessObjectBase {

    private Long invoiceDetailIdentifier;
    private String documentNumber;
    private String categoryCode;
    private KualiDecimal totalBudget = KualiDecimal.ZERO;
    private KualiDecimal invoiceAmount = KualiDecimal.ZERO;
    private KualiDecimal cumulativeExpenditures = KualiDecimal.ZERO;
    private KualiDecimal totalPreviouslyBilled = KualiDecimal.ZERO;
    private boolean indirectCostIndicator;

    private ContractsGrantsInvoiceDocument invoiceDocument;
    private CostCategory costCategory;

    /**
     * Gets the categoryCode attribute.
     *
     * @return Returns the categoryCode.
     */
    public String getCategoryCode() {
        return categoryCode;
    }

    /**
     * Sets the categoryCode attribute value.
     *
     * @param categoryCode The categoryCode to set.
     */
    public void setCategoryCode(String categoryCode) {
        this.categoryCode = categoryCode;
    }

    /**
     * Gets the invoiceDetailIdentifier attribute.
     *
     * @return Returns the invoiceDetailIdentifier.
     */
    public Long getInvoiceDetailIdentifier() {
        return invoiceDetailIdentifier;
    }

    /**
     * Sets the invoiceDetailIdentifier attribute value.
     *
     * @param invoiceDetailIdentifier The invoiceDetailIdentifier to set.
     */
    public void setInvoiceDetailIdentifier(Long invoiceDetailIdentifier) {
        this.invoiceDetailIdentifier = invoiceDetailIdentifier;
    }

    /**
     * Gets the documentNumber attribute.
     *
     * @return Returns the documentNumber.
     */
    public String getDocumentNumber() {
        return documentNumber;
    }

    /**
     * Sets the documentNumber attribute value.
     *
     * @param documentNumber The documentNumber to set.
     */
    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }

    /**
     * Gets the totalBudget attribute.
     *
     * @return Returns the totalBudget.
     */
    public KualiDecimal getTotalBudget() {
        return totalBudget;
    }

    /**
     * Sets the totalBudget attribute value.
     *
     * @param totalBudget The totalBudget to set.
     */
    public void setTotalBudget(KualiDecimal totalBudget) {
        this.totalBudget = totalBudget;
    }

    /**
     * Gets the invoiceAmount attribute.
     *
     * @return Returns the invoiceAmount.
     */
    public KualiDecimal getInvoiceAmount() {
        return invoiceAmount;
    }

    /**
     * Sets the invoiceAmount attribute value.
     *
     * @param invoiceAmount The invoiceAmount to set.
     */
    public void setInvoiceAmount(KualiDecimal invoiceAmount) {
        this.invoiceAmount = invoiceAmount;
    }

    /**
     * Gets the cumulativeExpenditures attribute.
     *
     * @return Returns the cumulativeExpenditures.
     */
    public KualiDecimal getCumulativeExpenditures() {
        return cumulativeExpenditures;
    }

    /**
     * Sets the cumulativeExpenditures attribute value.
     *
     * @param cumulativeExpenditures The cumulativeExpenditures to set.
     */
    public void setCumulativeExpenditures(KualiDecimal cumulativeExpenditures) {
        this.cumulativeExpenditures = cumulativeExpenditures;
    }

    public KualiDecimal getBudgetRemaining() {
        // Balance = Budget-Cumulative
        return totalBudget.subtract(cumulativeExpenditures);
    }

    public KualiDecimal getTotalPreviouslyBilled() {
        return totalPreviouslyBilled;
    }

    public void setTotalPreviouslyBilled(KualiDecimal totalPreviouslyBilled) {
        this.totalPreviouslyBilled = totalPreviouslyBilled;
    }

    public ContractsGrantsInvoiceDocument getInvoiceDocument() {
        return invoiceDocument;
    }

    public void setInvoiceDocument(ContractsGrantsInvoiceDocument invoiceDocument) {
        this.invoiceDocument = invoiceDocument;
    }

    public boolean isIndirectCostIndicator() {
        return indirectCostIndicator;
    }

    public void setIndirectCostIndicator(boolean indirectCostIndicator) {
        this.indirectCostIndicator = indirectCostIndicator;
    }

    public CostCategory getCostCategory() {
        return costCategory;
    }

    public void setCostCategory(CostCategory costCategory) {
        this.costCategory = costCategory;
    }

    /**
     * @return the calculated total amount billed to date (the total previously billed minus the invoice amount)
     */
    public KualiDecimal getTotalAmountBilledToDate() {
        return totalPreviouslyBilled.add(invoiceAmount);
    }

    /**
     * @return the calculated amount remaining to bill (the total budget minus the amount billed to date)
     */
    public KualiDecimal getAmountRemainingToBill() {
        return totalBudget.subtract(getTotalAmountBilledToDate());
    }

    /**
     * Adds the values from the given ContractsGrantsInvoiceDetail onto this one
     *
     * @param contractsGrantsInvoiceDetail the detail to sum into this
     */
    public void sumInvoiceDetail(ContractsGrantsInvoiceDetail contractsGrantsInvoiceDetail) {
        if (null != contractsGrantsInvoiceDetail.getTotalBudget()) {
            totalBudget = totalBudget.add(contractsGrantsInvoiceDetail.getTotalBudget());
        }
        if (null != contractsGrantsInvoiceDetail.getCumulativeExpenditures()) {
            cumulativeExpenditures = cumulativeExpenditures.add(contractsGrantsInvoiceDetail.getCumulativeExpenditures());
        }
        if (null != contractsGrantsInvoiceDetail.getInvoiceAmount()) {
            invoiceAmount = invoiceAmount.add(contractsGrantsInvoiceDetail.getInvoiceAmount());
        }
        if (null != contractsGrantsInvoiceDetail.getTotalPreviouslyBilled()) {
            totalPreviouslyBilled = totalPreviouslyBilled.add(contractsGrantsInvoiceDetail.getTotalPreviouslyBilled());
        }
    }
}
