/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.actiontaken.ActionTaken;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.action.ActionType;
import org.kuali.kfs.kew.api.exception.WorkflowException;
import org.kuali.kfs.kim.api.identity.IdentityService;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.principal.Principal;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.dao.DocumentDao;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.exception.InfrastructureException;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.CustomerBillingStatement;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceRecurrenceDetails;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.businessobject.NonInvoicedDistribution;
import org.kuali.kfs.module.ar.businessobject.OrganizationOptions;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.dataaccess.CustomerInvoiceDocumentDao;
import org.kuali.kfs.module.ar.document.service.AccountsReceivableDocumentHeaderService;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.InvoicePaidAppliedService;
import org.kuali.kfs.module.ar.document.service.NonInvoicedDistributionService;
import org.kuali.kfs.module.ar.report.util.CustomerStatementResultHolder;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.ChartOrgHolder;
import org.kuali.kfs.sys.businessobject.FinancialSystemDocumentHeader;
import org.kuali.kfs.sys.service.FinancialSystemUserService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.sys.util.KfsDateUtils;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Date;
import java.sql.Timestamp;
import java.text.MessageFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Stream;

@Transactional
public class CustomerInvoiceDocumentServiceImpl implements CustomerInvoiceDocumentService {

    private static final Logger LOG = LogManager.getLogger();

    protected AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService;
    protected BusinessObjectService businessObjectService;
    protected CustomerAddressService customerAddressService;
    protected CustomerInvoiceDetailService customerInvoiceDetailService;
    protected CustomerInvoiceDocumentDao customerInvoiceDocumentDao;
    protected ConfigurationService configurationService;
    protected DateTimeService dateTimeService;
    protected DocumentService documentService;
    protected DocumentDao documentDao;
    protected FinancialSystemUserService financialSystemUserService;
    protected InvoicePaidAppliedService<CustomerInvoiceDetail> invoicePaidAppliedService;
    protected NonInvoicedDistributionService nonInvoicedDistributionService;
    protected PersonService personService;
    protected UniversityDateService universityDateService;
    protected NoteService noteService;
    protected IdentityService identityService;

    @Override
    public void convertDiscountsToPaidApplieds(CustomerInvoiceDocument invoice) {
        // this needs a little explanation. we have to calculate manually whether we've written off the whole thing,
        // because the regular code uses the invoice paid applieds to discount, but since those are added but not
        // committed in this transaction, they're also not visible in this transaction, so we do it manually.
        KualiDecimal openAmount = invoice.getOpenAmount();

        String invoiceNumber = invoice.getDocumentNumber();
        List<CustomerInvoiceDetail> discounts = invoice.getDiscounts();

        // retrieve the number of current paid applieds, so we dont have item number overlap
        Integer paidAppliedItemNumber = 0;

        for (CustomerInvoiceDetail discount : discounts) {
            // if credit amount is zero, do nothing
            if (KualiDecimal.ZERO.equals(discount.getAmount())) {
                continue;
            }

            if (paidAppliedItemNumber == 0) {
                paidAppliedItemNumber = invoicePaidAppliedService.getNumberOfInvoicePaidAppliedsForInvoiceDetail(
                        invoiceNumber, discount.getInvoiceItemNumber());
            }

            // create and save the paidApplied
            InvoicePaidApplied invoicePaidApplied = new InvoicePaidApplied();
            invoicePaidApplied.setDocumentNumber(invoiceNumber);
            invoicePaidApplied.setPaidAppliedItemNumber(paidAppliedItemNumber++);
            invoicePaidApplied.setFinancialDocumentReferenceInvoiceNumber(invoiceNumber);
            invoicePaidApplied.setInvoiceItemNumber(discount.getInvoiceItemNumber());
            invoicePaidApplied.setUniversityFiscalYear(universityDateService.getCurrentFiscalYear());
            invoicePaidApplied.setUniversityFiscalPeriodCode(universityDateService.getCurrentUniversityDate()
                    .getUniversityFiscalAccountingPeriod());
            invoicePaidApplied.setInvoiceItemAppliedAmount(discount.getAmount().abs());
            openAmount = openAmount.subtract(discount.getAmount().abs());
            businessObjectService.save(invoicePaidApplied);
        }

        // if its open, but now with a zero open amount, then close it
        if (KualiDecimal.ZERO.equals(openAmount)) {
            invoice.setOpenInvoiceIndicator(false);
            invoice.setClosedDate(dateTimeService.getCurrentSqlDate());
            documentService.updateDocument(invoice);
        }
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocuments() {
        return getAllOpenCustomerInvoiceDocuments(true);
    }

    /**
     * @param includeWorkflowHeaders
     * @return
     */
    public Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocuments(boolean includeWorkflowHeaders) {
        // retrieve the set of documents without workflow headers
        Collection<CustomerInvoiceDocument> invoices = customerInvoiceDocumentDao.getAllOpen();

        // if we dont need workflow headers, then we're done
        if (!includeWorkflowHeaders) {
            return invoices;
        }

        // make a list of necessary workflow docs to retrieve
        List<String> documentHeaderIds = new ArrayList<>();
        for (CustomerInvoiceDocument invoice : invoices) {
            documentHeaderIds.add(invoice.getDocumentNumber());
        }

        // get all of our docs with full workflow headers
        List<CustomerInvoiceDocument> docs = new ArrayList<>();
        try {
            for (Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(CustomerInvoiceDocument.class,
                    documentHeaderIds)) {
                docs.add((CustomerInvoiceDocument) doc);
            }
        } catch (WorkflowException e) {
            throw new RuntimeException("Unable to retrieve Customer Invoice Documents", e);
        }

        return docs;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocumentsWithoutWorkflow() {
        return getAllOpenCustomerInvoiceDocuments(false);
    }

    @Override
    public Collection<CustomerInvoiceDocument> attachWorkflowHeadersToTheInvoices(
            Collection<CustomerInvoiceDocument> invoices) {
        List<CustomerInvoiceDocument> docs = new ArrayList<>();
        if (invoices == null || invoices.isEmpty()) {
            return docs;
        }

        // make a list of necessary workflow docs to retrieve
        List<String> documentHeaderIds = new ArrayList<>();
        for (CustomerInvoiceDocument invoice : invoices) {
            documentHeaderIds.add(invoice.getDocumentNumber());
        }

        // get all of our docs with full workflow headers
        try {
            for (Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(CustomerInvoiceDocument.class,
                    documentHeaderIds)) {
                docs.add((CustomerInvoiceDocument) doc);
            }
        } catch (WorkflowException e) {
            throw new RuntimeException("Unable to retrieve Customer Invoice Documents", e);
        }

        return docs;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerNumber(String customerNumber) {
        // customer number is not required to be populated, so we need to check that it's not null first
        if (StringUtils.isNotEmpty(customerNumber)) {
            // trim and force-caps the customer number
            customerNumber = customerNumber.trim().toUpperCase(Locale.US);
        }

        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerNumber(customerNumber));
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerNameByCustomerType(String customerName,
            String customerTypeCode) {
        // trim and force-caps the customer name
        customerName = StringUtils.replace(customerName, KFSConstants.WILDCARD_CHARACTER, KFSConstants.PERCENTAGE_SIGN);
        customerName = customerName.trim();
        if (!customerName.contains("%")) {
            customerName += "%";
        }

        // trim and force-caps
        customerTypeCode = customerTypeCode.trim().toUpperCase(Locale.US);

        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerNameByCustomerType(customerName,
                customerTypeCode));
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerName(String customerName) {
        // trim and force-caps the customer name
        customerName = StringUtils.replace(customerName, KFSConstants.WILDCARD_CHARACTER, KFSConstants.PERCENTAGE_SIGN);
        customerName = customerName.trim();
        if (!customerName.contains("%")) {
            customerName += "%";
        }

        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerName(customerName));
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerType(String customerTypeCode) {
        // trim and force-caps
        customerTypeCode = customerTypeCode.trim().toUpperCase(Locale.US);
        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerType(customerTypeCode));
    }

    @Override
    public Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocument(
            CustomerInvoiceDocument customerInvoiceDocument) {
        return getCustomerInvoiceDetailsForCustomerInvoiceDocument(customerInvoiceDocument.getDocumentNumber());
    }

    @Override
    public Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocument(
            String customerInvoiceDocumentNumber) {
        return customerInvoiceDetailService.getCustomerInvoiceDetailsForInvoice(customerInvoiceDocumentNumber);
    }

    @Override
    public Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocumentWithCaching(
            CustomerInvoiceDocument customerInvoiceDocument) {
        return customerInvoiceDetailService.getCustomerInvoiceDetailsForInvoiceWithCaching(customerInvoiceDocument
                .getDocumentNumber());
    }

    @Override
    public KualiDecimal getOpenAmountForCustomerInvoiceDocument(String customerInvoiceDocumentNumber) {
        if (null == customerInvoiceDocumentNumber) {
            return null;
        }
        return getOpenAmountForCustomerInvoiceDocument(getInvoiceByInvoiceDocumentNumber(customerInvoiceDocumentNumber));
    }

    @Override
    public KualiDecimal getOpenAmountForCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument) {
        KualiDecimal total = new KualiDecimal(0);
        if (customerInvoiceDocument.isOpenInvoiceIndicator()) {
            Collection<CustomerInvoiceDetail> customerInvoiceDetails = customerInvoiceDocument
                    .getCustomerInvoiceDetailsWithoutDiscounts();
            for (CustomerInvoiceDetail detail : customerInvoiceDetails) {
                // note that we're now dealing with conditionally applying discounts depending on whether the doc is
                // saved or approved one level down, in the CustomerInvoiceDetail.getAmountOpen()
                detail.setCustomerInvoiceDocument(customerInvoiceDocument);
                total = total.add(detail.getAmountOpen());
            }
        }
        return total;
    }

    @Override
    public KualiDecimal getOriginalTotalAmountForCustomerInvoiceDocument(
            CustomerInvoiceDocument customerInvoiceDocument) {
        LOG.info("\n\n\n\t\t invoice: " + customerInvoiceDocument.getDocumentNumber() +
                "\n\t\t 111111111 HEADER TOTAL AMOUNT (should be null): " +
                customerInvoiceDocument.getFinancialSystemDocumentHeader().getFinancialDocumentTotalAmount() + "\n\n");
        HashMap<String, String> criteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.DOCUMENT_NUMBER, customerInvoiceDocument.getDocumentHeader()
                .getDocumentTemplateNumber());
        FinancialSystemDocumentHeader financialSystemDocumentHeader = businessObjectService.findByPrimaryKey(
                FinancialSystemDocumentHeader.class, criteria);
        KualiDecimal originalTotalAmount = financialSystemDocumentHeader.getFinancialDocumentTotalAmount();

        LOG.info("\n\n\n\t\t invoice: " + customerInvoiceDocument.getDocumentNumber() +
                "\n\t\t 333333333333 HEADER TOTAL AMOUNT (should be set now): " +
                customerInvoiceDocument.getFinancialSystemDocumentHeader().getFinancialDocumentTotalAmount() + "\n\n");
        return originalTotalAmount;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByCustomerNumber(String customerNumber) {
        Collection<CustomerInvoiceDocument> invoices = new ArrayList<>();

        Map<String, String> fieldValues = new HashMap<>();
        fieldValues.put("customerNumber", customerNumber);

        Collection<AccountsReceivableDocumentHeader> documentHeaders = businessObjectService.findMatching(
                AccountsReceivableDocumentHeader.class, fieldValues);

        List<String> documentHeaderIds = new ArrayList<>();
        for (AccountsReceivableDocumentHeader header : documentHeaders) {
            documentHeaderIds.add(header.getDocumentHeader().getDocumentNumber());
        }

        if (0 < documentHeaderIds.size()) {
            try {
                for (Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                        CustomerInvoiceDocument.class, documentHeaderIds)) {
                    invoices.add((CustomerInvoiceDocument) doc);
                }
            } catch (WorkflowException e) {
                LOG.error("getCustomerInvoiceDocumentsByCustomerNumber " + customerNumber + " failed", e);
            }
        }
        return invoices;
    }

    @Override
    public Customer getCustomerByOrganizationInvoiceNumber(String organizationInvoiceNumber) {
        CustomerInvoiceDocument invoice = getInvoiceByOrganizationInvoiceNumber(organizationInvoiceNumber);
        return invoice.getAccountsReceivableDocumentHeader().getCustomer();
    }

    @Override
    public CustomerInvoiceDocument getInvoiceByOrganizationInvoiceNumber(String organizationInvoiceNumber) {
        return customerInvoiceDocumentDao.getInvoiceByOrganizationInvoiceNumber(organizationInvoiceNumber);
    }

    @Override
    public Customer getCustomerByInvoiceDocumentNumber(String invoiceDocumentNumber) {
        CustomerInvoiceDocument invoice = getInvoiceByInvoiceDocumentNumber(invoiceDocumentNumber);
        return invoice.getAccountsReceivableDocumentHeader().getCustomer();
    }

    @Override
    public CustomerInvoiceDocument getInvoiceByInvoiceDocumentNumber(String invoiceDocumentNumber) {
        return customerInvoiceDocumentDao.getInvoiceByInvoiceDocumentNumber(invoiceDocumentNumber);
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByInitiatorPrincipalName(
            String initiatorPrincipalName) {
        if (StringUtils.isBlank(initiatorPrincipalName)) {
            throw new IllegalArgumentException("The parameter [initiatorPrincipalName] passed in was null or blank.");
        }

        // IMPORTANT NOTES ABOUT THIS METHOD
        //
        // This method behaves differently than the other invoice printing methods. This is because there's no way
        // from within KFS to do a direct DB call to get all the invoices you want. This is because workflow holds the
        // document initiator, and you can't guarantee that in a given implementation that you have access to that
        // other db. It could be on another box in another network, and you only have web-services access to the Rice
        // box.
        //
        // Given that, we try to minimize the resource hit of this call as much as possible. First we retrieve all
        // invoices that haven't been printed (ie, dont have a print date) and that are marked for the USER print
        // queue. At any given time that should be a manageable number of documents.
        //
        // Then we walk through them, retrieve the full workflow-populated version of it, and only return the ones
        // that match the initiator.
        //
        // This isn't as performant a solution as the other getPrintableCustomerInvoiceBy...
        // methods, but its the best we can do in this release, and it should be manageable.

        //
        // attempt to retrieve the initiator person specified, and puke if not found
        Principal initiator = getIdentityService().getPrincipalByPrincipalName(initiatorPrincipalName);
        if (initiator == null) {
            throw new IllegalArgumentException("The parameter value for initiatorPrincipalName [" +
                    initiatorPrincipalName + "] passed in doesnt map to a person.");
        }

        // retrieve all the ready-to-print docs in the user-queue for all users
        List<String> printableUserQueueDocNumbers = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersFromUserQueue();

        // get all the documents that might be right, but this set includes documents generated by the wrong user
        List<CustomerInvoiceDocument> customerInvoiceDocumentsSuperSet = new ArrayList<>();
        if (printableUserQueueDocNumbers.size() > 0) {
            try {
                for (Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                        CustomerInvoiceDocument.class, printableUserQueueDocNumbers)) {
                    customerInvoiceDocumentsSuperSet.add((CustomerInvoiceDocument) doc);
                }
            } catch (WorkflowException e) {
                throw new RuntimeException("Unable to retrieve Customer Invoice Documents", e);
            }
        } else {
            customerInvoiceDocumentsSuperSet = new ArrayList<>();
        }

        // filter only the ones initiated by the correct user
        List<CustomerInvoiceDocument> customerInvoiceDocuments = new ArrayList<>();
        for (CustomerInvoiceDocument superSetDocument : customerInvoiceDocumentsSuperSet) {
            if (StringUtils.equalsIgnoreCase(superSetDocument.getDocumentHeader().getWorkflowDocument()
                    .getInitiatorPrincipalId(), initiator.getPrincipalId())) {
                customerInvoiceDocuments.add(superSetDocument);
            }
        }
        return customerInvoiceDocuments;
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByBillingChartAndOrg(
            String chartOfAccountsCode, String organizationCode) {
        List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersByBillingChartAndOrg(chartOfAccountsCode, organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    protected List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByDocumentNumbers(
            List<String> documentHeaderIds) {
        List<CustomerInvoiceDocument> customerInvoiceDocuments = new ArrayList<>(documentHeaderIds.size());
        if (documentHeaderIds != null && !documentHeaderIds.isEmpty()) {
            try {
                for (Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                        CustomerInvoiceDocument.class, documentHeaderIds)) {
                    customerInvoiceDocuments.add((CustomerInvoiceDocument) doc);
                }
            } catch (WorkflowException e) {
                throw new RuntimeException("Unable to retrieve Customer Invoice Documents", e);
            }
        }
        return customerInvoiceDocuments;
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsForBillingStatementByBillingChartAndOrg(
            String chartOfAccountsCode, String organizationCode) {
        List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersForBillingStatementByBillingChartAndOrg(
                        chartOfAccountsCode, organizationCode);

        List<CustomerInvoiceDocument> customerInvoiceDocuments = new ArrayList<>();
        if (documentHeaderIds != null && !documentHeaderIds.isEmpty()) {
            try {
                for (Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                        CustomerInvoiceDocument.class, documentHeaderIds)) {
                    customerInvoiceDocuments.add((CustomerInvoiceDocument) doc);
                }
            } catch (WorkflowException e) {
                throw new InfrastructureException("Unable to retrieve Customer Invoice Documents", e);
            }
        }
        return customerInvoiceDocuments;
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByProcessingChartAndOrg(
            String chartOfAccountsCode, String organizationCode) {
        List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersByProcessingChartAndOrg(chartOfAccountsCode,
                        organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public Collection<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByAccountNumber(String accountNumber) {
        List<String> documentHeaderIds = customerInvoiceDetailService
                .getCustomerInvoiceDocumentNumbersByAccountNumber(accountNumber);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByBillingChartAndOrg(String chartOfAccountsCode,
            String organizationCode) {
        List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getCustomerInvoiceDocumentNumbersByBillingChartAndOrg(chartOfAccountsCode, organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByProcessingChartAndOrg(
            String chartOfAccountsCode, String organizationCode) {
        List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getCustomerInvoiceDocumentNumbersByProcessingChartAndOrg(chartOfAccountsCode, organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public void setupDefaultValuesForNewCustomerInvoiceDocument(CustomerInvoiceDocument document) {
        setupBasicDefaultValuesForCustomerInvoiceDocument(document);

        // set up the default values for the AR DOC Header

        AccountsReceivableDocumentHeader accountsReceivableDocumentHeader =
                accountsReceivableDocumentHeaderService.getNewAccountsReceivableDocumentHeaderForCurrentUser();
        accountsReceivableDocumentHeader.setDocumentNumber(document.getDocumentNumber());
        document.setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);

        // set up the primary key for AR_INV_RCURRNC_DTL_T
        CustomerInvoiceRecurrenceDetails recurrenceDetails = new CustomerInvoiceRecurrenceDetails();
        recurrenceDetails.setInvoiceNumber(document.getDocumentNumber());
        document.setCustomerInvoiceRecurrenceDetails(recurrenceDetails);

        Map<String, String> criteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, document.getBillByChartOfAccountCode());
        criteria.put(KFSPropertyConstants.ORGANIZATION_CODE, document.getBilledByOrganizationCode());
        OrganizationOptions organizationOptions = businessObjectService.findByPrimaryKey(OrganizationOptions.class,
                criteria);

        if (ObjectUtils.isNotNull(organizationOptions)) {
            document.setPrintInvoiceIndicator(organizationOptions.getPrintInvoiceIndicator());
            document.setInvoiceTermsText(organizationOptions.getOrganizationPaymentTermsText());
        }
    }

    @Override
    public void loadCustomerAddressesForCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument) {
        // if address identifier is provided, try to refresh customer address data
        if (ObjectUtils.isNotNull(customerInvoiceDocument.getAccountsReceivableDocumentHeader())) {
            CustomerAddress customerShipToAddress = customerAddressService.getByPrimaryKey(
                    customerInvoiceDocument.getAccountsReceivableDocumentHeader().getCustomerNumber(),
                    customerInvoiceDocument.getCustomerShipToAddressIdentifier());
            CustomerAddress customerBillToAddress = customerAddressService.getByPrimaryKey(
                    customerInvoiceDocument.getAccountsReceivableDocumentHeader().getCustomerNumber(),
                    customerInvoiceDocument.getCustomerBillToAddressIdentifier());

            if (ObjectUtils.isNotNull(customerShipToAddress)) {
                customerInvoiceDocument.setCustomerShipToAddress(customerShipToAddress);
                customerInvoiceDocument.setCustomerShipToAddressOnInvoice(customerShipToAddress);
            }

            if (ObjectUtils.isNotNull(customerBillToAddress)) {
                customerInvoiceDocument.setCustomerBillToAddress(customerBillToAddress);
                customerInvoiceDocument.setCustomerBillToAddressOnInvoice(customerBillToAddress);
            }
        }
    }

    @Override
    public void setupDefaultValuesForCopiedCustomerInvoiceDocument(CustomerInvoiceDocument document) {
        setupBasicDefaultValuesForCustomerInvoiceDocument(document);

        // Save customer number since it will get overwritten when we retrieve the accounts receivable document
        // header from service
        String customerNumber = document.getAccountsReceivableDocumentHeader().getCustomerNumber();

        // Set up the default values for the AR DOC Header
        AccountsReceivableDocumentHeader accountsReceivableDocumentHeader =
                accountsReceivableDocumentHeaderService.getNewAccountsReceivableDocumentHeaderForCurrentUser();
        accountsReceivableDocumentHeader.setDocumentNumber(document.getDocumentNumber());
        accountsReceivableDocumentHeader.setCustomerNumber(customerNumber);
        document.setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);

        // set up the primary key for AR_INV_RCURRNC_DTL_T
        CustomerInvoiceRecurrenceDetails recurrenceDetails = new CustomerInvoiceRecurrenceDetails();
        recurrenceDetails.setInvoiceNumber(document.getDocumentNumber());
        document.setCustomerInvoiceRecurrenceDetails(recurrenceDetails);

        // make open invoice indicator to true
        document.setOpenInvoiceIndicator(true);
        document.setPrintDate(null);
        document.setBillingDate(dateTimeService.getCurrentSqlDateMidnight());
    }

    @Override
    public Collection<NonInvoicedDistribution> getNonInvoicedDistributionsForInvoice(String documentNumber) {
        return nonInvoicedDistributionService.getNonInvoicedDistributionsForInvoice(documentNumber);
    }

    @Override
    public KualiDecimal getNonInvoicedTotalForInvoice(CustomerInvoiceDocument invoice) {
        Collection<NonInvoicedDistribution> payments = this.nonInvoicedDistributionService
                .getNonInvoicedDistributionsForInvoice(invoice);
        KualiDecimal total = new KualiDecimal(0);
        for (NonInvoicedDistribution payment : payments) {
            total = total.add(payment.getFinancialDocumentLineAmount());
        }
        return total;
    }

    @Override
    public KualiDecimal getNonInvoicedTotalForInvoice(String documentNumber) {
        return getNonInvoicedTotalForInvoice(getInvoiceByInvoiceDocumentNumber(documentNumber));
    }

    @Override
    public KualiDecimal getPaidAppliedTotalForInvoice(CustomerInvoiceDocument invoice) {
        Collection<InvoicePaidApplied> payments = invoicePaidAppliedService.getActiveInvoicePaidAppliedsForInvoice(invoice);
        KualiDecimal total = new KualiDecimal(0);
        for (InvoicePaidApplied payment : payments) {
            total = total.add(payment.getInvoiceItemAppliedAmount());
        }
        return total;
    }

    @Override
    public KualiDecimal getPaidAppliedTotalForInvoice(String documentNumber) {
        return getPaidAppliedTotalForInvoice(getInvoiceByInvoiceDocumentNumber(documentNumber));
    }

    /**
     * @param document
     */
    protected void setupBasicDefaultValuesForCustomerInvoiceDocument(CustomerInvoiceDocument document) {
        ChartOrgHolder currentUser = financialSystemUserService.getPrimaryOrganization(
                GlobalVariables.getUserSession().getPerson(), ArConstants.AR_NAMESPACE_CODE);
        if (currentUser != null) {
            document.setBillByChartOfAccountCode(currentUser.getChartOfAccountsCode());
            document.setBilledByOrganizationCode(currentUser.getOrganizationCode());
        }
        document.setInvoiceDueDate(getDefaultInvoiceDueDate());
        document.setOpenInvoiceIndicator(true);
    }

    /**
     * This method sets due date equal to todays date +30 days by default
     */
    protected Date getDefaultInvoiceDueDate() {
        Calendar cal = dateTimeService.getCurrentCalendar();
        cal.add(Calendar.DATE, 30);
        Date sqlDueDate = null;
        try {
            sqlDueDate = dateTimeService.convertToSqlDate(new Timestamp(cal.getTime().getTime()));
        } catch (ParseException e) {
            // TODO: throw an error here, but don't die
        }
        return sqlDueDate;
    }

    @Override
    public void closeCustomerInvoiceDocument(CustomerInvoiceDocument customerInvoiceDocument) {
        customerInvoiceDocument.setOpenInvoiceIndicator(false);
        customerInvoiceDocument.setClosedDate(dateTimeService.getCurrentSqlDate());
        businessObjectService.save(customerInvoiceDocument);
    }

    @Override
    public void updateReportedDate(String docNumber) {
        HashMap<String, String> criteria = new HashMap<>();
        criteria.put("documentNumber", docNumber);
        CustomerInvoiceDocument customerInvoiceDocument = businessObjectService.findByPrimaryKey(
                CustomerInvoiceDocument.class, criteria);
        Date reportedDate = dateTimeService.getCurrentSqlDate();
        if (ObjectUtils.isNotNull(customerInvoiceDocument)) {
            customerInvoiceDocument.setReportedDate(reportedDate);
            businessObjectService.save(customerInvoiceDocument);
        }
    }

    @Override
    public void updateReportedInvoiceInfo(CustomerStatementResultHolder data) {
        HashMap<String, String> criteria = new HashMap<>();
        criteria.put("customerNumber", data.getCustomerNumber());
        CustomerBillingStatement customerBillingStatement = businessObjectService.findByPrimaryKey(
                CustomerBillingStatement.class, criteria);
        if (ObjectUtils.isNotNull(customerBillingStatement)) {
            customerBillingStatement.setPreviouslyBilledAmount(data.getCurrentBilledAmount());
            customerBillingStatement.setReportedDate(dateTimeService.getCurrentSqlDate());
        } else {
            customerBillingStatement = new CustomerBillingStatement();
            customerBillingStatement.setCustomerNumber(data.getCustomerNumber());
            customerBillingStatement.setPreviouslyBilledAmount(data.getCurrentBilledAmount());
            customerBillingStatement.setReportedDate(dateTimeService.getCurrentSqlDate());
        }
        businessObjectService.save(customerBillingStatement);
    }

    @Override
    public boolean checkIfInvoiceNumberIsFinal(String invDocumentNumber) {
        boolean isSuccess = true;
        if (StringUtils.isBlank(invDocumentNumber)) {
            isSuccess = false;
        } else {
            CustomerInvoiceDocument customerInvoiceDocument = getInvoiceByInvoiceDocumentNumber(invDocumentNumber);

            if (ObjectUtils.isNull(customerInvoiceDocument)) {
                isSuccess = false;
            } else {
                Document doc = null;
                try {
                    doc = documentService.getByDocumentHeaderId(invDocumentNumber);
                } catch (WorkflowException e) {
                    isSuccess = false;
                }
                if (ObjectUtils.isNull(doc) || ObjectUtils.isNull(doc.getDocumentHeader())
                        || doc.getDocumentHeader().getWorkflowDocument() == null
                        || !(doc.getDocumentHeader().getWorkflowDocument().isApproved()
                        || doc.getDocumentHeader().getWorkflowDocument().isProcessed())) {
                    isSuccess = false;
                }
            }
        }
        return isSuccess;
    }

    /**
     * get the date before the given amount of days
     */
    protected Date getPastDate(Integer amount) {
        int pastDateAmount = -1 * amount;

        java.util.Date today = this.getDateTimeService().getCurrentDate();
        java.util.Date pastDate = DateUtils.addDays(today, pastDateAmount);

        return KfsDateUtils.convertToSqlDate(pastDate);
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllAgingInvoiceDocumentsByCustomerTypes(List<String> customerTypes,
            Integer invoiceAge, Date invoiceDueDateFrom) {
        Date pastDate = this.getPastDate(invoiceAge - 1);
        Date invoiceDueDateTo = KfsDateUtils.convertToSqlDate(DateUtils.addDays(pastDate, 1));
        LOG.info("invoiceDueDateTo" + invoiceDueDateTo);

        return customerInvoiceDocumentDao.getAllAgingInvoiceDocumentsByCustomerTypes(customerTypes,
                invoiceDueDateFrom, invoiceDueDateTo);
    }

    @Override
    public void addCloseNote(CustomerInvoiceDocument documentToClose, WorkflowDocument closingDocument) {
        if (!documentToClose.isOpenInvoiceIndicator()) {
            // If it already is closed, no need to add a note
            return;
        }

        String principalName = "Unknown";
        List<ActionTaken> actionsTaken = closingDocument.getActionsTaken();
        if (ObjectUtils.isNotNull(actionsTaken)) {
            for (ActionTaken action : actionsTaken) {
                // we're looking for the person who completed the closing document, so we want the COMPLETE action
                if (isActionClose(action.getActionTaken())) {
                    principalName = getPersonService().getPerson(action.getPrincipalId()).getName();
                    break;
                }
            }
        }

        final String noteTextPattern = getConfigurationService().getPropertyValueAsString(
                ArKeyConstants.INVOICE_CLOSE_NOTE_TEXT);
        Object[] arguments = {principalName, closingDocument.getDocumentTypeName(), closingDocument.getDocumentId()};
        String noteText = MessageFormat.format(noteTextPattern, arguments);

        Note note = getDocumentService().createNoteFromDocument(documentToClose, noteText);
        note.setAuthorUniversalIdentifier(getIdentityService().getPrincipalByPrincipalName(
                KFSConstants.SYSTEM_USER).getPrincipalId());
        documentToClose.addNote(noteService.save(note));
    }

    /**
     * Looks up all invoice paid applieds applied to this invoice owned by payment applications or customer credit
     * memos, and sums the invoice item applied amount
     */
    @Override
    public KualiDecimal calculateAppliedPaymentAmount(CustomerInvoiceDocument invoice) {
        Map<String, Object> criteria = new HashMap<>();
        KualiDecimal totalPayments = KualiDecimal.ZERO;
        criteria.put(ArPropertyConstants.CustomerInvoiceDocumentFields.FINANCIAL_DOCUMENT_REF_INVOICE_NUMBER,
                invoice.getDocumentNumber());
        List<String> allowedOwningDocumentTypes = new ArrayList<>();
        allowedOwningDocumentTypes.add(ArConstants.CUSTOMER_CREDIT_MEMO_DOCUMENT_TYPE_CODE);
        criteria.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME,
                allowedOwningDocumentTypes);

        Collection<InvoicePaidApplied> creditMemoInvoicePaidApplieds = businessObjectService.findMatching(
                InvoicePaidApplied.class, criteria);
        List<InvoicePaidApplied> paymentAdjustableInvoicePaidApplieds = invoicePaidAppliedService
                .getActiveInvoicePaidAppliedsForInvoice(invoice);

        return Stream.of(creditMemoInvoicePaidApplieds, paymentAdjustableInvoicePaidApplieds)
                .flatMap(Collection::stream)
                .map(InvoicePaidApplied::getInvoiceItemAppliedAmount)
                .reduce(KualiDecimal.ZERO, KualiDecimal::add);
    }

    private boolean isActionClose(String actionTaken) {
        return actionTaken.equals(ActionType.COMPLETE.getCode())
                || actionTaken.equals(ActionType.SU_BLANKET_APPROVE.getCode())
                || actionTaken.equals(ActionType.BLANKET_APPROVE.getCode())
                || actionTaken.equals(ActionType.SU_COMPLETE.getCode());
    }

    public CustomerInvoiceDocumentDao getCustomerInvoiceDocumentDao() {
        return customerInvoiceDocumentDao;
    }

    public void setCustomerInvoiceDocumentDao(CustomerInvoiceDocumentDao customerInvoiceDocumentDao) {
        this.customerInvoiceDocumentDao = customerInvoiceDocumentDao;
    }

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public AccountsReceivableDocumentHeaderService getAccountsReceivableDocumentHeaderService() {
        return accountsReceivableDocumentHeaderService;
    }

    public void setAccountsReceivableDocumentHeaderService(
            AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService) {
        this.accountsReceivableDocumentHeaderService = accountsReceivableDocumentHeaderService;
    }

    public CustomerAddressService getCustomerAddressService() {
        return customerAddressService;
    }

    public void setCustomerAddressService(CustomerAddressService customerAddressService) {
        this.customerAddressService = customerAddressService;
    }

    public void setDocumentDao(DocumentDao documentDao) {
        this.documentDao = documentDao;
    }

    public void setInvoicePaidAppliedService(InvoicePaidAppliedService invoicePaidAppliedService) {
        this.invoicePaidAppliedService = invoicePaidAppliedService;
    }

    public void setNonInvoicedDistributionService(NonInvoicedDistributionService nonInvoicedDistributionService) {
        this.nonInvoicedDistributionService = nonInvoicedDistributionService;
    }

    public void setCustomerInvoiceDetailService(CustomerInvoiceDetailService customerInvoiceDetailService) {
        this.customerInvoiceDetailService = customerInvoiceDetailService;
    }

    public void setUniversityDateService(UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public UniversityDateService getUniversityDateService() {
        return universityDateService;
    }

    public void setNoteService(NoteService noteService) {
        this.noteService = noteService;
    }

    public void setPersonService(PersonService personService) {
        this.personService = personService;
    }

    public PersonService getPersonService() {
        return personService;
    }

    public IdentityService getIdentityService() {
        return identityService;
    }

    public void setIdentityService(IdentityService identityService) {
        this.identityService = identityService;
    }

    public ConfigurationService getConfigurationService() {
        return configurationService;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public FinancialSystemUserService getFinancialSystemUserService() {
        return financialSystemUserService;
    }

    public void setFinancialSystemUserService(FinancialSystemUserService financialSystemUserService) {
        this.financialSystemUserService = financialSystemUserService;
    }
}
