/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.api.parameter.EvaluationOperator;
import org.kuali.kfs.coreservice.impl.parameter.Parameter;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerAddressEmail;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.krad.UserSession;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.batch.service.LockboxService;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.Lockbox;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceWriteoffDocument;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceWriteoffDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.batch.AbstractStep;
import org.kuali.kfs.sys.batch.Job;
import org.kuali.kfs.sys.batch.TestingStep;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

public class CustomerInvoiceDocumentBatchStep extends AbstractStep implements TestingStep {

    private static final Logger LOG = LogManager.getLogger();

    protected static final long MAX_SEQ_NBR_OFFSET = 1000;

    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected BusinessObjectService businessObjectService;
    protected DocumentService documentService;
    protected DateTimeService dateTimeService;
    protected LockboxService lockboxService;
    protected CustomerAddressService customerAddressService;
    protected CustomerInvoiceWriteoffDocumentService writeoffService;
    protected Collection<String> createdInvoices = new ArrayList<>();

    // parameter constants and logging
    protected static final int NUMBER_OF_INVOICES_TO_CREATE = 5;
    protected static final String RUN_INDICATOR_PARAMETER_NAMESPACE_CODE = ArConstants.AR_NAMESPACE_CODE;
    protected static final String RUN_INDICATOR_PARAMETER_APPLICATION_NAMESPACE_CODE = KFSConstants.APPLICATION_NAMESPACE_CODE;
    protected static final String RUN_INDICATOR_PARAMETER_NAMESPACE_STEP = CustomerInvoiceDocumentBatchStep.class.getSimpleName();
    protected static final String RUN_INDICATOR_PARAMETER_VALUE = "N";
    protected static final String RUN_INDICATOR_PARAMETER_DESCRIPTION = "Tells the job framework whether to run this job or not; because the CustomerInvoiceDocumentBatchStep needs to only be run once after database initialization.";
    protected static final String RUN_INDICATOR_PARAMETER_TYPE = "CONFG";
    protected static final String INITIATOR_PRINCIPAL_NAME = "khuntley";

    protected final int currentYear = LocalDate.now().getYear();

    @Override
    public boolean execute(final String jobName, final Date jobRunDate) throws InterruptedException {
        final Parameter runIndicatorParameter = getParameterService().getParameter(
                RUN_INDICATOR_PARAMETER_NAMESPACE_CODE, RUN_INDICATOR_PARAMETER_NAMESPACE_STEP, Job.STEP_RUN_PARM_NM);
        if (runIndicatorParameter == null || StringUtils.equals("Y", runIndicatorParameter.getValue())) {
            GlobalVariables.clear();
            GlobalVariables.setUserSession(new UserSession(INITIATOR_PRINCIPAL_NAME));

            Date billingDate = dateTimeService.getCurrentDate();
            final List<String> customernames;

            if (jobName.length() <= 8 && jobName.length() >= 4) {
                customernames = List.of(jobName);
            } else {
                customernames = Arrays.asList("ABB2", "3MC17500", "ACE21725", "ANT7297", "CAR23612", "CON19567",
                        "DEL14448", "EAT17609", "GAP17272");
            }

            // create non-random data
            if (customernames.size() > 1) {
                for (int i = 0; i < NUMBER_OF_INVOICES_TO_CREATE; i++) {
                    billingDate = DateUtils.addDays(billingDate, -30);

                    // $10 entries
                    createCustomerInvoiceDocumentForFunctionalTesting("HIL22195", billingDate, 1,
                            new KualiDecimal(10), new BigDecimal(1), "2336320", "BL", "BUSCF");
                    // $20 entries
                    createCustomerInvoiceDocumentForFunctionalTesting("IBM2655", billingDate, 2,
                            new KualiDecimal(10), new BigDecimal(1), "2336320", "BL", "IBCE");
                    // $30 entries
                    createCustomerInvoiceDocumentForFunctionalTesting("JAS19572", billingDate, 3,
                            new KualiDecimal(10), new BigDecimal(1), "2336320", "BL", "WRB");

                    Thread.sleep(500);
                }
            }

            // easy dynamic data creation
            if (customernames.size() == 1) {
                billingDate = jobRunDate;
                // $10 entries
                createCustomerInvoiceDocumentForFunctionalTesting(customernames.get(0), billingDate, 1,
                        new KualiDecimal(10), new BigDecimal(1), "1111111", "BA", "MATT");
                Thread.sleep(500);
            }

            // create lockboxes for the non-random invoices
            Long seqNbr = findAvailableLockboxBaseSeqNbr();
            int scenarioNbr = 1;
            for (final String createdInvoice : createdInvoices) {
                createLockboxesForFunctionalTesting(createdInvoice, seqNbr, scenarioNbr);
                Thread.sleep(500);
                seqNbr++;
                if (scenarioNbr <= 6) {
                    scenarioNbr++;
                } else {
                    scenarioNbr = 1;
                }
            }

            // save runParameter as "N" so that the job won't run until DB has been cleared
            setInitiatedParameter();
        }
        return true;
    }

    private long findAvailableLockboxBaseSeqNbr() {
        return lockboxService.getMaxLockboxSequenceNumber() + MAX_SEQ_NBR_OFFSET;
    }

    private boolean dupLockboxRecordExists(final Long seqNbr) {
        final Map<String, Long> pks = new HashMap<>();
        pks.put("invoiceSequenceNumber", seqNbr);
        final Lockbox dupLockBox = businessObjectService.findByPrimaryKey(Lockbox.class, pks);
        return dupLockBox != null;
    }

    /**
     * This method sets a parameter that tells the step that it has already run and it does not need to run again.
     */
    private void setInitiatedParameter() {
        // first see if we can find an existing Parameter object with this key
        final Parameter runIndicatorParameter = getParameterService().getParameter(RUN_INDICATOR_PARAMETER_NAMESPACE_CODE,
                RUN_INDICATOR_PARAMETER_NAMESPACE_STEP, Job.STEP_RUN_PARM_NM);
        if (runIndicatorParameter == null) {
            final Parameter newParameter = new Parameter();
            newParameter.setNamespaceCode(RUN_INDICATOR_PARAMETER_NAMESPACE_CODE);
            newParameter.setComponentCode(RUN_INDICATOR_PARAMETER_NAMESPACE_STEP);
            newParameter.setName(Job.STEP_RUN_PARM_NM);
            newParameter.setParameterTypeCode(RUN_INDICATOR_PARAMETER_TYPE);
            newParameter.setEvaluationOperatorCode(EvaluationOperator.ALLOW.getCode());
            newParameter.setDescription(RUN_INDICATOR_PARAMETER_DESCRIPTION);
            newParameter.setValue(RUN_INDICATOR_PARAMETER_VALUE);
            getParameterService().createParameter(newParameter);
        } else {
            runIndicatorParameter.setValue(RUN_INDICATOR_PARAMETER_VALUE);
            getParameterService().updateParameter(runIndicatorParameter);
        }
    }

    private Lockbox populateLockbox(final String invoiceNumber, final Long seqNbr) {
        final CustomerInvoiceDocument customerInvoiceDocument = customerInvoiceDocumentService
                .getInvoiceByInvoiceDocumentNumber(invoiceNumber);

        final Lockbox newLockbox = new Lockbox();
        newLockbox.setFinancialDocumentReferenceInvoiceNumber(invoiceNumber);
        newLockbox.setCustomerNumber(customerInvoiceDocument.getCustomer().getCustomerNumber());
        newLockbox.setInvoiceTotalAmount(customerInvoiceDocument.getTotalDollarAmount());
        newLockbox.setInvoicePaidOrAppliedAmount(customerInvoiceDocument.getOpenAmount());
        newLockbox.setBillingDate(customerInvoiceDocument.getBillingDate());
        newLockbox.setCustomerPaymentMediumCode("CK");
        newLockbox.setBankCode("1003");
        newLockbox.setBatchSequenceNumber(8004);
        newLockbox.setInvoiceSequenceNumber(seqNbr);
        newLockbox.setLockboxNumber("66249");

        return newLockbox;
    }

    private void createLockboxesForFunctionalTesting(final String invoiceNumber, final Long seqNbr, final int testtype) {
        final Lockbox newLockbox = populateLockbox(invoiceNumber, seqNbr);

        // 1) Payment matches customer (CUST_NBR), invoice number (FDOC_REF_INV_NBR), and amount (AR_INV_PD_APLD_AMT).
        // These should auto-approve, the remaining scenarios should not.
        // dont need to do anything, its auto-set to pay the OpenAmount on the invoice

        // 2) Payment matches customer and invoice, but the invoice has no outstanding balance (due to a previous
        // payment, a credit memo, or a write-off)
        if (testtype == 2) {
            newLockbox.setInvoicePaidOrAppliedAmount(newLockbox.getInvoiceTotalAmount());
            writeoffInvoice(invoiceNumber);
        }

        // 3) Payment matches customer and invoice, but the amount of the payment exceeds the outstanding balance on
        // the invoice.
        if (testtype == 3) {
            newLockbox.setInvoicePaidOrAppliedAmount(newLockbox.getInvoicePaidOrAppliedAmount()
                    .add(new KualiDecimal("100.00")));
        }

        // 4) The payment matches customer and invoice, but the amount is short-paid (less than the invoice
        // outstanding balance)
        if (testtype == 4) {
            newLockbox.setInvoicePaidOrAppliedAmount(newLockbox.getInvoicePaidOrAppliedAmount()
                    .subtract(new KualiDecimal("1.00")));
        }

        // 5) The payment matches a customer number, but the invoice number is missing
        if (testtype == 5) {
            newLockbox.setFinancialDocumentReferenceInvoiceNumber(null);
        }

        // 6) The payment matches a customer number, but the invoice number is invalid
        if (testtype == 6) {
            newLockbox.setFinancialDocumentReferenceInvoiceNumber("999999");
        }

        // 7) The payment matches nothing (not even the customer number)
        if (testtype == 7) {
            newLockbox.setFinancialDocumentReferenceInvoiceNumber("999999");
            newLockbox.setCustomerNumber("KEY17536");
        }

        LOG.info("Creating customer LOCKBOX [{}] for invoice {}", seqNbr, invoiceNumber);
        if (dupLockboxRecordExists(seqNbr)) {
            throw new RuntimeException("Trying to enter duplicate Lockbox.invoiceSequenceNumber, which will fail, " +
                    "and should never happen.");
        }
        businessObjectService.save(newLockbox);
    }

    public void writeoffInvoice(final String invoiceNumberToWriteOff) {
        //  have the service create us a new writeoff doc
        final String writeoffDocNumber = writeoffService.createCustomerInvoiceWriteoffDocument(invoiceNumberToWriteOff,
                "Created by CustomerInvoiceDocumentBatch process.");

        //  load the newly created writeoff doc from the db
        final CustomerInvoiceWriteoffDocument writeoff = (CustomerInvoiceWriteoffDocument) documentService
                .getByDocumentHeaderId(writeoffDocNumber);

        boolean wentToFinal;
        try {
            wentToFinal = waitForStatusChange(60, writeoff.getDocumentHeader().getWorkflowDocument(),
                    new String[]{"F", "P"});
        } catch (final Exception e) {
            throw new RuntimeException("An Exception was thrown when trying to monitor writeoff doc #" +
                    writeoffDocNumber + " going to FINAL.", e);
        }

        //  if the doc didnt go to final, then blanket approve the doc, bypassing all rules
        if (!wentToFinal) {
            try {
                writeoff.getDocumentHeader().getWorkflowDocument().blanketApprove("BlanketApproved by " +
                        "CustomerInvoiceDocumentBatch process.");
            } catch (final Exception e) {
                throw new RuntimeException("A WorkflowException was thrown when trying to blanketApprove Invoice " +
                        "Writeoff doc #" + writeoffDocNumber + ".", e);
            }

            //  wait for it to go to final
            try {
                wentToFinal = waitForStatusChange(60, writeoff.getDocumentHeader().getWorkflowDocument(),
                        new String[]{"F", "P"});
            } catch (final Exception e) {
                throw new RuntimeException("An Exception was thrown when trying to monitor writeoff doc #" +
                        writeoffDocNumber + " going to FINAL.", e);
            }
        }

        if (!wentToFinal) {
            throw new RuntimeException("InvoiceWriteoff document #" + writeoffDocNumber + " failed to route to FINAL.");
        }
    }

    public boolean waitForStatusChange(final int numSeconds, final WorkflowDocument document, final String[] desiredStatus) throws
            Exception {
        final DocWorkflowStatusMonitor monitor = new DocWorkflowStatusMonitor(documentService,
                document.getDocumentId(), desiredStatus);
        return waitUntilChange(monitor, numSeconds, 5);
    }

    /**
     * Iterates, with pauseSeconds seconds between iterations, until either the given ChangeMonitor's valueChanged
     * method returns true, or at least maxWaitSeconds seconds have passed.
     *
     * @param monitor        ChangeMonitor instance which watches for whatever change your test is waiting for
     * @param maxWaitSeconds
     * @param pauseSeconds
     * @return true if the the ChangeMonitor's valueChanged method returned true before time ran out
     */
    public boolean waitUntilChange(final DocWorkflowStatusMonitor monitor, final int maxWaitSeconds, final int pauseSeconds) throws
            Exception {
        final long maxWaitMs = maxWaitSeconds * 1000L;
        final long pauseMs = pauseSeconds * 1000L;

        boolean valueChanged;
        boolean interrupted = false;
        final long startTimeMs = System.currentTimeMillis();
        final long endTimeMs = startTimeMs + maxWaitMs;

        // the first time through, sleep a fraction of the specified time
        Thread.sleep(pauseMs / 10);
        valueChanged = monitor.valueChanged();
        while (!interrupted && !valueChanged && System.currentTimeMillis() < endTimeMs) {
            try {
                Thread.sleep(pauseMs);
            } catch (final InterruptedException e) {
                interrupted = true;
            }
            valueChanged = monitor.valueChanged();
        }
        return valueChanged;
    }

    public void createCustomerInvoiceDocumentForFunctionalTesting(
            final String customerNumber, final Date billingDate,
            final int numInvoiceDetails, final KualiDecimal nonRandomQuantity, final BigDecimal nonRandomuUnitPrice,
            final String accountNumber, final String chartCode, final String invoiceItemCode) {
        final SimpleDateFormat sdf = new SimpleDateFormat("MM/dd/yyyy", Locale.US);

        final CustomerInvoiceDocument customerInvoiceDocument;
        customerInvoiceDocument = (CustomerInvoiceDocument) documentService.getNewDocument(CustomerInvoiceDocument.class);
        LOG.info("Created customer invoice document {}", customerInvoiceDocument::getDocumentNumber);

        customerInvoiceDocumentService.setupDefaultValuesForNewCustomerInvoiceDocument(customerInvoiceDocument);
        customerInvoiceDocument.getDocumentHeader().setDocumentDescription("TEST CUSTOMER INVOICE DOCUMENT");
        customerInvoiceDocument.getAccountsReceivableDocumentHeader().setCustomerNumber(customerNumber);
        customerInvoiceDocument.setBillingDate(new java.sql.Date(billingDate.getTime()));

        final CustomerAddress customerBillToAddress = customerAddressService.getPrimaryAddress(customerNumber);

        customerInvoiceDocument.setCustomerBillToAddress(customerBillToAddress);
        customerInvoiceDocument.setCustomerBillToAddressIdentifier(1);
        customerInvoiceDocument.setBillingAddressTypeCode("P");
        customerInvoiceDocument.setBillingAddressName(customerBillToAddress.getCustomerAddressName());
        customerInvoiceDocument.setBillingLine1StreetAddress(customerBillToAddress.getCustomerLine1StreetAddress());
        customerInvoiceDocument.setBillingLine2StreetAddress(customerBillToAddress.getCustomerLine2StreetAddress());
        customerInvoiceDocument.setBillingCityName(customerBillToAddress.getCustomerCityName());
        customerInvoiceDocument.setBillingStateCode(customerBillToAddress.getCustomerStateCode());
        customerInvoiceDocument.setBillingZipCode(customerBillToAddress.getCustomerZipCode());
        customerInvoiceDocument.setBillingCountryCode(customerBillToAddress.getCustomerCountryCode());
        customerInvoiceDocument.setBillingAddressInternationalProvinceName(
                customerBillToAddress.getCustomerAddressInternationalProvinceName());
        customerInvoiceDocument.setBillingInternationalMailCode(customerBillToAddress.getCustomerInternationalMailCode());
        final List<AccountsReceivableCustomerAddressEmail> customerBillToAddressEmails =
                customerBillToAddress.getCustomerAddressEmails();
        if (customerBillToAddressEmails.size() > 0) {
            customerInvoiceDocument.setBillingEmailAddress(customerBillToAddressEmails.get(0).getCustomerEmailAddress());
        }

        if (ObjectUtils.isNotNull(nonRandomQuantity) && ObjectUtils.isNotNull(nonRandomuUnitPrice)
                && numInvoiceDetails >= 1) {
            for (int i = 0; i < numInvoiceDetails; i++) {
                customerInvoiceDocument.addSourceAccountingLine(
                        createCustomerInvoiceDetailForFunctionalTesting(customerInvoiceDocument, nonRandomQuantity,
                                nonRandomuUnitPrice, accountNumber, chartCode, invoiceItemCode));
            }
        } else {
            // add up to 9
            int randomNumInvoiceDetails = (int) (Math.random() * 9);
            if (randomNumInvoiceDetails == 0) {
                // add at least one
                randomNumInvoiceDetails = 1;
            }
            for (int i = 0; i < randomNumInvoiceDetails; i++) {
                customerInvoiceDocument.addSourceAccountingLine(
                        createCustomerInvoiceDetailForFunctionalTesting(customerInvoiceDocument, null, null,
                                accountNumber, chartCode, invoiceItemCode));
            }
        }
        documentService.blanketApproveDocument(customerInvoiceDocument, null, null);
        createdInvoices.add(customerInvoiceDocument.getDocumentNumber());
        LOG.info(
                "Submitted customer invoice document {} for {} - {}\n\n",
                customerInvoiceDocument::getDocumentNumber,
                () -> customerNumber,
                () -> sdf.format(billingDate)
        );
    }

    public CustomerInvoiceDetail createCustomerInvoiceDetailForFunctionalTesting(
            final CustomerInvoiceDocument customerInvoiceDocument, final KualiDecimal nonRandomQuantity,
            final BigDecimal nonRandomUnitPrice, final String accountNumber, final String chartCode, final String invoiceItemCode) {
        final KualiDecimal quantity;
        final BigDecimal unitprice;

        if (ObjectUtils.isNull(nonRandomQuantity)) {
            // random number 0 to 100 total items
            quantity = new KualiDecimal(100 * Math.random());
            // TODO FIXME  <-- InvoiceItemQuantities of more than 2 decimal places cause rule errors; BigDecimal
            //  values such as 5.3333333333 should be valid InvoiceItemQuantities
        } else {
            quantity = nonRandomQuantity;
        }
        if (ObjectUtils.isNull(nonRandomUnitPrice)) {
            // 0.00 to 100.00 dollars per item
            unitprice = new BigDecimal(1);
        } else {
            unitprice = nonRandomUnitPrice;
        }

        // setAmount has to be set explicitly below; so we calculate it here
        final KualiDecimal amount = quantity.multiply(new KualiDecimal(unitprice));

        final CustomerInvoiceDetail customerInvoiceDetail = new CustomerInvoiceDetail();
        customerInvoiceDetail.setDocumentNumber(customerInvoiceDocument.getDocumentNumber());
        customerInvoiceDetail.setChartOfAccountsCode(chartCode);
        customerInvoiceDetail.setAccountNumber(accountNumber);
        customerInvoiceDetail.setFinancialObjectCode("1800");
        customerInvoiceDetail.setAccountsReceivableObjectCode("8118");
        customerInvoiceDetail.setInvoiceItemCode(invoiceItemCode);
        customerInvoiceDetail.setInvoiceItemServiceDate(dateTimeService.getCurrentSqlDate());
        customerInvoiceDetail.setInvoiceItemUnitPrice(unitprice);
        customerInvoiceDetail.setInvoiceItemQuantity(quantity.bigDecimalValue());
        customerInvoiceDetail.setInvoiceItemTaxAmount(new KualiDecimal(100));
        customerInvoiceDetail.setTaxableIndicator(true);
        customerInvoiceDetail.setAmount(amount);
        customerInvoiceDetail.setPostingYear(currentYear);

        return customerInvoiceDetail;
    }

    @Override
    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    @Override
    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public CustomerInvoiceDocumentService getCustomerInvoiceDocumentService() {
        return customerInvoiceDocumentService;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public LockboxService getLockboxService() {
        return lockboxService;
    }

    public void setLockboxService(final LockboxService lockboxService) {
        this.lockboxService = lockboxService;
    }

    public CustomerInvoiceWriteoffDocumentService getWriteoffService() {
        return writeoffService;
    }

    public void setWriteoffService(final CustomerInvoiceWriteoffDocumentService writeoffService) {
        this.writeoffService = writeoffService;
    }

    public CustomerAddressService getCustomerAddressService() {
        return customerAddressService;
    }

    public void setCustomerAddressService(final CustomerAddressService customerAddressService) {
        this.customerAddressService = customerAddressService;
    }

    private class DocWorkflowStatusMonitor {
        final DocumentService documentService;
        private final String docHeaderId;
        private final String[] desiredWorkflowStates;

        DocWorkflowStatusMonitor(final DocumentService documentService, final String docHeaderId, final String[] desiredWorkflowStates) {
            this.documentService = documentService;
            this.docHeaderId = docHeaderId;
            this.desiredWorkflowStates = desiredWorkflowStates;
        }

        public boolean valueChanged() throws Exception {
            final Document d = documentService.getByDocumentHeaderId(docHeaderId);

            final String currentStatus = d.getDocumentHeader().getWorkflowDocument().getStatus().getCode();

            for (final String desiredWorkflowState : desiredWorkflowStates) {
                if (StringUtils.equals(desiredWorkflowState, currentStatus)) {
                    return true;
                }
            }
            return false;
        }
    }
}
