/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.batch.service.CustomerLoadService;
import org.kuali.kfs.module.ar.batch.vo.CustomersVO;
import org.kuali.kfs.sys.batch.XmlBatchInputFileTypeBase;

import java.io.File;

public class CustomerLoadInputFileType extends XmlBatchInputFileTypeBase<CustomersVO> {
    private static final Logger LOG = LogManager.getLogger();

    private static final String FILE_NAME_PREFIX = "customer_load";
    private static final String FILE_NAME_DELIM = "_";

    private CustomerLoadService customerLoadService;

    @Override
    public String getFileName(
            final String principalName, final Object parsedFileContents, final String fileUserIdentifier
    ) {
        return customerLoadService.getFileName(principalName, fileUserIdentifier, FILE_NAME_PREFIX, FILE_NAME_DELIM);
    }

    @Override
    public String getFileTypeIdentifier() {
        return ArConstants.CustomerLoad.CUSTOMER_LOAD_FILE_TYPE_IDENTIFIER;
    }

    @Override
    public boolean validate(final Object parsedFileContents) {
        //  attempt to cast the parsedFileContents into the expected type
        try {
            final CustomersVO customersVO = (CustomersVO) parsedFileContents;
            return customerLoadService.validate(customersVO.getCustomerVOs());

        } catch (final Exception e) {
            LOG.error(
                    "Could not convert the passed-in parsedFileContents of type [{}] to CustomersVO.",
                    parsedFileContents::getClass
            );
            throw new RuntimeException("Could not convert the passed-in parsedFileContents of type [" + parsedFileContents.getClass()
                    .toString() + "] to CustomersVO.",
                    e
            );
        }
    }

    @Override
    public void process(final String fileName, final Object parsedFileContents) {
        super.process(fileName, parsedFileContents);
    }

    @Override
    public String getTitleKey() {
        return ArKeyConstants.CustomerLoad.MESSAGE_BATCH_UPLOAD_XML_TITLE_CUSTOMER;
    }

    public void setCustomerLoadService(final CustomerLoadService customerLoadService) {
        this.customerLoadService = customerLoadService;
    }

    @Override
    public String getAuthorPrincipalName(final File file) {
        final String[] fileNameParts = StringUtils.split(file.getName(), FILE_NAME_DELIM);
        if (fileNameParts.length > 3) {
            return fileNameParts[2];
        }
        return null;
    }
}
