/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.mo.common.active.MutableInactivatable;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleBillingService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Bills to be used for Billing Schedule under Contracts & Grants
 */
public class Bill extends BillBase implements MutableInactivatable {

    private String proposalNumber;
    private String chartOfAccountsCode;
    private String accountNumber;
    private boolean billed;
    private boolean active;
    private List<InvoiceBill> invoiceBills = new ArrayList<InvoiceBill>();

    private ContractsAndGrantsBillingAward award;

    private transient ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService;

    public Bill() {
    }

    private Bill(
            final String proposalNumber, final String chartOfAccountsCode, final String accountNumber, final boolean billed,
            final boolean active, final Long billIdentifier, final String billNumber, final String billDescription, final Date billDate,
            final KualiDecimal estimatedAmount) {
        this.proposalNumber = proposalNumber;
        this.chartOfAccountsCode = chartOfAccountsCode;
        this.accountNumber = accountNumber;
        this.billed = billed;
        this.active = active;
        this.billIdentifier = billIdentifier;
        this.billNumber = billNumber;
        this.billDescription = billDescription;
        this.billDate = billDate;
        this.estimatedAmount = estimatedAmount;
    }

    public String getProposalNumber() {
        return proposalNumber;
    }

    public void setProposalNumber(final String proposalNumber) {
        this.proposalNumber = proposalNumber;
    }

    public String getChartOfAccountsCode() {
        return chartOfAccountsCode;
    }

    public void setChartOfAccountsCode(final String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    public String getAccountNumber() {
        return accountNumber;
    }

    public void setAccountNumber(final String accountNumber) {
        this.accountNumber = accountNumber;
    }

    public boolean isBilled() {
        return billed;
    }

    public void setBilled(final boolean billed) {
        this.billed = billed;
    }

    public List<InvoiceBill> getInvoiceBills() {
        return invoiceBills;
    }

    public void setInvoiceBills(final List<InvoiceBill> invoiceBills) {
        this.invoiceBills = invoiceBills;
    }

    public ContractsAndGrantsBillingAward getAward() {
        final ContractsAndGrantsBillingAward updatedAward = getContractsAndGrantsModuleBillingService().
                updateAwardIfNecessary(proposalNumber, award);

        // If the updated award is null, we return the original award instead so that we don't get an NPE in the case
        // where it has been set by PojoPropertyUtilsBean.
        if (ObjectUtils.isNull(updatedAward)) {
            return award;
        }
        return updatedAward;
    }

    public void setAward(final ContractsAndGrantsBillingAward award) {
        this.award = award;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public void setActive(final boolean active) {
        this.active = active;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Bill)) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        final Bill bill = (Bill) o;
        return billed == bill.billed &&
                active == bill.active &&
                Objects.equals(proposalNumber, bill.proposalNumber) &&
                Objects.equals(chartOfAccountsCode, bill.chartOfAccountsCode) &&
                Objects.equals(accountNumber, bill.accountNumber);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), proposalNumber, chartOfAccountsCode, accountNumber, billed, active);
    }

    protected ContractsAndGrantsModuleBillingService getContractsAndGrantsModuleBillingService() {
        if (contractsAndGrantsModuleBillingService == null) {
            contractsAndGrantsModuleBillingService = SpringContext.getBean(ContractsAndGrantsModuleBillingService.class);
        }
        return contractsAndGrantsModuleBillingService;
    }

    protected void setContractsAndGrantsModuleBillingService(
            final ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService) {
        this.contractsAndGrantsModuleBillingService = contractsAndGrantsModuleBillingService;
    }

    public String getMostRecentInvoiceDocumentNumber() {
        String invoiceDocumentNumber = null;

        if (CollectionUtils.isNotEmpty(invoiceBills)) {
            final Long latestDocumentNumber = invoiceBills.stream()
                    .mapToLong(invoiceBill -> Long.parseLong(invoiceBill.getDocumentNumber()))
                    .max().getAsLong();
            invoiceDocumentNumber = String.valueOf(latestDocumentNumber);
        }

        return invoiceDocumentNumber;
    }

    public static class BillBuilder {
        private String proposalNumber;
        private String chartOfAccountsCode;
        private String accountNumber;
        private boolean billed;
        private boolean active;
        private Long billIdentifier;
        private String billNumber;
        private String billDescription;
        private Date billDate;
        private KualiDecimal estimatedAmount;

        public BillBuilder setProposalNumber(final String proposalNumber) {
            this.proposalNumber = proposalNumber;
            return this;
        }

        public BillBuilder setChartOfAccountsCode(final String chartOfAccountsCode) {
            this.chartOfAccountsCode = chartOfAccountsCode;
            return this;
        }

        public BillBuilder setAccountNumber(final String accountNumber) {
            this.accountNumber = accountNumber;
            return this;
        }

        public BillBuilder setBilled(final boolean billed) {
            this.billed = billed;
            return this;
        }

        public BillBuilder setActive(final boolean active) {
            this.active = active;
            return this;
        }

        public BillBuilder setBillIdentifier(final Long billIdentifier) {
            this.billIdentifier = billIdentifier;
            return this;
        }

        public BillBuilder setBillNumber(final String billNumber) {
            this.billNumber = billNumber;
            return this;
        }

        public BillBuilder setBillDescription(final String billDescription) {
            this.billDescription = billDescription;
            return this;
        }

        public BillBuilder setBillDate(final Date billDate) {
            this.billDate = billDate;
            return this;
        }

        public BillBuilder setEstimatedAmount(final KualiDecimal estimatedAmount) {
            this.estimatedAmount = estimatedAmount;
            return this;
        }

        public Bill build() {
            validate();
            return new Bill(proposalNumber, chartOfAccountsCode, accountNumber, billed, active, billIdentifier,
                    billNumber, billDescription, billDate, estimatedAmount);
        }

        private void validate() {
            if (StringUtils.isBlank(proposalNumber)) {
                throw new IllegalStateException("Proposal Number is required.");
            }
            if (StringUtils.isBlank(chartOfAccountsCode)) {
                throw new IllegalStateException("Chart of Accounts Code is required.");
            }
            if (StringUtils.isBlank(accountNumber)) {
                throw new IllegalStateException("Account Number is required.");
            }
            if (StringUtils.isBlank(billNumber)) {
                throw new IllegalStateException("Bill Number is required.");
            }
            if (estimatedAmount == null) {
                throw new IllegalStateException("Estimated Amount is required.");
            }
            if (billDate == null) {
                throw new IllegalStateException("Bill Date is required.");
            }
        }
    }
}
