/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.dataaccess.impl;

import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.broker.query.ReportQueryByCriteria;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.dataaccess.CustomerAgingReportDao;
import org.kuali.kfs.module.ar.document.CustomerInvoiceWriteoffDocument;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.core.framework.persistence.ojb.dao.PlatformAwareDaoBaseOjb;

import java.util.HashMap;
import java.util.Iterator;

/**
 * This is Implementation class for CustomerAgingReportDao DAO Interface.
 */
public class CustomerAgingReportDaoOjb extends PlatformAwareDaoBaseOjb implements CustomerAgingReportDao {

    @Override
    public HashMap<String, KualiDecimal> findInvoiceAmountByProcessingChartAndOrg(
            final String chart, final String org,
            final java.sql.Date begin, final java.sql.Date end) {
        final HashMap<String, KualiDecimal> map = new HashMap<>();
        final Criteria criteria = new Criteria();
        if (ObjectUtils.isNotNull(begin)) {
            criteria.addGreaterOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            criteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CHART_CODE, chart);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_ORG_CODE, org);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;
    }

    @Override
    public HashMap<String, KualiDecimal> findAppliedAmountByProcessingChartAndOrg(
            final String chart, final String org,
            final java.sql.Date begin, final java.sql.Date end) {
        final HashMap<String, KualiDecimal> map = new HashMap<>();

        // get approved application payments only
        final Criteria subCriteria = new Criteria();
        subCriteria.addEqualTo(KFSPropertyConstants.FINANCIAL_DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        final ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(DocumentHeader.class, subCriteria);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.DOCUMENT_NUMBER});

        final Criteria criteria = new Criteria();
        criteria.addIn(KFSPropertyConstants.DOCUMENT_NUMBER, subQuery);
        if (begin != null) {
            criteria.addGreaterOrEqualThan("customerInvoiceDocument.billingDate", begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            criteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CHART_CODE, chart);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_ORG_CODE, org);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        criteria.addEqualTo(ArPropertyConstants.InvoicePaidAppliedFields.ADJUSTED, false);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(InvoicePaidApplied.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.INVOICE_ITEM_APPLIED_AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;
    }

    @Override
    public HashMap<String, KualiDecimal> findDiscountAmountByProcessingChartAndOrg(final String chart, final String org, final java.sql.Date begin, final java.sql.Date end) {

        final HashMap<String, KualiDecimal> map = new HashMap<>();
        final Criteria subCriteria = new Criteria();
        if (ObjectUtils.isNotNull(begin)) {
            subCriteria.addGreaterOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            subCriteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CHART_CODE, chart);
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_ORG_CODE, org);
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        subCriteria.addEqualToField(KFSPropertyConstants.DOCUMENT_NUMBER, Criteria.PARENT_QUERY_PREFIX + KFSPropertyConstants.DOCUMENT_NUMBER);

        final ReportQueryByCriteria subReportQuery = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.INVOICE_ITEM_DISCOUNT_LINE_NUMBER}, subCriteria);

        final Criteria criteria = new Criteria();
        criteria.addIn(KFSPropertyConstants.SEQUENCE_NUMBER, subReportQuery);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;

    }

    @Override
    public HashMap<String, KualiDecimal> findInvoiceAmountByBillingChartAndOrg(
            final String chart, final String org,
            final java.sql.Date begin, final java.sql.Date end) {
        final HashMap<String, KualiDecimal> map = new HashMap<>();
        final Criteria criteria = new Criteria();
        if (ObjectUtils.isNotNull(begin)) {
            criteria.addGreaterOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            criteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.BILL_BY_CHART_OF_ACCOUNT_CODE, chart);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.BILLED_BY_ORGANIZATION_CODE, org);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;
    }

    @Override
    public HashMap<String, KualiDecimal> findAppliedAmountByBillingChartAndOrg(
            final String chart, final String org,
            final java.sql.Date begin, final java.sql.Date end) {
        final HashMap<String, KualiDecimal> map = new HashMap<>();

        // get approved application payments only
        final Criteria subCriteria = new Criteria();
        subCriteria.addEqualTo(KFSPropertyConstants.FINANCIAL_DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        final ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(DocumentHeader.class, subCriteria);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.DOCUMENT_NUMBER});

        final Criteria criteria = new Criteria();
        criteria.addIn(KFSPropertyConstants.DOCUMENT_NUMBER, subQuery);
        if (begin != null) {
            criteria.addGreaterOrEqualThan("customerInvoiceDocument.billingDate", begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            criteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.BILL_BY_CHART_OF_ACCOUNT_CODE, chart);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.BILLED_BY_ORGANIZATION_CODE, org);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        criteria.addEqualTo(ArPropertyConstants.InvoicePaidAppliedFields.ADJUSTED, false);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(InvoicePaidApplied.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.INVOICE_ITEM_APPLIED_AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;
    }

    @Override
    public HashMap<String, KualiDecimal> findDiscountAmountByBillingChartAndOrg(
            final String chart, final String org,
            final java.sql.Date begin, final java.sql.Date end) {

        final HashMap<String, KualiDecimal> map = new HashMap<>();
        final Criteria subCriteria = new Criteria();
        if (ObjectUtils.isNotNull(begin)) {
            subCriteria.addGreaterOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            subCriteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.BILL_BY_CHART_OF_ACCOUNT_CODE, chart);
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.BILLED_BY_ORGANIZATION_CODE, org);
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        subCriteria.addEqualToField(KFSPropertyConstants.DOCUMENT_NUMBER, Criteria.PARENT_QUERY_PREFIX + KFSPropertyConstants.DOCUMENT_NUMBER);

        final ReportQueryByCriteria subReportQuery = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.INVOICE_ITEM_DISCOUNT_LINE_NUMBER}, subCriteria);

        final Criteria criteria = new Criteria();
        criteria.addIn(KFSPropertyConstants.SEQUENCE_NUMBER, subReportQuery);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;

    }

    @Override
    public HashMap<String, KualiDecimal> findInvoiceAmountByAccount(
            final String chart, final String account, final java.sql.Date begin,
            final java.sql.Date end) {
        final HashMap<String, KualiDecimal> map = new HashMap<>();
        final Criteria criteria = new Criteria();
        if (ObjectUtils.isNotNull(begin)) {
            criteria.addGreaterOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            criteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        criteria.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chart);
        criteria.addEqualTo(KFSPropertyConstants.ACCOUNT_NUMBER, account);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;
    }

    @Override
    public HashMap<String, KualiDecimal> findAppliedAmountByAccount(
            final String chart, final String account, final java.sql.Date begin,
            final java.sql.Date end) {
        final HashMap<String, KualiDecimal> map = new HashMap<>();

        // get approved application payments only
        final Criteria subCriteria = new Criteria();
        subCriteria.addEqualTo(KFSPropertyConstants.FINANCIAL_DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        final ReportQueryByCriteria subQuery = QueryFactory.newReportQuery(DocumentHeader.class, subCriteria);
        subQuery.setAttributes(new String[]{KFSPropertyConstants.DOCUMENT_NUMBER});

        final Criteria criteria = new Criteria();
        criteria.addIn(KFSPropertyConstants.DOCUMENT_NUMBER, subQuery);
        if (begin != null) {
            criteria.addGreaterOrEqualThan("customerInvoiceDocument.billingDate", begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            criteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.CHART_OF_ACCOUNTS_CODE, chart);
        criteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNT_NUMBER, account);
        criteria.addEqualTo(ArPropertyConstants.InvoicePaidAppliedFields.ADJUSTED, false);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(InvoicePaidApplied.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.INVOICE_ITEM_APPLIED_AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;
    }

    @Override
    public HashMap<String, KualiDecimal> findDiscountAmountByAccount(
            final String chart, final String account, final java.sql.Date begin,
            final java.sql.Date end) {

        final HashMap<String, KualiDecimal> map = new HashMap<>();
        final Criteria subCriteria = new Criteria();
        if (ObjectUtils.isNotNull(begin)) {
            subCriteria.addGreaterOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, begin);
        }
        if (ObjectUtils.isNotNull(end)) {
            subCriteria.addLessOrEqualThan(ArPropertyConstants.CustomerInvoiceDetailFields.BILLING_DATE, end);
        }
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.DOCUMENT_STATUS_CODE, KFSConstants.DocumentStatusCodes.APPROVED);
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.OPEN_INVOICE_IND, true);
        subCriteria.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chart);
        subCriteria.addEqualTo(KFSPropertyConstants.ACCOUNT_NUMBER, account);
        subCriteria.addEqualToField(KFSPropertyConstants.DOCUMENT_NUMBER, Criteria.PARENT_QUERY_PREFIX + KFSPropertyConstants.DOCUMENT_NUMBER);

        final ReportQueryByCriteria subReportQuery = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.INVOICE_ITEM_DISCOUNT_LINE_NUMBER}, subCriteria);

        final Criteria criteria = new Criteria();
        criteria.addIn(KFSPropertyConstants.SEQUENCE_NUMBER, subReportQuery);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(CustomerInvoiceDetail.class, new String[]{ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER, ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME, "sum(" + ArPropertyConstants.CustomerInvoiceDetailFields.AMOUNT + ")"}, criteria);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NUMBER);
        reportByCriteria.addGroupBy(ArPropertyConstants.CustomerInvoiceDetailFields.ACCOUNTS_RECEIVABLE_CUSTOMER_NAME);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);

        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            map.put(data[0] + "-" + data[1], (KualiDecimal) data[2]);
        }
        return map;

    }

    @Override
    public KualiDecimal findWriteOffAmountByCustomerNumber(final String customerNumber) {
        KualiDecimal writeOffAmt = KualiDecimal.ZERO;
        final Criteria subCriteria = new Criteria();
        subCriteria.addEqualTo(ArPropertyConstants.CustomerInvoiceDetailFields.WRITEOFF_CUSTOMER_NUMBER, customerNumber);
        final ReportQueryByCriteria reportByCriteria = new ReportQueryByCriteria(CustomerInvoiceWriteoffDocument.class,
                new String[]{"sum(" + ArPropertyConstants.CustomerInvoiceWriteoffDocumentFields.INVOICE_WRITEOFF_AMOUNT + ")"}, subCriteria);
        final Iterator<?> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportByCriteria);
        while (ObjectUtils.isNotNull(iterator) && iterator.hasNext()) {
            final Object[] data = (Object[]) iterator.next();
            writeOffAmt = (KualiDecimal) data[0];
        }
        return writeOffAmt;
    }

}
