/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.Maintainable;
import org.kuali.kfs.kns.web.ui.Field;
import org.kuali.kfs.kns.web.ui.Section;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.Bill;
import org.kuali.kfs.module.ar.businessobject.PredeterminedBillingSchedule;
import org.kuali.kfs.module.ar.document.service.PredeterminedBillingScheduleMaintenanceService;
import org.kuali.kfs.module.ar.document.validation.impl.PredeterminedBillingScheduleRuleUtil;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.FinancialSystemMaintainable;

import java.util.List;
import java.util.Map;

/**
 * Methods for the Predetermined Billing Schedule maintenance document UI.
 */
public class PredeterminedBillingScheduleMaintainableImpl extends FinancialSystemMaintainable {
    private static volatile PredeterminedBillingScheduleMaintenanceService predeterminedBillingScheduleMaintenanceService;

    /**
     * This method is called to check if the award/account already has bills set, and to validate on refresh
     */
    @Override
    public void refresh(final String refreshCaller, final Map fieldValues, final MaintenanceDocument document) {
        if (StringUtils.equals(ArConstants.AWARD_ACCOUNT_LOOKUP_IMPL, (String) fieldValues.get(KFSConstants.REFRESH_CALLER))) {
            final PredeterminedBillingSchedule predeterminedBillingSchedule = getPredeterminedBillingSchedule();
            predeterminedBillingSchedule.setProposalNumber(predeterminedBillingSchedule
                .getProposalNumberForAwardAccountLookup());
            predeterminedBillingSchedule.setChartOfAccountsCode(predeterminedBillingSchedule
                .getChartOfAccountsCodeForAwardAccountLookup());
            predeterminedBillingSchedule.setAccountNumber(predeterminedBillingSchedule
                .getAccountNumberForAwardAccountLookup());

            if (PredeterminedBillingScheduleRuleUtil.checkIfBillsExist(predeterminedBillingSchedule)) {
                final String pathToMaintainable = KFSPropertyConstants.DOCUMENT + "."
                                                  + KFSPropertyConstants.NEW_MAINTAINABLE_OBJECT;
                GlobalVariables.getMessageMap().addToErrorPath(pathToMaintainable);
                GlobalVariables.getMessageMap().putError(ArPropertyConstants.PROPOSAL_NUMBER_FOR_AWARD_ACCOUNT_LOOKUP,
                    ArKeyConstants.ERROR_AWARD_PREDETERMINED_BILLING_SCHEDULE_EXISTS,
                    predeterminedBillingSchedule.getProposalNumber(),
                    predeterminedBillingSchedule.getChartOfAccountsCode(),
                    predeterminedBillingSchedule.getAccountNumber());
                GlobalVariables.getMessageMap().removeFromErrorPath(pathToMaintainable);
            }
        } else {
            super.refresh(refreshCaller, fieldValues, document);
        }
    }

    /**
     * Not to copy over the Bills billed and billIdentifier values to prevent
     * bad data and PK issues when saving new Bills.
     */
    @Override
    public void processAfterCopy(final MaintenanceDocument document, final Map<String, String[]> parameters) {
        super.processAfterCopy(document, parameters);

        // clear out Bill IDs so new ones will get generated for these bills
        // reset billed indicator in case bill we're copying from was already billed
        final List<Bill> bills = getPredeterminedBillingSchedule().getBills();
        if (ObjectUtils.isNotNull(bills)) {
            for (final Bill bill : bills) {
                bill.setBilled(false);
                bill.setBillIdentifier(null);
            }
        }
    }

    /**
     * Refresh any references that may need to be reconstituted after the PredeterminedBillingSchedule has been
     * populated from the maintenance document xml contents.
     */
    @Override
    public void processAfterRetrieve() {
        final PredeterminedBillingSchedule predeterminedBillingSchedule = getPredeterminedBillingSchedule();
        predeterminedBillingSchedule.refreshNonUpdateableReferences();
        predeterminedBillingSchedule.forceAwardUpdate();
        super.processAfterRetrieve();
    }

    /**
     * Gets the underlying Predetermined Billing Schedule.
     *
     * @return
     */
    private PredeterminedBillingSchedule getPredeterminedBillingSchedule() {
        return (PredeterminedBillingSchedule) getBusinessObject();
    }

    /**
     * Override the getSections method on this maintainable so that the active field can be set to read-only when
     * a CINV doc has been created with this Predetermined Billing Schedule and Bills
     */
    @Override
    public List getSections(final MaintenanceDocument document, final Maintainable oldMaintainable) {
        final List<Section> sections = super.getSections(document, oldMaintainable);
        final String proposalNumber = getPredeterminedBillingSchedule().getProposalNumber();

        for (final Section section : sections) {
            final String sectionId = section.getSectionId();
            if (sectionId.equalsIgnoreCase(ArPropertyConstants.BILLS)) {
                prepareBillsTab(section, proposalNumber);
            }
        }

        return sections;
    }

    /**
     * Sets the Bill in the passed in section to be readonly if it has been copied to a CG Invoice doc.
     *
     * @param section        Bill section to review and possibly set readonly
     * @param proposalNumber used to look for CG Invoice docs
     */
    private void prepareBillsTab(final Section section, final String proposalNumber) {
        section.getRows().forEach(row -> row.getFields().stream()
                .filter(field -> field.getCONTAINER().equalsIgnoreCase(field.getFieldType()))
                .flatMap(field -> field.getContainerRows().stream())
                .flatMap(containerRow -> containerRow.getFields().stream())
                .filter(containerRowfield ->
                        ObjectUtils.getNestedAttributePrimitive(containerRowfield.getPropertyName())
                                .matches(ArPropertyConstants.BillFields.BILL_IDENTIFIER))
                .map(Field::getPropertyValue)
                .filter(StringUtils::isNotEmpty)
                .filter(billId -> getPredeterminedBillingScheduleMaintenanceService()
                        .hasBillBeenCopiedToInvoice(proposalNumber, billId))
                .flatMap(billId -> row.getFields().stream())
                .filter(rowfield -> rowfield.getCONTAINER().equalsIgnoreCase(rowfield.getFieldType()))
                .flatMap(rowfield -> rowfield.getContainerRows().stream())
                .flatMap(fieldContainerRow -> fieldContainerRow.getFields().stream())
                .forEach(fieldContainerRowField -> fieldContainerRowField.setReadOnly(true)));
    }

    private PredeterminedBillingScheduleMaintenanceService getPredeterminedBillingScheduleMaintenanceService() {
        if (predeterminedBillingScheduleMaintenanceService == null) {
            predeterminedBillingScheduleMaintenanceService = SpringContext
                .getBean(PredeterminedBillingScheduleMaintenanceService.class);
        }
        return predeterminedBillingScheduleMaintenanceService;
    }
}
