/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kew.doctype.service.DocumentTypeService;
import org.kuali.kfs.kew.service.KEWServiceLocator;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.CashControlDetail;
import org.kuali.kfs.module.ar.document.CashControlDocument;
import org.kuali.kfs.module.ar.document.service.AccountsReceivableDocumentHeaderService;
import org.kuali.kfs.module.ar.document.service.CashControlDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.ElectronicPaymentClaim;
import org.kuali.kfs.sys.service.ElectronicPaymentClaimingDocumentGenerationStrategy;
import org.kuali.kfs.sys.service.ElectronicPaymentClaimingService;
import org.kuali.kfs.sys.service.FinancialSystemUserService;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class CashControlElectronicPaymentClaimingHelperImpl implements
        ElectronicPaymentClaimingDocumentGenerationStrategy {

    protected DataDictionaryService dataDictionaryService;
    protected DocumentService documentService;
    protected ElectronicPaymentClaimingService electronicPaymentClaimingService;
    protected CashControlDocumentService cashControlDocumentService;
    protected ConfigurationService kualiConfigurationService;
    protected FinancialSystemUserService financialSystemUserService;
    private static DocumentTypeService documentTypeService;
    private AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService;

    @Override
    public String createDocumentFromElectronicPayments(final List<ElectronicPaymentClaim> electronicPayments, final Person user) {
        final CashControlDocument document;
        document = (CashControlDocument) documentService.getNewDocument(getClaimingDocumentWorkflowDocumentType());
        document.setCustomerPaymentMediumCode(ArConstants.PaymentMediumCode.WIRE_TRANSFER);

        //create and set AccountsReceivableDocumentHeader
        final AccountsReceivableDocumentHeader accountsReceivableDocumentHeader =
                accountsReceivableDocumentHeaderService.getNewAccountsReceivableDocumentHeaderForCurrentUser();
        accountsReceivableDocumentHeader.setDocumentNumber(document.getDocumentNumber());
        document.setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);

        addDescriptionToDocument(document);
        addNotesToDocument(document, electronicPayments, user);
        addCashControlDetailsToDocument(document, electronicPayments);
        documentService.saveDocument(document);
        electronicPaymentClaimingService.claimElectronicPayments(electronicPayments, document.getDocumentNumber());

        return getURLForDocument(document);
    }

    /**
     * This method add a description to the cash control document
     *
     * @param document the cash control document
     */
    protected void addDescriptionToDocument(final CashControlDocument document) {
        document.getDocumentHeader().setDocumentDescription(kualiConfigurationService.getPropertyValueAsString(
                ArKeyConstants.ELECTRONIC_PAYMENT_CLAIM));
    }

    /**
     * This method adds notes to the cash control document
     *
     * @param claimingDoc the cash control document
     * @param claims      the list of electronic payments being claimed
     * @param user        the current user
     */
    protected void addNotesToDocument(
            final CashControlDocument claimingDoc, final List<ElectronicPaymentClaim> claims,
            final Person user) {
        for (final String noteText : electronicPaymentClaimingService.constructNoteTextsForClaims(claims)) {
            final Note note = documentService.createNoteFromDocument(claimingDoc, noteText);
            claimingDoc.addNote(note);
            documentService.saveDocumentNotes(claimingDoc);
        }
    }

    /**
     * This method adds new cash control details to the cash control document based on the list of electronic payments.
     *
     * @param document           cash control document
     * @param electronicPayments the electronic payments to be claimed
     */
    protected void addCashControlDetailsToDocument(
            final CashControlDocument document,
            final List<ElectronicPaymentClaim> electronicPayments) {
        for (final ElectronicPaymentClaim electronicPaymentClaim : electronicPayments) {
            final CashControlDetail newCashControlDetail = new CashControlDetail();
            newCashControlDetail.setCashControlDocument(document);
            newCashControlDetail.setDocumentNumber(document.getDocumentNumber());
            newCashControlDetail.setFinancialDocumentLineAmount(electronicPaymentClaim.getGeneratingAccountingLine()
                    .getAmount());
            newCashControlDetail.setCustomerPaymentDescription(electronicPaymentClaim.getGeneratingAccountingLine()
                    .getFinancialDocumentLineDescription());
            cashControlDocumentService.addNewCashControlDetail(kualiConfigurationService.getPropertyValueAsString(
                    ArKeyConstants.CREATED_BY_CASH_CTRL_DOC), document, newCashControlDetail);
        }
    }

    /**
     * Builds the URL that can be used to redirect to the correct document
     *
     * @param doc the document to build the URL for
     * @return the relative URL to redirect to
     */
    protected String getURLForDocument(final CashControlDocument doc) {
        final Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.DOC_HANDLER_METHOD);
        parameters.put(KFSConstants.PARAMETER_COMMAND, KewApiConstants.DOCSEARCH_COMMAND);
        parameters.put(KFSConstants.PARAMETER_DOC_ID, doc.getDocumentNumber());

        return UrlFactory.parameterizeUrl(ArConstants.UrlActions.CASH_CONTROL_DOCUMENT, parameters);
    }

    /**
     * @return the name CashControlDocument workflow document type
     */
    @Override
    public String getClaimingDocumentWorkflowDocumentType() {
        return ArConstants.ArDocumentTypeCodes.CASH_CONTROL;
    }

    @Override
    public String getDocumentLabel() {
        return getDocumentTypeService().getDocumentTypeByName(getClaimingDocumentWorkflowDocumentType()).getLabel();
    }

    @Override
    public boolean isDocumentReferenceValid(final String referenceDocumentNumber) {
        boolean isValid = false;
        try {
            final long docNumberAsLong = Long.parseLong(referenceDocumentNumber);
            if (docNumberAsLong > 0L) {
                isValid = documentService.documentExists(referenceDocumentNumber);
            }
        } catch (final NumberFormatException nfe) {
            isValid = false;
        }
        return isValid;
    }

    @Override
    public boolean userMayUseToClaim(final Person claimingUser) {
        final String documentTypeName = getClaimingDocumentWorkflowDocumentType();

        return electronicPaymentClaimingService.isAuthorizedForClaimingElectronicPayment(claimingUser,
                documentTypeName) || electronicPaymentClaimingService.isAuthorizedForClaimingElectronicPayment(
                claimingUser, null);
    }

    public void setCashControlDocumentService(final CashControlDocumentService cashControlDocumentService) {
        this.cashControlDocumentService = cashControlDocumentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setElectronicPaymentClaimingService(final ElectronicPaymentClaimingService electronicPaymentClaimingService) {
        this.electronicPaymentClaimingService = electronicPaymentClaimingService;
    }

    public void setDataDictionaryService(final DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public void setKualiConfigurationService(final ConfigurationService kualiConfigurationService) {
        this.kualiConfigurationService = kualiConfigurationService;
    }

    public DocumentTypeService getDocumentTypeService() {
        if (documentTypeService == null) {
            documentTypeService = KEWServiceLocator.getDocumentTypeService();
        }
        return documentTypeService;
    }

    public void setAccountsReceivableDocumentHeaderService(
            final AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService) {
        this.accountsReceivableDocumentHeaderService = accountsReceivableDocumentHeaderService;
    }

    public FinancialSystemUserService getFinancialSystemUserService() {
        return financialSystemUserService;
    }

    public void setFinancialSystemUserService(final FinancialSystemUserService financialSystemUserService) {
        this.financialSystemUserService = financialSystemUserService;
    }
}
