/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.kew.doctype.service.DocumentTypeService;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.CollectionEvent;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsCollectionActivityInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.document.ContractsGrantsCollectionActivityDocument;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.dataaccess.ContractsGrantsInvoiceDocumentDao;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsCollectionActivityDocumentService;
import org.kuali.kfs.module.ar.document.service.InvoicePaidAppliedService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

public class ContractsGrantsCollectionActivityDocumentServiceImpl implements
        ContractsGrantsCollectionActivityDocumentService {

    protected ContractsGrantsInvoiceDocumentDao contractsGrantsInvoiceDocumentDao;
    protected DocumentService documentService;
    protected DateTimeService dateTimeService;
    protected BusinessObjectService businessObjectService;
    protected DocumentTypeService documentTypeService;
    protected InvoicePaidAppliedService invoicePaidAppliedService;
    protected KualiModuleService kualiModuleService;
    private PersonService personService;

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setDocumentTypeService(final DocumentTypeService documentTypeService) {
        this.documentTypeService = documentTypeService;
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    @Override
    @Transactional
    public void createAndSaveCollectionEvents(final ContractsGrantsCollectionActivityDocument colActDoc) {
        for (final ContractsGrantsCollectionActivityInvoiceDetail invoiceDetail : colActDoc.getInvoiceDetails()) {
            final CollectionEvent newCollectionEvent = new CollectionEvent();
            final Timestamp now = dateTimeService.getCurrentTimestamp();

            newCollectionEvent.setPostedDate(now);
            newCollectionEvent.setActivityCode(colActDoc.getActivityCode());
            newCollectionEvent.setActivityDate(colActDoc.getActivityDate());
            newCollectionEvent.setActivityText(colActDoc.getActivityText());
            newCollectionEvent.setFollowupDate(colActDoc.getFollowupDate());
            newCollectionEvent.setCompletedDate(colActDoc.getCompletedDate());

            final String initiatorPrincipalId = colActDoc.getDocumentHeader().getWorkflowDocument()
                    .getInitiatorPrincipalId();
            final Person initiator = personService.getPerson(initiatorPrincipalId);
            newCollectionEvent.setUserPrincipalId(initiatorPrincipalId);
            newCollectionEvent.setUser(initiator);

            final ContractsGrantsInvoiceDocument invoice = invoiceDetail.getInvoiceDocument();
            newCollectionEvent.setCollectionEventCode(invoice.getNextCollectionEventCode());
            newCollectionEvent.setInvoiceNumber(invoice.getDocumentNumber());
            newCollectionEvent.setProposalNumber(colActDoc.getProposalNumber());
            businessObjectService.save(newCollectionEvent);
        }
    }

    @Override
    @Transactional
    public ContractsAndGrantsBillingAward retrieveAwardByProposalNumber(final String proposalNumber) {
        ContractsAndGrantsBillingAward award = null;
        if (ObjectUtils.isNotNull(proposalNumber)) {
            final Map<String, Object> map = new HashMap<>();
            map.put(KFSPropertyConstants.PROPOSAL_NUMBER, proposalNumber);
            award = kualiModuleService.getResponsibleModuleService(ContractsAndGrantsBillingAward.class)
                    .getExternalizableBusinessObject(ContractsAndGrantsBillingAward.class, map);
        }
        return award;
    }

    @Override
    @Transactional
    public KualiDecimal retrievePaymentAmountByDocumentNumber(final String documentNumber) {
        KualiDecimal paymentAmount = KualiDecimal.ZERO;
        final Collection<InvoicePaidApplied> invoicePaidApplieds =
                invoicePaidAppliedService.getInvoicePaidAppliedsForInvoice(documentNumber);
        if (invoicePaidApplieds != null && !invoicePaidApplieds.isEmpty()) {
            for (final InvoicePaidApplied invPaidApp : invoicePaidApplieds) {
                paymentAmount = paymentAmount.add(invPaidApp.getInvoiceItemAppliedAmount());
            }
        }
        return paymentAmount;
    }

    /**
     * This method retrieves all collection activity eligible Contracts & Grants Invoices associated with the given
     * proposal number. All Contracts & Grants Invoices retrieved will meet the following criteria:
     * <ul>
     * <li>Must not be fully paid</li>
     * <li>Must not error correct another CINV</li>
     * <li>Must not be error corrected by another CINV</li>
     * <li>Must be final or processed</li>
     * </ul>
     *
     * @param proposalNumber
     * @return a Collection of collection activity eligible Contracts & Grants Invoices associated with the given proposal number
     */
    @Override
    @Transactional
    public Collection<ContractsGrantsInvoiceDocument> retrieveCollectionActivityEligibleContractsGrantsInvoicesByProposalNumber(
            final String proposalNumber) {
        final Collection<ContractsGrantsInvoiceDocument> cgInvoices = contractsGrantsInvoiceDocumentDao
                .getCollectionEligibleContractsGrantsInvoicesByProposalNumber(proposalNumber);
        if (CollectionUtils.isEmpty(cgInvoices)) {
            return cgInvoices;
        }
        final Collection<ContractsGrantsInvoiceDocument> filteredInvoices = new ArrayList<>();
        for (final ContractsGrantsInvoiceDocument invoice : cgInvoices) {
            if (!isFullyPaid(invoice)) {
                filteredInvoices.add(invoice);
            }
        }
        return filteredInvoices;
    }

    /**
     * Determines if a Contracts & Grants Invoice is fully paid or not
     *
     * @param contractsGrantsInvoiceDocument the Contracts & Grants Invoice to check
     * @return true if the document is fully paid, false otherwise
     */
    protected boolean isFullyPaid(final ContractsGrantsInvoiceDocument contractsGrantsInvoiceDocument) {
        final KualiDecimal openAmount = contractsGrantsInvoiceDocument.getOpenAmount();
        return ObjectUtils.isNotNull(openAmount) && openAmount.equals(KualiDecimal.ZERO);
    }

    public InvoicePaidAppliedService getInvoicePaidAppliedService() {
        return invoicePaidAppliedService;
    }

    public void setInvoicePaidAppliedService(final InvoicePaidAppliedService invoicePaidAppliedService) {
        this.invoicePaidAppliedService = invoicePaidAppliedService;
    }

    public ContractsGrantsInvoiceDocumentDao getContractsGrantsInvoiceDocumentDao() {
        return contractsGrantsInvoiceDocumentDao;
    }

    public void setContractsGrantsInvoiceDocumentDao(final ContractsGrantsInvoiceDocumentDao contractsGrantsInvoiceDocumentDao) {
        this.contractsGrantsInvoiceDocumentDao = contractsGrantsInvoiceDocumentDao;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }
}
