/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.SequenceAccessorService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.util.TransactionalServiceUtils;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

@Transactional
public class CustomerAddressServiceImpl implements CustomerAddressService {
    private BusinessObjectService businessObjectService;
    private SequenceAccessorService sequenceAccessorService;
    private DateTimeService dateTimeService;

    @Override
    @SuppressWarnings("unchecked")
    public CustomerAddress getByPrimaryKey(final String customerNumber, final Integer customerAddressIdentifier) {

        CustomerAddress customerAddress = null;
        if (StringUtils.isNotBlank(customerNumber) && ObjectUtils.isNotNull(customerAddressIdentifier)) {
            final Map criteria = new HashMap();
            criteria.put(KFSPropertyConstants.CUSTOMER_NUMBER, customerNumber);
            criteria.put(KFSPropertyConstants.CUSTOMER_ADDRESS_IDENTIFIER, customerAddressIdentifier);

            customerAddress = businessObjectService.findByPrimaryKey(CustomerAddress.class, criteria);
        }
        return customerAddress;
    }

    @Override
    @SuppressWarnings("unchecked")
    public CustomerAddress getPrimaryAddress(final String customerNumber) {
        CustomerAddress primaryAddress = null;
        if (StringUtils.isNotBlank(customerNumber)) {
            final Map criteria = new HashMap();
            criteria.put(KFSPropertyConstants.CUSTOMER_NUMBER, customerNumber);
            criteria.put(KFSPropertyConstants.CUSTOMER_ADDRESS_TYPE_CODE, "P");

            final Collection<CustomerAddress> primaryAddresses = businessObjectService.findMatching(CustomerAddress.class, criteria);
            if (!CollectionUtils.isEmpty(primaryAddresses)) {
                primaryAddress = TransactionalServiceUtils.retrieveFirstAndExhaustIterator(primaryAddresses.iterator());
            }
        }

        return primaryAddress;
    }

    /**
     * This method returns true if customer address is active
     *
     * @param customerNumber
     * @param customerAddressIdentifier
     * @return
     */
    @Override
    public boolean customerAddressActive(final String customerNumber, final Integer customerAddressIdentifier) {
        final CustomerAddress customerAddress = getByPrimaryKey(customerNumber, customerAddressIdentifier);

        if (ObjectUtils.isNotNull(customerAddress)) {
            if (ObjectUtils.isNotNull(customerAddress.getCustomerAddressEndDate())) {
                final Timestamp currentDateTimestamp = new Timestamp(dateTimeService.getCurrentDate().getTime());
                final Timestamp addressEndDateTimestamp = new Timestamp(customerAddress.getCustomerAddressEndDate().getTime());
                return !addressEndDateTimestamp.before(currentDateTimestamp);
            }
        }
        return true;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    @Override
    public boolean customerAddressExists(final String customerNumber, final Integer customerAddressIdentifier) {
        return ObjectUtils.isNotNull(getByPrimaryKey(customerNumber, customerAddressIdentifier));
    }

    @Override
    public Integer getNextCustomerAddressIdentifier() {
        final Long nextId = sequenceAccessorService.getNextAvailableSequenceNumber(ArConstants.CUST_ADDR_ID_SEQ,
                CustomerAddress.class);

        return nextId.intValue();
    }

    public SequenceAccessorService getSequenceAccessorService() {
        return sequenceAccessorService;
    }

    public void setSequenceAccessorService(final SequenceAccessorService sequenceAccessorService) {
        this.sequenceAccessorService = sequenceAccessorService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }
}
