/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.maintenance.rules.MaintenanceDocumentRuleBase;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.InvoiceRecurrence;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceRecurrenceUtils;
import org.kuali.kfs.module.ar.document.service.InvoiceRecurrenceDocumentService;
import org.kuali.kfs.sys.context.SpringContext;

import java.sql.Date;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class InvoiceRecurrenceRule extends MaintenanceDocumentRuleBase {

    protected InvoiceRecurrence oldInvoiceRecurrence;
    protected InvoiceRecurrence newInvoiceRecurrence;

    protected DateTimeService dateTimeService;

    @Override
    public void setupConvenienceObjects() {
        oldInvoiceRecurrence = (InvoiceRecurrence) super.getOldBo();
        newInvoiceRecurrence = (InvoiceRecurrence) super.getNewBo();
    }

    @Override
    protected boolean processCustomRouteDocumentBusinessRules(final MaintenanceDocument document) {
        boolean success;

        success = checkIfInvoiceIsApproved(newInvoiceRecurrence.getInvoiceNumber());
        success &= validateDocumentRecurrenceEndDate(newInvoiceRecurrence.getDocumentRecurrenceBeginDate(),
                newInvoiceRecurrence.getDocumentRecurrenceEndDate());
        success &= validateIfBothEndDateAndTotalRecurrenceNumberAreEntered(
                newInvoiceRecurrence.getDocumentRecurrenceBeginDate(),
                newInvoiceRecurrence.getDocumentRecurrenceEndDate(),
                newInvoiceRecurrence.getDocumentTotalRecurrenceNumber(),
                newInvoiceRecurrence.getDocumentRecurrenceIntervalCode());
        success &= validateEndDateOrTotalNumberofRecurrences(newInvoiceRecurrence.getDocumentRecurrenceEndDate(),
                newInvoiceRecurrence.getDocumentTotalRecurrenceNumber());
        success &= validateMaximumNumberOfRecurrences(newInvoiceRecurrence.getDocumentTotalRecurrenceNumber(),
                newInvoiceRecurrence.getDocumentRecurrenceIntervalCode());
        return success;
    }

    @Override
    protected boolean processCustomSaveDocumentBusinessRules(final MaintenanceDocument document) {
        // always return true even if there are business rule failures.
        processCustomRouteDocumentBusinessRules(document);
        return true;
    }

    /**
     * Check if an Invoice Recurrence Maintenance document already exists.
     */
    protected boolean checkIfRecurrenceMaintenanceAlreadyExists(final InvoiceRecurrence newInvoiceRecurrence) {
        boolean success = true;
        final Map<String, String> criteria = new HashMap<>();
        criteria.put("invoiceNumber", newInvoiceRecurrence.getInvoiceNumber());
        final InvoiceRecurrence invoiceRecurrence = SpringContext.getBean(BusinessObjectService.class)
                .findByPrimaryKey(InvoiceRecurrence.class, criteria);
        if (ObjectUtils.isNotNull(invoiceRecurrence) && !oldInvoiceRecurrence.equals(invoiceRecurrence)) {
            putFieldError(ArPropertyConstants.INVOICE_NUMBER, ArKeyConstants.ERROR_MAINTENANCE_DOCUMENT_ALREADY_EXISTS);
            success = false;
        }
        return success;
    }

    /**
     * Validate if the invoice has an approved status.
     */
    protected boolean checkIfInvoiceIsApproved(final String recurrenceInvoiceNumber) {
        boolean success = true;

        if (ObjectUtils.isNull(recurrenceInvoiceNumber)) {
            return true;
        }

        CustomerInvoiceDocument customerInvoiceDocument = null;
        if (!SpringContext.getBean(DocumentService.class).documentExists(recurrenceInvoiceNumber)) {
            putFieldError(ArPropertyConstants.INVOICE_NUMBER, ArKeyConstants.ERROR_INVOICE_DOES_NOT_EXIST);
            success = false;
        } else {
            customerInvoiceDocument = (CustomerInvoiceDocument) SpringContext.getBean(DocumentService.class)
                    .getByDocumentHeaderId(recurrenceInvoiceNumber);
            if (ObjectUtils.isNotNull(customerInvoiceDocument)) {
                final WorkflowDocument workflowDocument = customerInvoiceDocument.getDocumentHeader().getWorkflowDocument();
                if (!workflowDocument.isApproved()) {
                    putFieldError(ArPropertyConstants.INVOICE_NUMBER,
                            ArKeyConstants.ERROR_RECURRING_INVOICE_NUMBER_MUST_BE_APPROVED);
                    success = false;
                }
            }
        }
        return success;
    }

    /**
     * Validate Begin Date.
     */
    protected boolean validateDocumentRecurrenceBeginDate(final InvoiceRecurrence newInvoiceRecurrence) {
        if (ObjectUtils.isNull(newInvoiceRecurrence.getDocumentRecurrenceBeginDate())) {
            return true;
        }
        final Timestamp currentDate = new Timestamp(getDateTimeService().getCurrentDate().getTime());
        final Timestamp beginDateTimestamp = new Timestamp(newInvoiceRecurrence.getDocumentRecurrenceBeginDate().getTime());
        if (beginDateTimestamp.before(currentDate) || beginDateTimestamp.equals(currentDate)) {
            putFieldError(ArPropertyConstants.InvoiceRecurrenceFields.INVOICE_RECURRENCE_BEGIN_DATE,
                    ArKeyConstants.ERROR_INVOICE_RECURRENCE_BEGIN_DATE_EARLIER_THAN_TODAY);
            return false;
        }
        return true;
    }

    /**
     * Validate End Date.
     */
    public boolean validateDocumentRecurrenceEndDate(
            final Date newInvoiceRecurrenceBeginDate,
            final Date newInvoiceRecurrenceEndDate) {
        if (!SpringContext.getBean(InvoiceRecurrenceDocumentService.class)
                .isValidRecurrenceEndDate(newInvoiceRecurrenceBeginDate, newInvoiceRecurrenceEndDate)) {
            putFieldError(ArPropertyConstants.InvoiceRecurrenceFields.INVOICE_RECURRENCE_END_DATE,
                    ArKeyConstants.ERROR_END_DATE_EARLIER_THAN_BEGIN_DATE);
            return false;
        }
        return true;
    }

    /**
     * This method checks that End Date and Total Recurrence Number are valid when both are entered.
     *
     * @param recurrenceBeginDate
     * @param recurrenceEndDate
     * @param totalRecurrenceNumber
     * @param recurrenceIntervalCode
     * @return
     */
    protected boolean validateIfBothEndDateAndTotalRecurrenceNumberAreEntered(
            final Date recurrenceBeginDate,
            final Date recurrenceEndDate, final Integer totalRecurrenceNumber, final String recurrenceIntervalCode) {
        if (ObjectUtils.isNull(recurrenceBeginDate) ||
            ObjectUtils.isNull(recurrenceIntervalCode) ||
            ObjectUtils.isNull(recurrenceEndDate) ||
            ObjectUtils.isNull(totalRecurrenceNumber)) {
            return true;
        }
        final LocalDate beginDate = recurrenceBeginDate.toLocalDate();
        final LocalDate endDate = recurrenceEndDate.toLocalDate();
        final boolean isValid = CustomerInvoiceRecurrenceUtils.isRecurrenceTotalValid(beginDate, endDate, recurrenceIntervalCode, totalRecurrenceNumber);

        if (!isValid) {
            putFieldError(
                    ArPropertyConstants.InvoiceRecurrenceFields.INVOICE_RECURRENCE_END_DATE,
                    ArKeyConstants.ERROR_END_DATE_AND_TOTAL_NUMBER_OF_RECURRENCES_NOT_VALID);
        }

        return isValid;
    }

    /**
     * Validate that either End Date or Total Number of Recurrences must be entered.
     */
    protected boolean validateEndDateOrTotalNumberofRecurrences(final Date recurrenceEndDate, final Integer totalRecurrenceNumber) {
        if (ObjectUtils.isNull(recurrenceEndDate) && ObjectUtils.isNull(totalRecurrenceNumber)) {
            putFieldError(ArPropertyConstants.InvoiceRecurrenceFields.INVOICE_RECURRENCE_END_DATE,
                    ArKeyConstants.ERROR_END_DATE_OR_TOTAL_NUMBER_OF_RECURRENCES);
            return false;
        }
        return true;
    }

    /**
     * Total number of recurrences may not be greater than the number defined in system parameter
     * {@link ArParameterConstants#RECURRENCES}.
     */
    protected boolean validateMaximumNumberOfRecurrences(
            final Integer totalRecurrenceNumber,
            final String recurrenceIntervalCode) {
        if (ObjectUtils.isNull(recurrenceIntervalCode) || ObjectUtils.isNull(totalRecurrenceNumber)) {
            return true;
        }
        final Integer maximumRecurrencesByInterval;
        if (ObjectUtils.isNotNull(recurrenceIntervalCode)) {
            final List<String> maximumRecurrences = new ArrayList<>(SpringContext.getBean(ParameterService.class)
                    .getSubParameterValuesAsString(InvoiceRecurrence.class,
                            ArParameterConstants.RECURRENCES, recurrenceIntervalCode));
            if (maximumRecurrences.size() > 0 && StringUtils.isNotBlank(maximumRecurrences.get(0))) {
                maximumRecurrencesByInterval = Integer.valueOf(maximumRecurrences.get(0));
                if (totalRecurrenceNumber > maximumRecurrencesByInterval) {
                    putFieldError(
                            ArPropertyConstants.InvoiceRecurrenceFields.INVOICE_RECURRENCE_TOTAL_RECURRENCE_NUMBER,
                            ArKeyConstants.ERROR_TOTAL_NUMBER_OF_RECURRENCES_GREATER_THAN_ALLOWED,
                            maximumRecurrences.get(0));
                    return false;
                }
            }
        }
        return true;
    }

}
