/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.web.struts;

import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kns.util.WebUtils;
import org.kuali.kfs.kns.web.struts.action.KualiTransactionalDocumentActionBase;
import org.kuali.kfs.kns.web.struts.form.KualiDocumentFormBase;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiRuleService;
import org.kuali.kfs.module.ar.businessobject.CustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.module.ar.document.service.CustomerCreditMemoDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerCreditMemoDocumentService;
import org.kuali.kfs.module.ar.document.validation.event.ContinueCustomerCreditMemoDocumentEvent;
import org.kuali.kfs.module.ar.document.validation.event.RecalculateCustomerCreditMemoDetailEvent;
import org.kuali.kfs.module.ar.document.validation.event.RecalculateCustomerCreditMemoDocumentEvent;
import org.kuali.kfs.module.ar.report.service.AccountsReceivableReportService;
import org.kuali.kfs.module.ar.service.AccountsReceivablePdfHelperService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.nio.file.Files;
import java.util.List;

public class CustomerCreditMemoAction extends KualiTransactionalDocumentActionBase {

    public CustomerCreditMemoAction() {
        super();
    }

    /**
     * Do initialization for a new customer credit memo.
     */
    @Override
    protected void createDocument(final KualiDocumentFormBase kualiDocumentFormBase) {
        super.createDocument(kualiDocumentFormBase);
        ((CustomerCreditMemoDocument) kualiDocumentFormBase.getDocument()).initiateDocument();
    }

    /**
     * This method loads the document by its provided document header id. This has been abstracted out so that it can
     * be overridden in children if the need arises.
     *
     * @param kualiDocumentFormBase
     */
    @Override
    protected void loadDocument(final KualiDocumentFormBase kualiDocumentFormBase) {
        super.loadDocument(kualiDocumentFormBase);
        ((CustomerCreditMemoDocument) kualiDocumentFormBase.getDocument()).populateCustomerCreditMemoDetailsAfterLoad();
    }

    /**
     * Clears out init tab.
     *
     * @param mapping  An ActionMapping
     * @param form     An ActionForm
     * @param request  The HttpServletRequest
     * @param response The HttpServletResponse
     * @return An ActionForward
     * @throws Exception
     */
    public ActionForward clearInitTab(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final CustomerCreditMemoForm customerCreditMemoForm = (CustomerCreditMemoForm) form;
        final CustomerCreditMemoDocument customerCreditMemoDocument =
                (CustomerCreditMemoDocument) customerCreditMemoForm.getDocument();
        customerCreditMemoDocument.clearInitFields();

        return super.refresh(mapping, form, request, response);
    }

    /**
     * Handles continue request. This request comes from the initial screen which gives ref. invoice number.
     * Based on that, the customer credit memo is initially populated.
     *
     * @param mapping  An ActionMapping
     * @param form     An ActionForm
     * @param request  The HttpServletRequest
     * @param response The HttpServletResponse
     * @return An ActionForward
     * @throws Exception
     */
    public ActionForward continueCreditMemo(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {

        final CustomerCreditMemoForm customerCreditMemoForm = (CustomerCreditMemoForm) form;
        final CustomerCreditMemoDocument customerCreditMemoDocument =
                (CustomerCreditMemoDocument) customerCreditMemoForm.getDocument();

        final String errorPath = KFSConstants.DOCUMENT_PROPERTY_NAME;
        final boolean rulePassed = SpringContext.getBean(KualiRuleService.class)
                .applyRules(new ContinueCustomerCreditMemoDocumentEvent(errorPath, customerCreditMemoDocument));
        if (rulePassed) {
            customerCreditMemoDocument.populateCustomerCreditMemoDetails();
        }

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Based on user input this method recalculates a customer credit memo detail
     *
     * @param mapping  action mapping
     * @param form     action form
     * @param request
     * @param response
     * @return action forward
     * @throws Exception
     */
    public ActionForward recalculateCustomerCreditMemoDetail(
            final ActionMapping mapping, final ActionForm form,
            final HttpServletRequest request, final HttpServletResponse response) throws Exception {
        final CustomerCreditMemoForm customerCreditMemoForm = (CustomerCreditMemoForm) form;
        final CustomerCreditMemoDocument customerCreditMemoDocument =
                (CustomerCreditMemoDocument) customerCreditMemoForm.getDocument();

        final int indexOfLineToRecalculate = getSelectedLine(request);
        final CustomerCreditMemoDetail customerCreditMemoDetail = customerCreditMemoDocument.getCreditMemoDetails()
                .get(indexOfLineToRecalculate);

        final String errorPath = KFSConstants.DOCUMENT_PROPERTY_NAME + "." +
                                 KFSConstants.CUSTOMER_CREDIT_MEMO_DETAIL_PROPERTY_NAME + "[" + indexOfLineToRecalculate + "]";

        final boolean rulePassed = SpringContext.getBean(KualiRuleService.class)
                .applyRules(new RecalculateCustomerCreditMemoDetailEvent(errorPath, customerCreditMemoDocument,
                        customerCreditMemoDetail));
        if (rulePassed) {
            final CustomerCreditMemoDetailService customerCreditMemoDetailService =
                    SpringContext.getBean(CustomerCreditMemoDetailService.class);
            customerCreditMemoDetailService.recalculateCustomerCreditMemoDetail(customerCreditMemoDetail,
                    customerCreditMemoDocument);
        } else {
            customerCreditMemoDocument.recalculateTotals(customerCreditMemoDetail);
        }
        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * This method refreshes a customer credit memo detail
     *
     * @param mapping  action mapping
     * @param form     action form
     * @param request
     * @param response
     * @return action forward
     * @throws Exception
     */
    public ActionForward refreshCustomerCreditMemoDetail(
            final ActionMapping mapping, final ActionForm form,
            final HttpServletRequest request, final HttpServletResponse response) throws Exception {
        final CustomerCreditMemoForm customerCreditMemoDocForm = (CustomerCreditMemoForm) form;
        final CustomerCreditMemoDocument customerCreditMemoDocument =
                (CustomerCreditMemoDocument) customerCreditMemoDocForm.getDocument();
        final int indexOfLineToRefresh = getSelectedLine(request);

        final CustomerCreditMemoDetail customerCreditMemoDetail = customerCreditMemoDocument.getCreditMemoDetails()
                .get(indexOfLineToRefresh);

        customerCreditMemoDetail.setCreditMemoItemQuantity(null);
        customerCreditMemoDetail.setCreditMemoItemTotalAmount(null);
        customerCreditMemoDetail.setCreditMemoItemTaxAmount(KualiDecimal.ZERO);
        customerCreditMemoDetail.setCreditMemoLineTotalAmount(KualiDecimal.ZERO);

        customerCreditMemoDocument.recalculateTotals(customerCreditMemoDetail);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * This method refreshes customer credit memo details and line with totals
     *
     * @param mapping  action mapping
     * @param form     action form
     * @param request
     * @param response
     * @return action forward
     * @throws Exception
     */
    public ActionForward refreshCustomerCreditMemoDocument(
            final ActionMapping mapping, final ActionForm form,
            final HttpServletRequest request, final HttpServletResponse response) throws Exception {
        final CustomerCreditMemoForm customerCreditMemoDocForm = (CustomerCreditMemoForm) form;
        final CustomerCreditMemoDocument customerCreditMemoDocument =
                (CustomerCreditMemoDocument) customerCreditMemoDocForm.getDocument();

        final List<CustomerCreditMemoDetail> customerCreditMemoDetails = customerCreditMemoDocument.getCreditMemoDetails();
        for (final CustomerCreditMemoDetail customerCreditMemoDetail : customerCreditMemoDetails) {
            customerCreditMemoDetail.setCreditMemoItemQuantity(null);
            customerCreditMemoDetail.setCreditMemoItemTotalAmount(null);
            customerCreditMemoDetail.setCreditMemoItemTaxAmount(KualiDecimal.ZERO);
            customerCreditMemoDetail.setCreditMemoLineTotalAmount(KualiDecimal.ZERO);
            customerCreditMemoDetail.setDuplicateCreditMemoItemTotalAmount(null);
        }
        customerCreditMemoDocument.setCrmTotalItemAmount(KualiDecimal.ZERO);
        customerCreditMemoDocument.setCrmTotalTaxAmount(KualiDecimal.ZERO);
        customerCreditMemoDocument.setCrmTotalAmount(KualiDecimal.ZERO);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * Based on user input this method recalculates customer credit memo document <=> all customer credit memo details
     *
     * @param mapping  action mapping
     * @param form     action form
     * @param request
     * @param response
     * @return action forward
     * @throws Exception
     */
    public ActionForward recalculateCustomerCreditMemoDocument(
            final ActionMapping mapping, final ActionForm form,
            final HttpServletRequest request, final HttpServletResponse response) throws Exception {
        final CustomerCreditMemoForm customerCreditMemoForm = (CustomerCreditMemoForm) form;
        final CustomerCreditMemoDocument customerCreditMemoDocument =
                (CustomerCreditMemoDocument) customerCreditMemoForm.getDocument();

        final String errorPath = KFSConstants.DOCUMENT_PROPERTY_NAME;
        final boolean rulePassed = SpringContext.getBean(KualiRuleService.class)
                .applyRules(new RecalculateCustomerCreditMemoDocumentEvent(errorPath, customerCreditMemoDocument,
                        false));
        if (rulePassed) {
            final CustomerCreditMemoDocumentService customerCreditMemoDocumentService =
                    SpringContext.getBean(CustomerCreditMemoDocumentService.class);
            customerCreditMemoDocumentService.recalculateCustomerCreditMemoDocument(customerCreditMemoDocument,
                    false);
        }
        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    /**
     * @param mapping
     * @param form
     * @param request
     * @param response
     * @return
     * @throws Exception
     */
    public ActionForward print(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final String creditMemoDocId = request.getParameter(KFSConstants.PARAMETER_DOC_ID);
        final CustomerCreditMemoDocument customerCreditMemoDocument =
                (CustomerCreditMemoDocument) SpringContext.getBean(DocumentService.class)
                        .getByDocumentHeaderId(creditMemoDocId);

        final AccountsReceivableReportService reportService = SpringContext.getBean(AccountsReceivableReportService.class);
        final File report = reportService.generateCreditMemo(customerCreditMemoDocument);

        if (report.length() == 0) {
            return mapping.findForward(KFSConstants.MAPPING_BASIC);
        }

        final byte[] content = Files.readAllBytes(report.toPath());
        final ByteArrayOutputStream baos = SpringContext.getBean(AccountsReceivablePdfHelperService.class)
                .buildPdfOutputStream(content);

        final String fileName = customerCreditMemoDocument.getFinancialDocumentReferenceInvoiceNumber() +
                                KFSConstants.DASH + customerCreditMemoDocument.getDocumentNumber() +
                                KFSConstants.ReportGeneration.PDF_FILE_EXTENSION;
        WebUtils.saveMimeOutputStreamAsFile(response, KFSConstants.ReportGeneration.PDF_MIME_TYPE, baos, fileName);

        return null;
    }

}
