/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.report.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.coa.businessobject.Organization;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.core.web.format.CurrencyFormatter;
import org.kuali.kfs.core.web.format.PhoneNumberFormatter;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.CustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.OrganizationOptions;
import org.kuali.kfs.module.ar.businessobject.SystemInformation;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.report.service.AccountsReceivableReportService;
import org.kuali.kfs.module.ar.report.service.CustomerCreditMemoReportService;
import org.kuali.kfs.module.ar.report.service.CustomerInvoiceReportService;
import org.kuali.kfs.module.ar.report.service.OCRLineService;
import org.kuali.kfs.module.ar.report.util.CustomerCreditMemoDetailReportDataHolder;
import org.kuali.kfs.module.ar.report.util.CustomerCreditMemoReportDataHolder;
import org.kuali.kfs.module.ar.report.util.CustomerInvoiceReportDataHolder;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.Country;
import org.kuali.kfs.sys.service.LocationService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

@Transactional
public class AccountsReceivableReportServiceImpl implements AccountsReceivableReportService {

    protected DateTimeService dateTimeService;
    protected DocumentService documentService;
    protected ParameterService parameterService;
    protected CustomerAddressService customerAddressService;
    protected PhoneNumberFormatter phoneNumberFormatter = new PhoneNumberFormatter();
    protected BusinessObjectService businessObjectService;
    protected CurrencyFormatter invoiceCurrencyFormatter = new CurrencyFormatter();
    protected UniversityDateService universityDateService;
    protected CustomerInvoiceDetailService customerInvoiceDetailService;
    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected CustomerCreditMemoReportService customerCreditMemoReportService;
    protected OCRLineService ocrLineService;
    protected CustomerInvoiceReportService customerInvoiceReportService;
    private LocationService locationService;
    private PersonService personService;

    public AccountsReceivableReportServiceImpl() {
        Map<String, String> settings = invoiceCurrencyFormatter.getSettings();
        if (settings == null) {
            settings = new HashMap<>();
            invoiceCurrencyFormatter.setSettings(settings);
        }
        settings.put(CurrencyFormatter.SHOW_SYMBOL, "true");
    }

    @Override
    public File generateCreditMemo(final CustomerCreditMemoDocument creditMemo) {
        final CustomerCreditMemoReportDataHolder reportDataHolder = new CustomerCreditMemoReportDataHolder();
        final String invoiceNumber = creditMemo.getFinancialDocumentReferenceInvoiceNumber();
        final CustomerInvoiceDocument invoice = (CustomerInvoiceDocument) documentService.getByDocumentHeaderId(invoiceNumber);
        final String custID = invoice.getAccountsReceivableDocumentHeader().getCustomerNumber();

        final Map<String, String> creditMemoMap = new HashMap<>();
        creditMemoMap.put("docNumber", creditMemo.getDocumentNumber());
        creditMemoMap.put("refDocNumber", invoice.getDocumentNumber());
        final Date date = creditMemo.getDocumentHeader().getDocumentFinalDate();
        if (ObjectUtils.isNotNull(date)) {
            creditMemoMap.put("createDate", dateTimeService.toDateString(date));
        }
        reportDataHolder.setCreditmemo(creditMemoMap);
        final Map<String, String> customerMap = new HashMap<>();
        customerMap.put("id", custID);
        customerMap.put("billToName", invoice.getBillingAddressName());
        customerMap.put("billToStreetAddressLine1", invoice.getBillingLine1StreetAddress());
        customerMap.put("billToStreetAddressLine2", invoice.getBillingLine2StreetAddress());
        final String billCityStateZip;

        if (KFSConstants.COUNTRY_CODE_UNITED_STATES.equals(invoice.getBillingCountryCode())) {
            billCityStateZip = generateCityStateZipLine(invoice.getBillingCityName(),
                    invoice.getBillingStateCode(), invoice.getBillingZipCode());
        } else {
            billCityStateZip = generateCityStateZipLine(invoice.getBillingCityName(),
                    invoice.getBillingAddressInternationalProvinceName(), invoice.getBillingInternationalMailCode());
            if (StringUtils.isNotBlank(invoice.getBillingCountryCode())) {
                final Country country = locationService.getCountry(invoice.getBillingCountryCode());
                if (ObjectUtils.isNotNull(country)) {
                    customerMap.put("billToCountry", country.getName());
                }
            }
        }
        customerMap.put("billToCityStateZip", billCityStateZip);

        reportDataHolder.setCustomer(customerMap);

        final Map<String, String> invoiceMap = new HashMap<>();
        if (ObjectUtils.isNotNull(invoice.getCustomerPurchaseOrderNumber())) {
            invoiceMap.put("poNumber", invoice.getCustomerPurchaseOrderNumber());
        }
        if (invoice.getCustomerPurchaseOrderDate() != null) {
            invoiceMap.put("poDate", dateTimeService.toDateString(invoice.getCustomerPurchaseOrderDate()));
        }

        final String initiatorID = invoice.getDocumentHeader().getWorkflowDocument().getInitiatorPrincipalId();
        final Person user = personService.getPerson(initiatorID);

        if (user == null) {
            throw new RuntimeException("User '" + initiatorID + "' could not be retrieved.");
        }

        invoiceMap.put("invoicePreparer", user.getFirstName() + " " + user.getLastName());
        invoiceMap.put("headerField",
                ObjectUtils.isNull(invoice.getInvoiceHeaderText()) ? "" : invoice.getInvoiceHeaderText());
        invoiceMap.put("billingOrgName", invoice.getBilledByOrganization().getOrganizationName());
        invoiceMap.put("pretaxAmount", invoice.getInvoiceItemPreTaxAmountTotal().toString());
        final boolean salesTaxInd = parameterService.getParameterValueAsBoolean("KFS-AR", "Document",
                ArParameterConstants.SALES_TAX_IND);
        if (salesTaxInd) {
            invoiceMap.put("taxAmount", invoice.getInvoiceItemTaxAmountTotal().toString());
            final KualiDecimal taxPercentage = new KualiDecimal(6.85);
            invoiceMap.put("taxPercentage", "*** " + taxPercentage.toString() + "%");
        }
        invoiceMap.put("invoiceAmountDue", invoice.getSourceTotal().toString());
        invoiceMap.put("ocrLine", "");
        reportDataHolder.setInvoice(invoiceMap);

        final Map<String, String> sysInfoMap = new HashMap<>();

        final Organization billingOrg = invoice.getBilledByOrganization();
        final String chart = billingOrg.getChartOfAccountsCode();
        final String org = billingOrg.getOrganizationCode();
        Map<String, String> criteria = new HashMap<>();
        criteria.put("chartOfAccountsCode", chart);
        criteria.put("organizationCode", org);
        final OrganizationOptions orgOptions = businessObjectService.findByPrimaryKey(OrganizationOptions.class, criteria);

        final String fiscalYear = universityDateService.getCurrentFiscalYear().toString();
        criteria = new HashMap<>();

        final Organization processingOrg = invoice.getAccountsReceivableDocumentHeader().getProcessingOrganization();

        criteria.put("universityFiscalYear", fiscalYear);
        criteria.put("processingChartOfAccountCode", processingOrg.getChartOfAccountsCode());
        criteria.put("processingOrganizationCode", processingOrg.getOrganizationCode());
        final SystemInformation sysInfo = businessObjectService.findByPrimaryKey(SystemInformation.class, criteria);

        sysInfoMap.put("univName", StringUtils.upperCase(sysInfo.getOrganizationCheckPayableToName()));
        final String univAddr = processingOrg.getOrganizationCityName() + ", " + processingOrg.getOrganizationStateCode() +
                                " " + processingOrg.getOrganizationZipCode();
        sysInfoMap.put("univAddr", univAddr);
        if (sysInfo != null) {
            sysInfoMap.put("FEIN", "FED ID #" + sysInfo.getUniversityFederalEmployerIdentificationNumber());
        }

        reportDataHolder.setSysinfo(sysInfoMap);

        invoiceMap.put("billingOrgFax", (String) phoneNumberFormatter.format(
                phoneNumberFormatter.convertFromPresentationFormat(orgOptions.getOrganizationFaxNumber())));
        invoiceMap.put("billingOrgPhone", (String) phoneNumberFormatter.format(
                phoneNumberFormatter.convertFromPresentationFormat(orgOptions.getOrganizationPhoneNumber())));

        creditMemo.populateCustomerCreditMemoDetailsAfterLoad();
        final List<CustomerCreditMemoDetail> detailsList = creditMemo.getCreditMemoDetails();
        final List<CustomerCreditMemoDetailReportDataHolder> details = new ArrayList<>();
        CustomerCreditMemoDetailReportDataHolder detailDataHolder;
        for (final CustomerCreditMemoDetail detail : detailsList) {
            if (detail.getCreditMemoLineTotalAmount().isGreaterThan(KualiDecimal.ZERO)) {
                detailDataHolder = new CustomerCreditMemoDetailReportDataHolder(detail,
                        detail.getCustomerInvoiceDetail());
                details.add(detailDataHolder);
            }
        }

        reportDataHolder.setDetails(details);

        final Date runDate = dateTimeService.getCurrentSqlDate();
        return customerCreditMemoReportService.generateReport(reportDataHolder, runDate);
    }

    @Override
    public File generateInvoice(final CustomerInvoiceDocument invoice) {
        final CustomerInvoiceReportDataHolder reportDataHolder = new CustomerInvoiceReportDataHolder();
        final String custID = invoice.getAccountsReceivableDocumentHeader().getCustomerNumber();
        final Integer customerBillToAddressIdentifier = invoice.getCustomerBillToAddressIdentifier();
        final Integer customerShipToAddressIdentifier = invoice.getCustomerShipToAddressIdentifier();
        final CustomerAddress billToAddr = customerAddressService.getByPrimaryKey(custID, customerBillToAddressIdentifier);
        final CustomerAddress shipToAddr = customerAddressService.getByPrimaryKey(custID, customerShipToAddressIdentifier);

        final Organization billingOrg = invoice.getBilledByOrganization();
        final String chart = billingOrg.getChartOfAccountsCode();
        final String org = billingOrg.getOrganizationCode();
        Map<String, String> criteria = new HashMap<>();
        criteria.put("chartOfAccountsCode", chart);
        criteria.put("organizationCode", org);
        final OrganizationOptions orgOptions = businessObjectService.findByPrimaryKey(OrganizationOptions.class, criteria);

        final Map<String, String> customerMap = new HashMap<>();
        customerMap.put("id", custID);
        if (billToAddr != null) {
            customerMap.put("billToName", billToAddr.getCustomerAddressName());
            customerMap.put("billToStreetAddressLine1", billToAddr.getCustomerLine1StreetAddress());
            customerMap.put("billToStreetAddressLine2", billToAddr.getCustomerLine2StreetAddress());
            final String billCityStateZip;
            String billToCountry = "";
            if (billToAddr.getCustomerCountry() != null) {
                billToCountry = billToAddr.getCustomerCountry().getName().toUpperCase(Locale.US);
                /* report can only display max of 32 characters for country name */
                if (billToCountry.length() > 32) {
                    billToCountry = billToCountry.substring(0, 31);
                }
            }
            if (billToAddr.getCustomerCountryCode().equals("US")) {
                billCityStateZip = generateCityStateZipLine(billToAddr.getCustomerCityName(),
                        billToAddr.getCustomerStateCode(), billToAddr.getCustomerZipCode());
                if (StringUtils.isNotBlank(orgOptions.getOrganizationRemitToCountryCode())
                        && !"US".equals(orgOptions.getOrganizationRemitToCountryCode())) {
                    customerMap.put("billToCountry", billToCountry);
                }
            } else {
                billCityStateZip = generateCityStateZipLine(billToAddr.getCustomerCityName(),
                        billToAddr.getCustomerAddressInternationalProvinceName(),
                        billToAddr.getCustomerInternationalMailCode());
                customerMap.put("billToCountry", billToCountry);
            }
            customerMap.put("billToCityStateZip", billCityStateZip);
        }
        if (shipToAddr != null) {
            customerMap.put("shipToName", shipToAddr.getCustomerAddressName());
            customerMap.put("shipToStreetAddressLine1", shipToAddr.getCustomerLine1StreetAddress());
            customerMap.put("shipToStreetAddressLine2", shipToAddr.getCustomerLine2StreetAddress());
            final String shipCityStateZip;
            if (shipToAddr.getCustomerCountryCode().equals("US")) {
                shipCityStateZip = generateCityStateZipLine(shipToAddr.getCustomerCityName(),
                        shipToAddr.getCustomerStateCode(), shipToAddr.getCustomerZipCode());
            } else {
                shipCityStateZip = generateCityStateZipLine(shipToAddr.getCustomerCityName(),
                        shipToAddr.getCustomerAddressInternationalProvinceName(),
                        shipToAddr.getCustomerInternationalMailCode());
                customerMap.put("shipToCountry", shipToAddr.getCustomerCountry().getName().toUpperCase(Locale.US));
            }
            customerMap.put("shipToCityStateZip", shipCityStateZip);
        }
        reportDataHolder.setCustomer(customerMap);

        final Map<String, String> invoiceMap = new HashMap<>();
        invoiceMap.put("poNumber", invoice.getCustomerPurchaseOrderNumber());
        if (invoice.getCustomerPurchaseOrderDate() != null) {
            invoiceMap.put("poDate", dateTimeService.toDateString(invoice.getCustomerPurchaseOrderDate()));
        }

        final String initiatorID = invoice.getDocumentHeader().getWorkflowDocument().getInitiatorPrincipalId();
        final Person user = personService.getPerson(initiatorID);

        if (user == null) {
            throw new RuntimeException("User '" + initiatorID + "' could not be retrieved.");
        }

        invoiceMap.put("invoicePreparer", user.getFirstName() + " " + user.getLastName());
        invoiceMap.put("headerField", invoice.getInvoiceHeaderText());
        invoiceMap.put("customerOrg", invoice.getBilledByOrganizationCode());
        invoiceMap.put("docNumber", invoice.getDocumentNumber());
        invoiceMap.put("invoiceDueDate", dateTimeService.toDateString(invoice.getInvoiceDueDate()));
        invoiceMap.put("createDate", dateTimeService.toDateString(invoice.getDocumentHeader().getWorkflowDocument()
                .getDateCreated().toDate()));
        invoiceMap.put("invoiceAttentionLineText", StringUtils.upperCase(invoice.getInvoiceAttentionLineText()));
        invoiceMap.put("billingOrgName", invoice.getBilledByOrganization().getOrganizationName());
        invoiceMap.put("pretaxAmount", invoiceCurrencyFormatter.format(invoice.getInvoiceItemPreTaxAmountTotal())
                .toString());
        final boolean salesTaxInd = parameterService.getParameterValueAsBoolean("KFS-AR", "Document",
                ArParameterConstants.SALES_TAX_IND);
        if (salesTaxInd) {
            invoiceMap.put("taxAmount", invoiceCurrencyFormatter.format(invoice.getInvoiceItemTaxAmountTotal())
                    .toString());
            // suppressing this as its useless
            invoiceMap.put("taxPercentage", "");
        }
        invoiceMap.put("invoiceAmountDue", invoiceCurrencyFormatter.format(invoice.getSourceTotal()).toString());
        invoiceMap.put("invoiceTermsText", invoice.getInvoiceTermsText());

        final String ocrLine = ocrLineService.generateOCRLine(invoice.getSourceTotal(), custID, invoice.getDocumentNumber());
        invoiceMap.put("ocrLine", ocrLine);
        final List<CustomerInvoiceDetail> detailsList = (List<CustomerInvoiceDetail>) customerInvoiceDetailService
                .getCustomerInvoiceDetailsForInvoice(invoice);
        final CustomerInvoiceDetail firstDetail = detailsList.get(0);
        final String firstChartCode = firstDetail.getChartOfAccountsCode();
        final String firstAccount = firstDetail.getAccountNumber();
        invoiceMap.put("chartAndAccountOfFirstItem", firstChartCode + firstAccount);

        final Map<String, String> sysinfoMap = new HashMap<>();

        final String fiscalYear = universityDateService.getCurrentFiscalYear().toString();
        criteria = new HashMap<>();

        final Organization processingOrg = invoice.getAccountsReceivableDocumentHeader().getProcessingOrganization();

        criteria.put("universityFiscalYear", fiscalYear);
        criteria.put("processingChartOfAccountCode", processingOrg.getChartOfAccountsCode());
        criteria.put("processingOrganizationCode", processingOrg.getOrganizationCode());
        final SystemInformation sysinfo = businessObjectService.findByPrimaryKey(SystemInformation.class, criteria);

        sysinfoMap.put("univName", StringUtils.upperCase(sysinfo.getOrganizationCheckPayableToName()));
        sysinfoMap.put("univAddr", generateCityStateZipLine(processingOrg.getOrganizationCityName(),
                processingOrg.getOrganizationStateCode(), processingOrg.getOrganizationZipCode()));
        if (sysinfo != null) {
            sysinfoMap.put("FEIN", "FED ID #" + sysinfo.getUniversityFederalEmployerIdentificationNumber());
        }
        sysinfoMap.put("checkPayableTo", orgOptions.getOrganizationCheckPayableToName());
        sysinfoMap.put("remitToName", orgOptions.getOrganizationRemitToAddressName());
        sysinfoMap.put("remitToAddressLine1", orgOptions.getOrganizationRemitToLine1StreetAddress());
        sysinfoMap.put("remitToAddressLine2", orgOptions.getOrganizationRemitToLine2StreetAddress());

        final String remitToCityStateZip = generateCityStateZipLine(orgOptions.getOrganizationRemitToCityName(),
                orgOptions.getOrganizationRemitToStateCode(), orgOptions.getOrganizationRemitToZipCode());
        if (billToAddr != null && !"US".equals(billToAddr.getCustomerCountryCode())
            && orgOptions.getOrgRemitToCountry() != null
            || StringUtils.isNotBlank(orgOptions.getOrganizationRemitToCountryCode())
            && !"US".equals(orgOptions.getOrganizationRemitToCountryCode())) {
            String remitToCountry = orgOptions.getOrgRemitToCountry().getName().toUpperCase(Locale.US);
            /* report can only display max of 32 characters for country name */
            if (remitToCountry.length() > 32) {
                remitToCountry = remitToCountry.substring(0, 31);
            }
            sysinfoMap.put("remitToCountry", remitToCountry);
        }
        sysinfoMap.put("remitToCityStateZip", remitToCityStateZip);

        invoiceMap.put("billingOrgFax", (String) phoneNumberFormatter.format(
                phoneNumberFormatter.convertFromPresentationFormat(orgOptions.getOrganizationFaxNumber())));
        invoiceMap.put("billingOrgPhone", (String) phoneNumberFormatter.format(
                phoneNumberFormatter.convertFromPresentationFormat(orgOptions.getOrganizationPhoneNumber())));

        invoiceMap.put("orgOptionsMessageText", orgOptions.getOrganizationMessageText());

        reportDataHolder.setSysinfo(sysinfoMap);
        reportDataHolder.setDetails(detailsList);
        reportDataHolder.setInvoice(invoiceMap);

        final Date runDate = dateTimeService.getCurrentSqlDate();
        final File report = customerInvoiceReportService.generateReport(reportDataHolder, runDate);

        invoice.setPrintDate(runDate);
        documentService.updateDocument(invoice);
        return report;
    }

    @Override
    public List<File> generateInvoicesByBillingOrg(final String chartCode, final String orgCode, final Date date) {
        final CustomerInvoiceDocumentService invoiceDocService = customerInvoiceDocumentService;
        final List<CustomerInvoiceDocument> invoices = invoiceDocService
                .getPrintableCustomerInvoiceDocumentsByBillingChartAndOrg(chartCode, orgCode);
        final List<File> reports = new ArrayList<>();

        for (final CustomerInvoiceDocument doc : invoices) {
            if (date == null) {
                reports.add(generateInvoice(doc));
            } else if (dateTimeService.toDateString(doc.getDocumentHeader().getWorkflowDocument().getDateCreated()
                    .toDate()) != null) {
                if (dateTimeService.toDateString(doc.getDocumentHeader().getWorkflowDocument().getDateCreated()
                        .toDate()).equals(dateTimeService.toDateString(date))) {
                    reports.add(generateInvoice(doc));
                }
            }
        }
        return reports;
    }

    @Override
    public List<File> generateInvoicesByProcessingOrg(final String chartCode, final String orgCode, final Date date) {
        final List<CustomerInvoiceDocument> invoices = customerInvoiceDocumentService
                .getPrintableCustomerInvoiceDocumentsByProcessingChartAndOrg(chartCode, orgCode);

        final List<File> reports = new ArrayList<>();
        for (final CustomerInvoiceDocument doc : invoices) {
            if (date == null) {
                reports.add(generateInvoice(doc));
            } else if (dateTimeService.toDateString(doc.getDocumentHeader().getWorkflowDocument().getDateCreated()
                    .toDate()) != null) {
                if (dateTimeService.toDateString(doc.getDocumentHeader().getWorkflowDocument().getDateCreated()
                        .toDate()).equals(dateTimeService.toDateString(date))) {
                    reports.add(generateInvoice(doc));
                }
            }
        }
        return reports;
    }

    @Override
    public List<File> generateInvoicesByInitiator(final String initiator, final java.sql.Date date) {
        final List<File> reports = new ArrayList<>();
        final Collection<CustomerInvoiceDocument> invoices = customerInvoiceDocumentService
                .getPrintableCustomerInvoiceDocumentsByInitiatorPrincipalName(initiator);
        for (final CustomerInvoiceDocument invoice : invoices) {
            if (date == null) {
                reports.add(generateInvoice(invoice));
            } else if (dateTimeService.toDateString(invoice.getDocumentHeader().getWorkflowDocument().getDateCreated()
                    .toDate()) != null) {
                if (dateTimeService.toDateString(invoice.getDocumentHeader().getWorkflowDocument().getDateCreated()
                        .toDate()).equals(dateTimeService.toDateString(date))) {
                    reports.add(generateInvoice(invoice));
                }
            }
        }
        return reports;
    }

    /**
     * @param city
     * @param state
     * @param zipCode
     * @return
     */
    protected String generateCityStateZipLine(final String city, final String state, final String zipCode) {
        final StringBuffer cityStateZip = new StringBuffer(city);

        if (StringUtils.isNotBlank(state)) {
            cityStateZip.append(", ").append(state);
        }

        if (StringUtils.isNotBlank(zipCode)) {
            cityStateZip.append("  ").append(zipCode);
        }

        return cityStateZip.toString();
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public CustomerAddressService getCustomerAddressService() {
        return customerAddressService;
    }

    public void setCustomerAddressService(final CustomerAddressService customerAddressService) {
        this.customerAddressService = customerAddressService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public CustomerInvoiceDetailService getCustomerInvoiceDetailService() {
        return customerInvoiceDetailService;
    }

    public void setCustomerInvoiceDetailService(final CustomerInvoiceDetailService customerInvoiceDetailService) {
        this.customerInvoiceDetailService = customerInvoiceDetailService;
    }

    public CustomerInvoiceDocumentService getCustomerInvoiceDocumentService() {
        return customerInvoiceDocumentService;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    public CustomerCreditMemoReportService getCustomerCreditMemoReportService() {
        return customerCreditMemoReportService;
    }

    public void setCustomerCreditMemoReportService(final CustomerCreditMemoReportService customerCreditMemoReportService) {
        this.customerCreditMemoReportService = customerCreditMemoReportService;
    }

    public OCRLineService getOcrLineService() {
        return ocrLineService;
    }

    public void setOcrLineService(final OCRLineService ocrLineService) {
        this.ocrLineService = ocrLineService;
    }

    public CustomerInvoiceReportService getCustomerInvoiceReportService() {
        return customerInvoiceReportService;
    }

    public void setCustomerInvoiceReportService(final CustomerInvoiceReportService customerInvoiceReportService) {
        this.customerInvoiceReportService = customerInvoiceReportService;
    }

    public void setLocationService(final LocationService locationService) {
        this.locationService = locationService;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }
}
