/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.report.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleBillingService;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.UserSession;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.LookupService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsInvoiceDocumentService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsAgingReportService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsReportHelperService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.springframework.transaction.annotation.Transactional;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This class is used to get the services for PDF generation and other services for CG Aging report.
 */
public class ContractsGrantsAgingReportServiceImpl implements ContractsGrantsAgingReportService {

    protected ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService;
    protected BusinessObjectService businessObjectService;
    protected PersonService personService;
    protected ContractsGrantsReportHelperService contractsGrantsReportHelperService;
    protected ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService;
    protected DateTimeService dateTimeService;
    protected LookupService lookupService;

    @Override
    @Transactional
    public Map<String, List<ContractsGrantsInvoiceDocument>> filterContractsGrantsAgingReport(
            final Map fieldValues,
            final java.sql.Date begin, final java.sql.Date end) throws ParseException {
        final Collection<ContractsGrantsInvoiceDocument> contractsGrantsInvoiceDocs =
                retrieveMatchingContractsGrantsInvoiceDocuments(fieldValues, begin, end);
        return generateMapFromContractsGrantsInvoiceDocuments(contractsGrantsInvoiceDocs);
    }

    /**
     * Utility method to lookup matching contracts & grants invoice documents for the given field values
     *
     * @param fieldValues the field values with criteria to lookup report on
     * @param begin       the begin da
     * @param end
     * @return
     */
    protected List<ContractsGrantsInvoiceDocument> retrieveMatchingContractsGrantsInvoiceDocuments(
            final Map fieldValues,
            final java.sql.Date begin, final java.sql.Date end) {
        final String reportOption = (String) fieldValues.get(ArPropertyConstants.REPORT_OPTION);
        final String orgCode = (String) fieldValues.get(KFSPropertyConstants.ORGANIZATION_CODE);
        final String chartCode = (String) fieldValues.get(ArPropertyConstants.PROCESSING_OR_BILLING_CHART_CODE);
        final String customerNumber = (String) fieldValues.get(KFSPropertyConstants.CUSTOMER_NUMBER);
        final String customerName = (String) fieldValues.get(KFSPropertyConstants.CUSTOMER_NAME);
        final String accountNumber = (String) fieldValues.get(KFSPropertyConstants.ACCOUNT_NUMBER);
        final String accountChartOfAccountsCode = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.ACCOUNT_CHART_CODE);
        final String fundManager = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.FUND_MANAGER);
        final String proposalNumber = (String) fieldValues.get(KFSPropertyConstants.PROPOSAL_NUMBER);
        final String collectorPrincName = (String) fieldValues.get(ArPropertyConstants.COLLECTOR_PRINC_NAME);
        String collectorPrincipalId = null;

        final List<ContractsGrantsInvoiceDocument> contractsGrantsInvoiceDocs = new ArrayList<>();

        if (StringUtils.isNotBlank(collectorPrincName)) {
            final Person collUser = personService.getPersonByPrincipalName(collectorPrincName);
            if (ObjectUtils.isNull(collUser)) {
                // if the principal name is not a real user, then return no values
                return contractsGrantsInvoiceDocs;
            } else {
                collectorPrincipalId = collUser.getPrincipalId();
            }
        }

        if (StringUtils.isNotBlank(fundManager)) {
            final Person fundManagerUser = getPersonService().getPersonByPrincipalName(fundManager);
            if (ObjectUtils.isNull(fundManagerUser)) {
                // the fund manager doesn't exist, so empty results
                return contractsGrantsInvoiceDocs;
            }
        }

        final String awardDocumentNumber = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.AWARD_DOCUMENT_NUMBER);
        final String markedAsFinal = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.MARKED_AS_FINAL);

        final String invoiceNumber = (String) fieldValues.get(ArPropertyConstants.INVOICE_NUMBER);

        final String invoiceDateFromString = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_FROM);
        final String invoiceDateToString = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.INVOICE_DATE_TO);
        final String invoiceDateCriteria = getContractsGrantsReportHelperService().fixDateCriteria(invoiceDateFromString, invoiceDateToString, true);

        final String responsibilityId = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.CG_ACCT_RESP_ID);

        final String awardEndFromDate = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.AWARD_END_DATE_FROM);
        final String awardEndToDate = (String) fieldValues.get(ArPropertyConstants.ContractsGrantsAgingReportFields.AWARD_END_DATE_TO);

        final Map<String, String> fieldValuesForInvoice = new HashMap<>();
        fieldValuesForInvoice.put(ArPropertyConstants.OPEN_INVOICE_IND, KFSConstants.Booleans.TRUE);

        fieldValuesForInvoice.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME,
                ArConstants.ArDocumentTypeCodes.CONTRACTS_GRANTS_INVOICE);

        //Now to involve reportOption and handle chart and org
        if (ObjectUtils.isNotNull(reportOption)) {
            if (reportOption.equalsIgnoreCase(ArConstants.ReportOptionFieldValues.PROCESSING_ORG)
                    && StringUtils.isNotBlank(chartCode) && StringUtils.isNotBlank(orgCode)) {
                fieldValuesForInvoice.put(ArPropertyConstants.CustomerInvoiceDocumentFields.PROCESSING_ORGANIZATION_CODE, orgCode);
                fieldValuesForInvoice.put(ArPropertyConstants.CustomerInvoiceDocumentFields.PROCESSING_CHART_OF_ACCOUNT_CODE, chartCode);
            }
            if (reportOption.equalsIgnoreCase(ArConstants.ReportOptionFieldValues.BILLING_ORG)
                    && StringUtils.isNotBlank(chartCode) && StringUtils.isNotBlank(orgCode)) {
                fieldValuesForInvoice.put(ArPropertyConstants.CustomerInvoiceDocumentFields.BILLED_BY_ORGANIZATION_CODE, orgCode);
                fieldValuesForInvoice.put(ArPropertyConstants.CustomerInvoiceDocumentFields.BILL_BY_CHART_OF_ACCOUNT_CODE, chartCode);
            }
        }
        if (StringUtils.isNotBlank(customerNumber)) {
            fieldValuesForInvoice.put(ArPropertyConstants.CustomerInvoiceDocumentFields.CUSTOMER_NUMBER, customerNumber);
        }
        if (StringUtils.isNotBlank(customerName)) {
            fieldValuesForInvoice.put(ArPropertyConstants.CUSTOMER_NAME, customerName);
        }
        if (StringUtils.isNotBlank(proposalNumber)) {
            fieldValuesForInvoice.put(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields.PROPOSAL_NUMBER, proposalNumber);
        }
        if (StringUtils.isNotBlank(markedAsFinal)) {
            if (markedAsFinal.equalsIgnoreCase(KFSConstants.ParameterValues.YES)) {
                fieldValuesForInvoice.put(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields.FINAL_BILL, KFSConstants.Booleans.TRUE);
            } else if (markedAsFinal.equalsIgnoreCase(KFSConstants.ParameterValues.NO)) {
                fieldValuesForInvoice.put(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields.FINAL_BILL, KFSConstants.Booleans.FALSE);
            }
        }
        if (StringUtils.isNotBlank(invoiceNumber)) {
            fieldValuesForInvoice.put(KFSPropertyConstants.DOCUMENT_NUMBER, invoiceNumber);
        }
        if (StringUtils.isNotBlank(responsibilityId)) {
            fieldValuesForInvoice.put(ArPropertyConstants.CustomerInvoiceDocumentFields.CG_ACCT_RESP_ID, responsibilityId);
        }

        if (StringUtils.isNotBlank(accountChartOfAccountsCode)) {
            fieldValuesForInvoice.put(KFSPropertyConstants.SOURCE_ACCOUNTING_LINES + "." + KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, accountChartOfAccountsCode);
        }
        if (StringUtils.isNotBlank(accountNumber)) {
            fieldValuesForInvoice.put(KFSPropertyConstants.SOURCE_ACCOUNTING_LINES + "." + KFSPropertyConstants.ACCOUNT_NUMBER, accountNumber);
        }

        if (StringUtils.isNotBlank(invoiceDateCriteria)) {
            fieldValuesForInvoice.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_CREATE_DATE, invoiceDateCriteria);
        }

        String billingBeginDateString = null;
        if (begin != null) {
            billingBeginDateString = getDateTimeService().toDateString(begin);
        }
        String billingEndDateString = null;
        if (end != null) {
            billingEndDateString = getDateTimeService().toDateString(end);
        }
        final String billingDateCriteria = getContractsGrantsReportHelperService().fixDateCriteria(billingBeginDateString, billingEndDateString, false);
        if (StringUtils.isNotBlank(billingDateCriteria)) {
            fieldValuesForInvoice.put(ArPropertyConstants.CustomerInvoiceDocumentFields.BILLING_DATE, billingDateCriteria);
        }

        final Set<String> awardIds = lookupBillingAwards(awardDocumentNumber, awardEndFromDate, awardEndToDate, fundManager);

        // here put all criterion and find the docs
        contractsGrantsInvoiceDocs.addAll(getLookupService().findCollectionBySearch(ContractsGrantsInvoiceDocument.class, fieldValuesForInvoice));

        filterContractsGrantsInvoiceDocumentsByAwardAndCollector(contractsGrantsInvoiceDocs, collectorPrincipalId, awardIds);

        return contractsGrantsInvoiceDocs;
    }

    /**
     * Removes contracts & grants invoice documents from the given collection if they do not match the given award
     * ids or if the collector permissions do not allow viewing of the document
     *
     * @param contractsGrantsInvoiceDocs a Collection of ContractsGrantsInvoiceDocuments
     * @param collectorPrincipalId       the principal name of the collector
     * @param awardIds                   a Set of proposal numbers of awards which match given criteria
     */
    protected void filterContractsGrantsInvoiceDocumentsByAwardAndCollector(
            final Collection<ContractsGrantsInvoiceDocument> contractsGrantsInvoiceDocs, final String collectorPrincipalId,
            final Set<String> awardIds) {
        // filter by collector and user performing the search
        final Person user = getUserSession().getPerson();

        if (!CollectionUtils.isEmpty(contractsGrantsInvoiceDocs)) {
            for (final Iterator<ContractsGrantsInvoiceDocument> iter = contractsGrantsInvoiceDocs.iterator(); iter.hasNext(); ) {
                final ContractsGrantsInvoiceDocument document = iter.next();
                if (ObjectUtils.isNotNull(document.getInvoiceGeneralDetail())
                        && ObjectUtils.isNotNull(document.getInvoiceGeneralDetail().getAward())
                        && awardIds != null
                        && !awardIds.contains(document.getInvoiceGeneralDetail().getAward().getProposalNumber())) {
                    iter.remove();
                } else if (StringUtils.isNotEmpty(collectorPrincipalId)) {
                    if (!contractsGrantsInvoiceDocumentService.canViewInvoice(document, collectorPrincipalId)) {
                        iter.remove();
                    }
                } else if (!contractsGrantsInvoiceDocumentService.canViewInvoice(document, user.getPrincipalId())) {
                    iter.remove();
                }
            }
        }
    }

    /**
     * Generates a Map keyed by customer number and name from a Collection of ContractsGrantsInvoiceDocuments
     *
     * @param contractsGrantsInvoiceDocs a Collection of ContractsGrantsInvoiceDocuments to convert into a Map
     * @return a Map of CINV docs, keyed by customer number and name
     */
    protected Map<String, List<ContractsGrantsInvoiceDocument>> generateMapFromContractsGrantsInvoiceDocuments(
            final Collection<ContractsGrantsInvoiceDocument> contractsGrantsInvoiceDocs) {
        Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer = null;
        if (!CollectionUtils.isEmpty(contractsGrantsInvoiceDocs)) {
            cgMapByCustomer = new HashMap<>();
            for (final ContractsGrantsInvoiceDocument cgDoc : contractsGrantsInvoiceDocs) {
                final List<ContractsGrantsInvoiceDocument> cgInvoiceDocs;
                final String customerNbr = cgDoc.getCustomer().getCustomerNumber();
                final String customerNm = cgDoc.getCustomer().getCustomerName();
                final String key = customerNbr + "-" + customerNm;
                if (cgMapByCustomer.containsKey(key)) {
                    cgInvoiceDocs = cgMapByCustomer.get(key);
                } else {
                    cgInvoiceDocs = new ArrayList<>();
                }
                cgInvoiceDocs.add(cgDoc);
                cgMapByCustomer.put(key, cgInvoiceDocs);
            }
        }
        return cgMapByCustomer;
    }

    /**
     * Generates a Set of proposal ids for awards which match the given criteria
     *
     * @param awardDocumentNumber the document number of the award
     * @param awardEndFromDate    the award ending date of the award
     * @param awardEndToDate      the award ending date of the award
     * @param fundManager         the principal name of the fund manager
     * @return a Set of Award ids to filter on, or null if no search was actually completed
     */
    protected Set<String> lookupBillingAwards(
            final String awardDocumentNumber, final String awardEndFromDate,
            final String awardEndToDate, final String fundManager) {
        if (StringUtils.isBlank(awardDocumentNumber)
                && StringUtils.isBlank(awardEndFromDate)
                && StringUtils.isBlank(awardEndToDate)
                && StringUtils.isBlank(fundManager)) {
            // nothing to search on? then return null to note that no search was completed
            return null;
        }

        final Set<String> fundManagerIds = getContractsGrantsReportHelperService().lookupPrincipalIds(fundManager);

        final Map<String, String> fieldValues = new HashMap<>();
        if (StringUtils.isNotBlank(awardDocumentNumber)) {
            fieldValues.put(KFSPropertyConstants.AWARD_DOCUMENT_NUMBER, awardDocumentNumber);
        }

        final String awardEnd = getContractsGrantsReportHelperService().fixDateCriteria(awardEndFromDate, awardEndToDate, false);
        if (StringUtils.isNotBlank(awardEnd)) {
            fieldValues.put(KFSPropertyConstants.AWARD_ENDING_DATE, awardEnd);
        }
        fieldValues.put(KFSPropertyConstants.ACTIVE, KFSConstants.ACTIVE_INDICATOR);

        final List<? extends ContractsAndGrantsAward> awards = getContractsAndGrantsModuleBillingService().lookupAwards(fieldValues, true);

        final Set<String> billingAwardIds = new HashSet<>();
        for (final ContractsAndGrantsAward award : awards) {
            if (award instanceof ContractsAndGrantsBillingAward) {
                final ContractsAndGrantsBillingAward cgbAward = (ContractsAndGrantsBillingAward) award;
                if (ObjectUtils.isNull(cgbAward.getAwardPrimaryFundManager()) || fundManagerIds.isEmpty()
                        || fundManagerIds.contains(cgbAward.getAwardPrimaryFundManager().getPrincipalId())) {
                    billingAwardIds.add(cgbAward.getProposalNumber());
                }
            }
        }
        return billingAwardIds;
    }

    /**
     * Figures out the reportRunDate and then uses filterContractsGrantsAgingReport to look up the right documents
     * @param fieldValues
     */
    @Override
    public List<ContractsGrantsInvoiceDocument> lookupContractsGrantsInvoiceDocumentsForAging(
            final Map<String, String> fieldValues) {
        try {
            final java.util.Date today = getDateTimeService().getCurrentDate();
            final String reportRunDateStr = fieldValues.get(ArPropertyConstants.CustomerAgingReportFields.REPORT_RUN_DATE);

            final java.util.Date reportRunDate = ObjectUtils.isNull(reportRunDateStr) || reportRunDateStr.isEmpty() ?
                today :
                getDateTimeService().convertToDate(reportRunDateStr);

            // retrieve filtered data according to the lookup
            return retrieveMatchingContractsGrantsInvoiceDocuments(fieldValues, null,
                    new java.sql.Date(reportRunDate.getTime()));

        } catch (final ParseException ex) {
            throw new RuntimeException("Could not parse report run date for lookup", ex);
        }
    }

    public PersonService getPersonService() {
        return personService;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }

    public ContractsGrantsReportHelperService getContractsGrantsReportHelperService() {
        return contractsGrantsReportHelperService;
    }

    public void setContractsGrantsReportHelperService(final ContractsGrantsReportHelperService contractsGrantsReportHelperService) {
        this.contractsGrantsReportHelperService = contractsGrantsReportHelperService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public ContractsAndGrantsModuleBillingService getContractsAndGrantsModuleBillingService() {
        return contractsAndGrantsModuleBillingService;
    }

    public void setContractsAndGrantsModuleBillingService(final ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService) {
        this.contractsAndGrantsModuleBillingService = contractsAndGrantsModuleBillingService;
    }

    public LookupService getLookupService() {
        return lookupService;
    }

    public void setLookupService(final LookupService lookupService) {
        this.lookupService = lookupService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public ContractsGrantsInvoiceDocumentService getContractsGrantsInvoiceDocumentService() {
        return contractsGrantsInvoiceDocumentService;
    }

    public void setContractsGrantsInvoiceDocumentService(final ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService) {
        this.contractsGrantsInvoiceDocumentService = contractsGrantsInvoiceDocumentService;
    }

    /*
     * Wrapping static utility class in a method so tests can use a spy to mock this call; this way,
     * static mocking is not necessary.
     */
    UserSession getUserSession() {
        return GlobalVariables.getUserSession();
    }

}
