/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.report.service.impl;

import net.sf.jasperreports.engine.JRParameter;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.report.service.CustomerInvoiceReportService;
import org.kuali.kfs.module.ar.report.util.CustomerInvoiceReportDataHolder;
import org.kuali.kfs.sys.KFSConstants.ReportGeneration;
import org.kuali.kfs.sys.report.ReportInfo;
import org.kuali.kfs.sys.service.ReportGenerationService;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.util.Date;
import java.util.Map;
import java.util.ResourceBundle;

/**
 * To generate the working progress reports for the effort certification
 */
@Transactional
public class CustomerInvoiceReportServiceImpl implements CustomerInvoiceReportService {
    private ReportGenerationService reportGenerationService;
    private ParameterService parameterService;
    private ReportInfo customerInvoiceReportInfo;
    private ReportInfo customerInvoiceReportInfoNoTax;

    @Override
    public File generateReport(final CustomerInvoiceReportDataHolder reportDataHolder, final Date runDate) {
        final String reportFileName;
        final String reportDirectory;
        final String reportTemplateClassPath;
        final String reportTemplateName;
        final ResourceBundle resourceBundle;
        final String subReportTemplateClassPath;
        final Map<String, String> subReports;

        if (parameterService.getParameterValueAsBoolean("KFS-AR", "Document", ArParameterConstants.SALES_TAX_IND)) {
            reportFileName = customerInvoiceReportInfo.getReportFileName();
            reportDirectory = customerInvoiceReportInfo.getReportsDirectory();
            reportTemplateClassPath = customerInvoiceReportInfo.getReportTemplateClassPath();
            reportTemplateName = customerInvoiceReportInfo.getReportTemplateName();
            resourceBundle = customerInvoiceReportInfo.getResourceBundle();
            subReportTemplateClassPath = customerInvoiceReportInfo.getSubReportTemplateClassPath();
            subReports = customerInvoiceReportInfo.getSubReports();
        } else {
            // no sales tax
            reportFileName = customerInvoiceReportInfoNoTax.getReportFileName();
            reportDirectory = customerInvoiceReportInfoNoTax.getReportsDirectory();
            reportTemplateClassPath = customerInvoiceReportInfoNoTax.getReportTemplateClassPath();
            reportTemplateName = customerInvoiceReportInfoNoTax.getReportTemplateName();
            resourceBundle = customerInvoiceReportInfoNoTax.getResourceBundle();
            subReportTemplateClassPath = customerInvoiceReportInfoNoTax.getSubReportTemplateClassPath();
            subReports = customerInvoiceReportInfoNoTax.getSubReports();

        }
        final Map<String, Object> reportData = reportDataHolder.getReportData();
        reportData.put(JRParameter.REPORT_RESOURCE_BUNDLE, resourceBundle);
        reportData.put(ReportGeneration.PARAMETER_NAME_SUBREPORT_DIR, subReportTemplateClassPath);
        reportData.put(ReportGeneration.PARAMETER_NAME_SUBREPORT_TEMPLATE_NAME, subReports);
        final String template = reportTemplateClassPath + reportTemplateName;
        final String fullReportFileName = reportGenerationService.buildFullFileName(runDate, reportDirectory, reportFileName,
                "");
        reportGenerationService.generateReportToPdfFile(reportData, template, fullReportFileName);
        return new File(fullReportFileName + ".pdf");
    }

    public void setCustomerInvoiceReportInfo(final ReportInfo customerInvoiceReportInfo) {
        this.customerInvoiceReportInfo = customerInvoiceReportInfo;
    }

    public void setReportGenerationService(final ReportGenerationService reportGenerationService) {
        this.reportGenerationService = reportGenerationService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public ReportInfo getCustomerInvoiceReportInfo() {
        return customerInvoiceReportInfo;
    }

    public ReportInfo getCustomerInvoiceReportInfoNoTax() {
        return customerInvoiceReportInfoNoTax;
    }

    public void setCustomerInvoiceReportInfoNoTax(final ReportInfo customerInvoiceReportInfoNoTax) {
        this.customerInvoiceReportInfoNoTax = customerInvoiceReportInfoNoTax;
    }

}
