/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import org.apache.commons.beanutils.converters.BooleanConverter;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.util.KNSGlobalVariables;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.util.ErrorMessage;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.Milestone;
import org.kuali.kfs.module.ar.businessobject.MilestoneSchedule;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.springframework.util.AutoPopulatingList;

import java.sql.Date;
import java.text.ParseException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class MilestoneScheduleCsvInputFileType extends ScheduleCsvInputFileType {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    protected Object convertParsedObjectToVO(Object parsedContent) {
        List<MilestoneSchedule> milestoneSchedules = new LinkedList<>();
        List<Map<String, String>> parseDataList = (List<Map<String, String>>) parsedContent;
        for (Map<String, String> row : parseDataList) {
            String proposalNumber = row.get(KFSPropertyConstants.PROPOSAL_NUMBER);
            String chartOfAccountsCode = row.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
            String accountNumber = row.get(KFSPropertyConstants.ACCOUNT_NUMBER);

            if (StringUtils.isEmpty(chartOfAccountsCode)) {
                chartOfAccountsCode = deriveChartOfAccountsCodeFromAccount(accountNumber);
            }

            MilestoneSchedule currentSchedule;

            try {
                currentSchedule = new MilestoneSchedule.MilestoneScheduleBuilder()
                        .setProposalNumber(proposalNumber)
                        .setChartOfAccountsCode(chartOfAccountsCode)
                        .setAccountNumber(accountNumber)
                        .addMilestone(createMilestone(proposalNumber, chartOfAccountsCode, accountNumber, row))
                        .build();
            } catch (IllegalStateException | ParseException e) {
                GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                        KFSKeyConstants.ERROR_BATCH_UPLOAD_PARSING, e.getMessage());
                return null;
            }

            if (milestoneSchedules.contains(currentSchedule)) {
                MilestoneSchedule existingSchedule = milestoneSchedules.get(milestoneSchedules.indexOf(currentSchedule));
                existingSchedule.addMilestone(currentSchedule.getMilestones().get(0));
            } else {
                milestoneSchedules.add(currentSchedule);
            }
        }

        return milestoneSchedules;
    }

    private Milestone createMilestone(String proposalNumber, String chartOfAccountsCode, String accountNumber,
            Map<String, String> row) throws ParseException {
        String milestoneNumber = row.get(ArPropertyConstants.MilestoneFields.MILESTONE_NUMBER);
        String milestoneDescription = row.get(ArPropertyConstants.MilestoneFields.MILESTONE_DESCRIPTION);

        final String milestoneAmountString = row.get(ArPropertyConstants.MilestoneFields.MILESTONE_AMOUNT);
        KualiDecimal milestoneAmount = null;
        if (StringUtils.isNotBlank(milestoneAmountString)) {
            milestoneAmount = new KualiDecimal(milestoneAmountString);
        }

        String milestoneExpectedCompletionDateString = row.get(
                ArPropertyConstants.MilestoneFields.MILESTONE_EXPECTED_COMPLETION_DATE);
        Date milestoneExpectedCompletionDate = null;
        if (milestoneExpectedCompletionDateString.length() > 0) {
            milestoneExpectedCompletionDate = dateTimeService.convertToSqlDate(milestoneExpectedCompletionDateString);
        }

        String milestoneActualCompletionDateString = row.get(
                    ArPropertyConstants.MilestoneFields.MILESTONE_ACTUAL_COMPLETION_DATE);
        Date milestoneActualCompletionDate = null;
        if (milestoneActualCompletionDateString.length() > 0) {
            milestoneActualCompletionDate = dateTimeService.convertToSqlDate(milestoneActualCompletionDateString);
        }

        BooleanConverter converter = new BooleanConverter(Boolean.FALSE);
        boolean active = converter.convert(Boolean.TYPE, row.get(KFSPropertyConstants.ACTIVE));

        Milestone milestone = new Milestone.MilestoneBuilder()
                .setProposalNumber(proposalNumber)
                .setChartOfAccountsCode(chartOfAccountsCode)
                .setAccountNumber(accountNumber)
                .setMilestoneExpectedCompletionDate(milestoneExpectedCompletionDate)
                .setActive(active)
                .setMilestoneNumber(milestoneNumber)
                .setMilestoneDescription(milestoneDescription)
                .setMilestoneAmount(milestoneAmount)
                .setMilestoneActualCompletionDate(milestoneActualCompletionDate)
                .build();

        milestone.setNewCollectionRecord(true);

        return milestone;
    }

    @Override
    public String getFileTypeIdentifier() {
        return ArConstants.MilestoneScheduleImport.FILE_TYPE_IDENTIFIER;
    }

    @Override
    public void process(String fileName, Object parsedFileContents) {
        List<MilestoneSchedule> milestoneSchedules = (List<MilestoneSchedule>) parsedFileContents;

        for (MilestoneSchedule milestoneSchedule: milestoneSchedules) {
            MaintenanceDocument milestoneScheduleDocument;
            milestoneScheduleDocument = (MaintenanceDocument) documentService.getNewDocument(
                    maintenanceDocumentDictionaryService.getDocumentTypeName(
                            MilestoneSchedule.class));

            milestoneScheduleDocument.getDocumentHeader().setDocumentDescription("Milestone Schedule Import");

            MilestoneSchedule existingMilestoneSchedule = getMilestoneSchedule(milestoneSchedule);
            if (ObjectUtils.isNull(existingMilestoneSchedule)) {
                milestoneScheduleDocument.getNewMaintainableObject()
                        .setMaintenanceAction(KRADConstants.MAINTENANCE_NEW_ACTION);
                milestoneScheduleDocument.getNewMaintainableObject().setBusinessObject(milestoneSchedule);
                milestoneScheduleDocument.getOldMaintainableObject().setBusinessObject(new MilestoneSchedule());
            } else {
                MilestoneSchedule copyOfExistingMilestoneSchedule =
                        (MilestoneSchedule) ObjectUtils.deepCopy(existingMilestoneSchedule);
                milestoneSchedule.getMilestones().forEach(milestone -> {
                    copyOfExistingMilestoneSchedule.addMilestone(milestone);
                    // add empty milestone to collection so that milestone collection sizes on old and new
                    // will match. yes this is dumb. see FieldUtils.meshFields (src of stupid)
                    existingMilestoneSchedule.addMilestone(new Milestone());
                });
                milestoneScheduleDocument.getNewMaintainableObject().setBusinessObject(copyOfExistingMilestoneSchedule);
                milestoneScheduleDocument.getNewMaintainableObject()
                        .setMaintenanceAction(KRADConstants.MAINTENANCE_EDIT_ACTION);
                milestoneScheduleDocument.getOldMaintainableObject().setBusinessObject(existingMilestoneSchedule);
            }

            try {
                documentService.saveDocument(milestoneScheduleDocument);
                documentService.routeDocument(milestoneScheduleDocument,
                        "Routed New/Edit Milestone Schedule Maintenance Document from Milestone Schedule Import Process",
                        null);
            } catch (ValidationException ve) {
                // ValidationException(s) that are mapped to GLOBAL_ERRORS may prevent the document from being saved
                // (e.g. when there is another maintenance document that has a lock on the same key as the given
                // document). In those cases we want to display an appropriate message to the user.
                final Map<String, AutoPopulatingList<ErrorMessage>> errorMessages = GlobalVariables.getMessageMap()
                        .getErrorMessages();
                errorMessages.keySet().stream()
                        .filter(key -> key.equals(KFSConstants.GLOBAL_ERRORS))
                        .forEach(key -> KNSGlobalVariables.getMessageList()
                                .add(ArKeyConstants.ERROR_AR_MILESTONE_SCHEDULE_IMPORT_SAVE_FAILURE,
                                milestoneSchedule.getProposalNumber(), milestoneSchedule.getChartOfAccountsCode(),
                                        milestoneSchedule.getAccountNumber(),
                                        flattenErrorMessages(errorMessages.get(key))));
                // We have elevated all the error messages we need to relay to the user, so we can clear them out.
                GlobalVariables.getMessageMap().clearErrorMessages();
                // no additional action is necessary if routing caused a ValidationException; in this case we want to
                // continue potentially processing additionally milestone schedules and leave the in error document
                // in saved state
            }
        }

        // used this method so that messaging would go to same place as the messaging coming from batch save process
        KNSGlobalVariables.getMessageList().add(ArKeyConstants.MESSAGE_AR_MILESTONE_SCHEDULE_IMPORT_SUCCESSFUL);
        removeFiles(fileName);
    }

    private MilestoneSchedule getMilestoneSchedule(MilestoneSchedule milestoneSchedule) {
        Map<String, Object> primaryKeys = new HashMap<>();
        primaryKeys.put(KFSPropertyConstants.PROPOSAL_NUMBER, milestoneSchedule.getProposalNumber());
        primaryKeys.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, milestoneSchedule.getChartOfAccountsCode());
        primaryKeys.put(KFSPropertyConstants.ACCOUNT_NUMBER, milestoneSchedule.getAccountNumber());
        return businessObjectService.findByPrimaryKey(MilestoneSchedule.class, primaryKeys);
    }

    @Override
    public String getTitleKey() {
        return ArKeyConstants.MESSAGE_AR_MILESTONE_SCHEDULE_IMPORT_TITLE;
    }

}
