/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.kfs.krad.document.TransactionalDocumentBase;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.FinalBilledIndicatorEntry;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsInvoiceDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.context.SpringContext;

import java.util.ArrayList;
import java.util.List;

/**
 * This class unfinalizes the invoices that have previously been finalized.
 */
public class FinalBilledIndicatorDocument extends TransactionalDocumentBase {

    private static final Logger LOG = LogManager.getLogger();

    private ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService;
    private DocumentService documentService;
    private List<FinalBilledIndicatorEntry> invoiceEntries = new ArrayList<>();

    protected void updateContractsGrantsInvoiceDocument() {
        for (FinalBilledIndicatorEntry entry : this.getInvoiceEntries()) {
            ContractsGrantsInvoiceDocument invoiceDocument;
            invoiceDocument = (ContractsGrantsInvoiceDocument) getDocumentService().getByDocumentHeaderId(entry.getInvoiceDocumentNumber());
            if (ObjectUtils.isNotNull(invoiceDocument)) {
                invoiceDocument.getInvoiceGeneralDetail().setFinalBillIndicator(false);
                getContractGrantsInvoiceDocumentService().updateUnfinalizationToAwardAccount(invoiceDocument.getAccountDetails(), invoiceDocument.getInvoiceGeneralDetail().getProposalNumber());
                invoiceDocument.refresh();
                getDocumentService().updateDocument(invoiceDocument);
            }
        }
    }

    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        super.doRouteStatusChange(statusChangeEvent);
        String newRouteStatus = statusChangeEvent.getNewRouteStatus();
        if (newRouteStatus.equalsIgnoreCase(KFSConstants.DocumentStatusCodes.PROCESSED) || newRouteStatus.equalsIgnoreCase(KFSConstants.DocumentStatusCodes.FINAL)) {
            updateContractsGrantsInvoiceDocument();
        }
    }

    public void addInvoiceEntry(FinalBilledIndicatorEntry invoiceEntry) {
        invoiceEntry.setDocumentId(getDocumentNumber());
        invoiceEntries.add(invoiceEntry);
    }

    public void removeInvoiceEntry(int deleteIndex) {
        invoiceEntries.remove(deleteIndex);
    }

    private ContractsGrantsInvoiceDocumentService getContractGrantsInvoiceDocumentService() {
        if (contractsGrantsInvoiceDocumentService == null) {
            contractsGrantsInvoiceDocumentService = SpringContext.getBean(ContractsGrantsInvoiceDocumentService.class);
        }
        return contractsGrantsInvoiceDocumentService;
    }

    protected void setContractsGrantsInvoiceDocumentService(
            ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService) {
        this.contractsGrantsInvoiceDocumentService = contractsGrantsInvoiceDocumentService;
    }

    private DocumentService getDocumentService() {
        if (documentService == null) {
            documentService = SpringContext.getBean(DocumentService.class);
        }
        return documentService;
    }

    protected void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public List<FinalBilledIndicatorEntry> getInvoiceEntries() {
        return invoiceEntries;
    }

    public void setInvoiceEntries(List<FinalBilledIndicatorEntry> invoiceEntries) {
        this.invoiceEntries = invoiceEntries;
    }
}
