/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch.service.impl;

import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.krad.UserSession;
import org.kuali.kfs.krad.bo.AdHocRoutePerson;
import org.kuali.kfs.krad.bo.AdHocRouteRecipient;
import org.kuali.kfs.krad.bo.AdHocRouteWorkgroup;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.batch.service.InvoiceRecurrenceService;
import org.kuali.kfs.module.ar.businessobject.InvoiceRecurrence;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Date;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Lockbox Iterators are sorted by processedInvoiceDate and batchSequenceNumber.
 * Potentially there could be many batches on the same date.
 * For each set of records with the same processedInvoiceDate and batchSequenceNumber,
 * there will be one Cash-Control document. Each record within this set will create one Application document.
 */

@Transactional
public class InvoiceRecurrenceServiceImpl implements InvoiceRecurrenceService {

    private DocumentService documentService;
    private BusinessObjectService boService;

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    @Override
    public boolean processInvoiceRecurrence() {
        final Collection<InvoiceRecurrence> recurrences = getAllActiveInvoiceRecurrences();
        CustomerInvoiceDocument customerInvoiceDocument = new CustomerInvoiceDocument();
        for (final InvoiceRecurrence invoiceRecurrence : recurrences) {
            /* Get some dates and calendars  */
            final LocalDate currentDate = LocalDate.now();

            final LocalDate beginDate = invoiceRecurrence.getDocumentRecurrenceBeginDate().toLocalDate();
            final int day = beginDate.getDayOfMonth();

            final LocalDate endDate = invoiceRecurrence.getDocumentRecurrenceEndDate().toLocalDate();
            final LocalDate lastCreateDate =
                    invoiceRecurrence.getDocumentLastCreateDate() == null ?
                            null : invoiceRecurrence.getDocumentLastCreateDate().toLocalDate();

            /* Calculate currentMonthProcessDate*/
            final LocalDate currentMonthProcessDate = currentDate.withDayOfMonth(day);

            final LocalDate nextProcessDate;

            /* Calculate the nextProcessDate */
            if (currentDate.isAfter(currentMonthProcessDate)) {
                nextProcessDate = currentMonthProcessDate.plusMonths(1L);
            } else {
                /* currentDate is less than or equal to currentMonthProcessDate
                   so the nextProcessDate is equal to the currentMonthProcessDate */
                nextProcessDate = currentMonthProcessDate;
            }

            /* Calculate the lastProcessDate by subtracting one month from nextProcessingDate */
            LocalDate lastProcessDate = nextProcessDate.minusMonths(1L);
            if (lastProcessDate.isBefore(beginDate)) {
                lastProcessDate = LocalDate.EPOCH;
            }
            /* if nextProcessDate is equal to currentDate create INV document */
            if (nextProcessDate.equals(currentDate)) {
                /* copy INV document to a new INV document */
                final String initiator = invoiceRecurrence.getDocumentInitiatorUserPersonUserIdentifier();
                GlobalVariables.setUserSession(new UserSession(initiator));

                customerInvoiceDocument = (CustomerInvoiceDocument) documentService.getByDocumentHeaderId(invoiceRecurrence.getInvoiceNumber());
                customerInvoiceDocument.toCopy();
                final List<AdHocRouteRecipient> adHocRouteRecipients = new ArrayList<>();
                adHocRouteRecipients.add(buildApprovePersonRecipient(initiator));
                documentService.routeDocument(customerInvoiceDocument, "This is a recurred Customer Invoice", adHocRouteRecipients);
                invoiceRecurrence.setDocumentLastCreateDate(Date.valueOf(currentDate));
                boService.save(invoiceRecurrence);

            }

            /* if nextProcessDate is greater than currentDate BUT less than or equal to endDate */
            if (nextProcessDate.isAfter(currentDate) && !nextProcessDate.isAfter(endDate)) {
                if (ObjectUtils.isNotNull(lastCreateDate) && lastProcessDate.isAfter(lastCreateDate) ||
                    ObjectUtils.isNull(lastCreateDate) && beginDate.isBefore(currentDate)) {
                    /* copy INV document to a new INV document */
                    final String initiator = invoiceRecurrence.getDocumentInitiatorUserPersonUserIdentifier();
                    GlobalVariables.setUserSession(new UserSession(initiator));

                    customerInvoiceDocument = (CustomerInvoiceDocument) documentService.getByDocumentHeaderId(invoiceRecurrence.getInvoiceNumber());
                    customerInvoiceDocument.toCopy();
                    final List<AdHocRouteRecipient> adHocRouteRecipients = new ArrayList<>();
                    adHocRouteRecipients.add(buildApprovePersonRecipient(initiator));
                    documentService.routeDocument(customerInvoiceDocument, "This is a recurred Customer Invoice", adHocRouteRecipients);
                    invoiceRecurrence.setDocumentLastCreateDate(Date.valueOf(currentDate));
                    boService.save(invoiceRecurrence);
                }
            }

            /* Check if this is the last recurrence. If yes, inactivate the INVR and send an FYI to the initiator and workgroup.  */
            if (!nextProcessDate.isBefore(endDate)) {
                /* Change the active indicator to 'N' and send an FYI */
                final String initiator = invoiceRecurrence.getDocumentInitiatorUserPersonUserIdentifier();
                GlobalVariables.setUserSession(new UserSession(initiator));

                final MaintenanceDocument newMaintDoc = (MaintenanceDocument) documentService.getNewDocument(getInvoiceRecurrenceMaintenanceDocumentTypeName());
                newMaintDoc.getOldMaintainableObject().setBusinessObject(invoiceRecurrence);
                final InvoiceRecurrence newInvoiceRecurrence = invoiceRecurrence;
                newInvoiceRecurrence.setActive(false);
                newMaintDoc.getDocumentHeader().setDocumentDescription("Generated by Batch process");
                newMaintDoc.getDocumentHeader().setExplanation("Inactivated by the Batch process");
                newMaintDoc.getNewMaintainableObject().setBusinessObject(newInvoiceRecurrence);
                newMaintDoc.getNewMaintainableObject().setMaintenanceAction(KRADConstants.MAINTENANCE_EDIT_ACTION);
                final List<AdHocRouteRecipient> adHocRouteRecipients = new ArrayList<>();
                adHocRouteRecipients.add(buildFyiPersonRecipient(initiator));
                documentService.routeDocument(newMaintDoc, null, adHocRouteRecipients);
                newInvoiceRecurrence.setDocumentLastCreateDate(Date.valueOf(currentDate));
                boService.save(newInvoiceRecurrence);
            }
        }
        return true;

    }

    /**
     * @return all active invoice recurrences
     */
    protected Collection<InvoiceRecurrence> getAllActiveInvoiceRecurrences() {
        final Map<String, Object> fieldValues = new HashMap<>();
        fieldValues.put(KFSPropertyConstants.ACTIVE, Boolean.TRUE);
        return boService.findMatchingOrderBy(InvoiceRecurrence.class,
                fieldValues, "invoiceNumber", true);
    }

    protected String getInvoiceRecurrenceMaintenanceDocumentTypeName() {
        return "INVR";
    }

    /**
     * This method builds a FYI recipient.
     *
     * @param userId
     * @return
     */
    protected AdHocRouteRecipient buildFyiPersonRecipient(final String userId) {
        final AdHocRouteRecipient adHocRouteRecipient = new AdHocRoutePerson();
        adHocRouteRecipient.setActionRequested(KewApiConstants.ACTION_REQUEST_FYI_REQ);
        adHocRouteRecipient.setId(userId);
        return adHocRouteRecipient;
    }

    /**
     * This method builds a recipient for Approval.
     *
     * @param userId
     * @return
     */
    protected AdHocRouteRecipient buildApprovePersonRecipient(final String userId) {
        final AdHocRouteRecipient adHocRouteRecipient = new AdHocRoutePerson();
        adHocRouteRecipient.setActionRequested(KewApiConstants.ACTION_REQUEST_APPROVE_REQ);
        adHocRouteRecipient.setId(userId);
        return adHocRouteRecipient;
    }

    /**
     * This method builds a FYI workgroup recipient.
     *
     * @param workgroupId
     * @return
     */
    protected AdHocRouteRecipient buildFyiWorkgroupRecipient(final String workgroupId) {
        final AdHocRouteRecipient adHocRouteRecipient = new AdHocRouteWorkgroup();
        adHocRouteRecipient.setActionRequested(KewApiConstants.ACTION_REQUEST_FYI_REQ);
        adHocRouteRecipient.setId(workgroupId);
        return adHocRouteRecipient;
    }

    /**
     * This method builds a workgroup recipient for Approval.
     *
     * @param workgroupId
     * @return
     */
    protected AdHocRouteRecipient buildApproveWorkgroupRecipient(final String workgroupId) {
        final AdHocRouteRecipient adHocRouteRecipient = new AdHocRouteWorkgroup();
        adHocRouteRecipient.setActionRequested(KewApiConstants.ACTION_REQUEST_APPROVE_REQ);
        adHocRouteRecipient.setId(workgroupId);
        return adHocRouteRecipient;
    }

    public void setBusinessObjectService(final BusinessObjectService boService) {
        this.boService = boService;
    }
}
