/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject;

import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.krad.bo.PersistableBusinessObjectBase;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.SystemInformationService;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.UniversityDateService;

import java.util.Collection;
import java.util.List;

public class InvoicePaidApplied extends PersistableBusinessObjectBase {

    // document the payment is being applied FROM
    private String documentNumber;
    private Integer paidAppliedItemNumber;
    // document the payment is being applied TO
    private String financialDocumentReferenceInvoiceNumber;
    private Integer invoiceItemNumber;
    private Integer universityFiscalYear;
    private String universityFiscalPeriodCode;
    private boolean adjusted;
    private KualiDecimal invoiceItemAppliedAmount = KualiDecimal.ZERO;
    private KualiDecimal invoiceItemOpenAmount;

    private CustomerInvoiceDetail invoiceDetail;
    private AccountingPeriod universityFiscalPeriod;
    private DocumentHeader documentHeader;
    private transient DocumentService documentService;
    private KualiDecimal paidAppliedDistributionAmount = KualiDecimal.ZERO;
    private Collection<NonInvoicedDistribution> nonInvoicedDistributions;
    private Collection<NonAppliedDistribution> nonAppliedDistributions;
    private transient CustomerInvoiceDocument customerInvoiceDocument;

    public InvoicePaidApplied() {
        super();
    }

    public InvoicePaidApplied(
            final String documentNumber, final String refInvoiceDocNumber, final Integer invoiceSequenceNumber,
            final KualiDecimal appliedAmount, final Integer paidAppliedItemNumber, final Integer universityFiscalYear,
            final String universityFiscalPeriodCode) {
        this.documentNumber = documentNumber;
        financialDocumentReferenceInvoiceNumber = refInvoiceDocNumber;
        invoiceItemNumber = invoiceSequenceNumber;
        this.paidAppliedItemNumber = paidAppliedItemNumber;
        invoiceItemAppliedAmount = appliedAmount;
        this.universityFiscalYear = universityFiscalYear;
        this.universityFiscalPeriodCode = universityFiscalPeriodCode;
    }

    /**
     * Constructs a InvoicePaidApplied object, and assumes the current Fiscal Year and FiscalPeriodCode.
     *
     * @param documentNumber
     * @param refInvoiceDocNumber
     * @param invoiceSequenceNumber
     * @param appliedAmount
     * @param paidAppliedItemNumber
     */
    public InvoicePaidApplied(
            final String documentNumber, final String refInvoiceDocNumber, final Integer invoiceSequenceNumber,
            final KualiDecimal appliedAmount, final Integer paidAppliedItemNumber) {
        this.documentNumber = documentNumber;
        financialDocumentReferenceInvoiceNumber = refInvoiceDocNumber;
        invoiceItemNumber = invoiceSequenceNumber;
        this.paidAppliedItemNumber = paidAppliedItemNumber;
        invoiceItemAppliedAmount = appliedAmount;

        final UniversityDateService universityDateService = SpringContext.getBean(UniversityDateService.class);
        universityFiscalYear = universityDateService.getCurrentFiscalYear();
        universityFiscalPeriodCode = universityDateService.getCurrentUniversityDate().getAccountingPeriod().getUniversityFiscalPeriodCode();
    }

    public static boolean referToSameInvoiceItem(final InvoicePaidApplied left, final InvoicePaidApplied right) {
        final boolean hasMatchingInvoice = left.getFinancialDocumentReferenceInvoiceNumber().equals(
                right.getFinancialDocumentReferenceInvoiceNumber());
        final boolean hasMatchingItemNumber = left.getInvoiceItemNumber().equals(
                right.getInvoiceItemNumber());
        return hasMatchingInvoice && hasMatchingItemNumber;
    }

    public DocumentService getDocumentService() {
        if (null == documentService) {
            documentService = SpringContext.getBean(DocumentService.class);
        }
        return documentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public CustomerInvoiceDocument getCustomerInvoiceDocument() {
        final CustomerInvoiceDocument customerInvoiceDocument;
        customerInvoiceDocument = (CustomerInvoiceDocument) getDocumentService().getByDocumentHeaderId(
                getFinancialDocumentReferenceInvoiceNumber());
        return customerInvoiceDocument;
    }

    public SystemInformation getSystemInformation() {
        final String processingOrgCode = getCustomerInvoiceDocument().getAccountsReceivableDocumentHeader()
                .getProcessingOrganizationCode();
        final String processingChartCode = getCustomerInvoiceDocument().getAccountsReceivableDocumentHeader()
                .getProcessingChartOfAccountCode();

        final SystemInformationService sysInfoService = SpringContext.getBean(SystemInformationService.class);
        final Integer currentFiscalYear = SpringContext.getBean(UniversityDateService.class).getCurrentFiscalYear();
        final SystemInformation systemInformation = sysInfoService.getByProcessingChartOrgAndFiscalYear(processingChartCode,
                processingOrgCode, currentFiscalYear);

        if (systemInformation == null) {
            throw new RuntimeException("The InvoicePaidApplied doesn't have an associated " +
                    "SystemInformation.  This should never happen.");
        }
        return systemInformation;
    }

    public String getDocumentNumber() {
        return documentNumber;
    }

    public void setDocumentNumber(final String documentNumber) {
        this.documentNumber = documentNumber;
    }

    public Integer getPaidAppliedItemNumber() {
        return paidAppliedItemNumber;
    }

    public void setPaidAppliedItemNumber(final Integer paidAppliedItemNumber) {
        this.paidAppliedItemNumber = paidAppliedItemNumber;
    }

    public String getFinancialDocumentReferenceInvoiceNumber() {
        return financialDocumentReferenceInvoiceNumber;
    }

    public void setFinancialDocumentReferenceInvoiceNumber(final String financialDocumentReferenceInvoiceNumber) {
        this.financialDocumentReferenceInvoiceNumber = financialDocumentReferenceInvoiceNumber;
    }

    public Integer getInvoiceItemNumber() {
        return invoiceItemNumber;
    }

    public void setInvoiceItemNumber(final Integer invoiceItemNumber) {
        this.invoiceItemNumber = invoiceItemNumber;
    }

    public Integer getUniversityFiscalYear() {
        return universityFiscalYear;
    }

    public void setUniversityFiscalYear(final Integer universityFiscalYear) {
        this.universityFiscalYear = universityFiscalYear;
    }

    public String getUniversityFiscalPeriodCode() {
        return universityFiscalPeriodCode;
    }

    public void setUniversityFiscalPeriodCode(final String universityFiscalPeriodCode) {
        this.universityFiscalPeriodCode = universityFiscalPeriodCode;
    }

    public DocumentHeader getDocumentHeader() {
        return documentHeader;
    }

    public void setDocumentHeader(final DocumentHeader documentHeader) {
        this.documentHeader = documentHeader;
    }

    public KualiDecimal getInvoiceItemAppliedAmount() {
        return invoiceItemAppliedAmount;
    }

    public void setInvoiceItemAppliedAmount(final KualiDecimal invoiceItemAppliedAmount) {
        this.invoiceItemAppliedAmount = invoiceItemAppliedAmount;
    }

    public void setInvoiceItemOpenAmount(final KualiDecimal invoiceItemOpenAmount) {
        this.invoiceItemOpenAmount = invoiceItemOpenAmount;
    }

    public KualiDecimal getInvoiceItemOpenAmount() {
        return invoiceItemOpenAmount;
    }

    public CustomerInvoiceDetail getInvoiceDetail() {
        return invoiceDetail;
    }

    public boolean isAdjusted() {
        return adjusted;
    }

    public void setAdjusted(final boolean adjusted) {
        this.adjusted = adjusted;
    }

    public AccountingPeriod getUniversityFiscalPeriod() {
        return universityFiscalPeriod;
    }

    @Deprecated
    public void setUniversityFiscalPeriod(final AccountingPeriod universityFiscalPeriod) {
        this.universityFiscalPeriod = universityFiscalPeriod;
    }

    public KualiDecimal getPaidAppiedDistributionAmount() {
        return paidAppliedDistributionAmount;
    }

    public void setPaidAppiedDistributionAmount(final KualiDecimal paidAppiedDistributionAmount) {
        paidAppliedDistributionAmount = paidAppiedDistributionAmount;
    }

    public Collection<NonInvoicedDistribution> getNonInvoicedDistributions() {
        return nonInvoicedDistributions;
    }

    public void setNonInvoicedDistributions(final Collection<NonInvoicedDistribution> nonInvoicedDistributions) {
        this.nonInvoicedDistributions = nonInvoicedDistributions;
    }

    public Collection<NonAppliedDistribution> getNonAppliedDistributions() {
        return nonAppliedDistributions;
    }

    public void setNonAppliedDistributions(final List<NonAppliedDistribution> nonAppliedDistributions) {
        this.nonAppliedDistributions = nonAppliedDistributions;
    }

    public void setCustomerInvoiceDocument(final CustomerInvoiceDocument customerInvoiceDocument) {
        this.customerInvoiceDocument = customerInvoiceDocument;
    }
}
