/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.core.web.format.DateFormatter;
import org.kuali.kfs.core.web.format.Formatter;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAgency;
import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.document.authorization.BusinessObjectRestrictions;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.kns.lookup.KualiLookupableHelperServiceImpl;
import org.kuali.kfs.kns.web.comparator.CellComparatorHelper;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArParameterConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.ContractsAndGrantsAgingReport;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsAgingOpenInvoicesReport;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.businessobject.CustomerAgingReportDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.module.ar.document.service.CustomerCreditMemoDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsAgingReportService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsReportHelperService;
import org.kuali.kfs.module.ar.report.service.CustomerAgingReportService;
import org.kuali.kfs.module.ar.web.struts.ContractsGrantsAgingReportForm;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.util.KfsDateUtils;
import org.springframework.beans.factory.InitializingBean;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ContractsGrantsAgingReportLookupableHelperServiceImpl extends KualiLookupableHelperServiceImpl implements
        InitializingBean {

    protected DateTimeService dateTimeService;
    protected ContractsGrantsAgingReportService contractsGrantsAgingReportService;
    protected ContractsGrantsReportHelperService contractsGrantsReportHelperService;
    protected KualiModuleService moduleService;
    protected CustomerAgingReportService customerAgingReportService;
    protected CustomerCreditMemoDocumentService customerCreditMemoDocumentService;
    private CustomerInvoiceDocumentService customerInvoiceDocumentService;

    private String customerNameLabel;
    private String customerNumberLabel;
    private String cutoffdate30Label;
    private String cutoffdate60Label;
    private String cutoffdate90Label;

    private KualiDecimal total0to30 = KualiDecimal.ZERO;
    private KualiDecimal total31to60 = KualiDecimal.ZERO;
    private KualiDecimal total61to90 = KualiDecimal.ZERO;
    private KualiDecimal total91toSYSPR = KualiDecimal.ZERO;
    private KualiDecimal totalSYSPRplus1orMore = KualiDecimal.ZERO;

    private KualiDecimal totalOpenInvoices = KualiDecimal.ZERO;
    private KualiDecimal totalCredits = KualiDecimal.ZERO;
    private KualiDecimal totalWriteOffs = KualiDecimal.ZERO;

    private Date reportRunDate;

    private String nbrDaysForLastBucket;
    // default is 120 days
    private String cutoffdate91toSYSPRlabel;
    private String cutoffdateSYSPRplus1orMorelabel;
    private final String agencyShortName = ArConstants.ContractsGrantsAgingReportFields.AGENCY_SHORT_NAME;

    /**
     * Get the search results that meet the input search criteria.
     *
     * @param fieldValues Map containing prop name keys and search values
     * @return a List of found business objects
     */
    @Override
    public List<? extends BusinessObject> getSearchResults(final Map<String, String> fieldValues) {
        final List<ContractsAndGrantsAgingReport> results = new ArrayList<>();
        setBackLocation(fieldValues.get(KFSConstants.BACK_LOCATION));
        setDocFormKey(fieldValues.get(KFSConstants.DOC_FORM_KEY));

        total0to30 = KualiDecimal.ZERO;
        total31to60 = KualiDecimal.ZERO;
        total61to90 = KualiDecimal.ZERO;
        total91toSYSPR = KualiDecimal.ZERO;
        totalSYSPRplus1orMore = KualiDecimal.ZERO;
        totalOpenInvoices = KualiDecimal.ZERO;
        totalWriteOffs = KualiDecimal.ZERO;
        totalCredits = KualiDecimal.ZERO;

        final Map<String, ContractsAndGrantsAgingReport> knownCustomers = new HashMap<>();

        try {
            final java.util.Date today = getDateTimeService().getCurrentDate();
            final String reportRunDateStr = fieldValues.get(ArPropertyConstants.CustomerAgingReportFields.REPORT_RUN_DATE);

            reportRunDate = ObjectUtils.isNull(reportRunDateStr) || reportRunDateStr.isEmpty() ?
                today : getDateTimeService().convertToDate(reportRunDateStr);

            // retrieve filtered data according to the lookup
            final Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer =
                    getContractsGrantsAgingReportService().filterContractsGrantsAgingReport(fieldValues, null,
                            new java.sql.Date(reportRunDate.getTime()));

            // set dates for buckets
            final Date cutoffdate30 = DateUtils.addDays(reportRunDate, -30);
            final Date cutoffdate31 = DateUtils.addDays(reportRunDate, -31);
            final Date cutoffdate60 = DateUtils.addDays(reportRunDate, -60);
            final Date cutoffdate61 = DateUtils.addDays(reportRunDate, -61);
            final Date cutoffdate90 = DateUtils.addDays(reportRunDate, -90);
            final Date cutoffdate91 = DateUtils.addDays(reportRunDate, -91);
            final Date cutoffdate120 = DateUtils.addDays(reportRunDate, -1 * Integer.parseInt(nbrDaysForLastBucket));
            final Date cutoffdate121 = DateUtils.addDays(cutoffdate120, -1);

            if (!MapUtils.isEmpty(cgMapByCustomer)) {
                // 30 days
                computeFor0To30DaysByBillingChartAndOrg(cgMapByCustomer, new java.sql.Date(cutoffdate30.getTime()),
                        new java.sql.Date(reportRunDate.getTime()), knownCustomers);
                // 60 days
                computeFor31To60DaysByBillingChartAndOrg(cgMapByCustomer, new java.sql.Date(cutoffdate60.getTime()),
                        new java.sql.Date(cutoffdate31.getTime()), knownCustomers);
                // 90 days
                computeFor61To90DaysByBillingChartAndOrg(cgMapByCustomer, new java.sql.Date(cutoffdate90.getTime()),
                        new java.sql.Date(cutoffdate61.getTime()), knownCustomers);
                // 120 days
                computeFor91ToSYSPRDaysByBillingChartAndOrg(cgMapByCustomer,
                        new java.sql.Date(cutoffdate120.getTime()), new java.sql.Date(cutoffdate91.getTime()),
                        knownCustomers);
                // 120 + older
                computeForSYSPRplus1orMoreDaysByBillingChartAndOrg(cgMapByCustomer, null,
                        new java.sql.Date(cutoffdate121.getTime()), knownCustomers);
                // credits
                calculateTotalCreditsForCustomers(cgMapByCustomer, knownCustomers);
            }

            // prepare customer map.
            for (final ContractsAndGrantsAgingReport detail : knownCustomers.values()) {
                // get agency name for customer
                final ContractsAndGrantsBillingAgency agencyObj = getAgencyByCustomer(detail.getCustomerNumber());
                if (ObjectUtils.isNotNull(agencyObj)) {
                    detail.setReportingName(agencyObj.getReportingName());
                    detail.setAgencyNumber(agencyObj.getAgencyNumber());
                }

                // set total open invoices
                final KualiDecimal amount = detail.getUnpaidBalance0to30().add(detail.getUnpaidBalance31to60())
                        .add(detail.getUnpaidBalance61to90()).add(detail.getUnpaidBalance91toSYSPR()
                                .add(detail.getUnpaidBalanceSYSPRplus1orMore()));
                detail.setTotalOpenInvoices(amount);
                totalOpenInvoices = totalOpenInvoices.add(amount);

                // find total writeoff
                KualiDecimal writeOffAmt = getCustomerAgingReportService().findWriteOffAmountByCustomerNumber(
                        detail.getCustomerNumber());
                if (ObjectUtils.isNotNull(writeOffAmt)) {
                    totalWriteOffs = totalWriteOffs.add(writeOffAmt);
                } else {
                    writeOffAmt = KualiDecimal.ZERO;
                }
                detail.setTotalWriteOff(writeOffAmt);

                // calculate total credits
                results.add(detail);
            }
        } catch (NumberFormatException | ParseException ex) {
            throw new RuntimeException("Could not parse report run date for lookup", ex);
        }

        return new CollectionIncomplete<>(results, (long) results.size());
    }

    /**
     * @return a List of the names of fields which are marked in data dictionary as return fields.
     */
    @Override
    public List<String> getReturnKeys() {
        return new ArrayList(fieldConversions.keySet());
    }

    /**
     * @return a List of the names of fields which are marked in data dictionary as return fields.
     */
    @Override
    protected Map<String, String> getParameters(
            final BusinessObject bo, final Map<String, String> fieldConversions,
            final String lookupImpl, final List returnKeys) {
        final Map<String, String> parameters = new HashMap<>();
        parameters.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.RETURN_METHOD_TO_CALL);
        parameters.put(KRADConstants.DOC_FORM_KEY, getDocFormKey());
        parameters.put(KRADConstants.REFRESH_CALLER, lookupImpl);
        if (ObjectUtils.isNotNull(getReferencesToRefresh())) {
            parameters.put(KRADConstants.REFERENCES_TO_REFRESH, getReferencesToRefresh());
        }

        for (final String key : getReturnKeys()) {
            String fieldNm = key;

            Object fieldVal = ObjectUtils.getPropertyValue(bo, fieldNm);
            if (ObjectUtils.isNull(fieldVal)) {
                fieldVal = KFSConstants.EMPTY_STRING;
            }

            // Encrypt value if it is a secure field
            if (fieldConversions.containsKey(fieldNm)) {
                fieldNm = fieldConversions.get(fieldNm);
            }

            // need to format date in url
            if (fieldVal instanceof Date) {
                final DateFormatter dateFormatter = new DateFormatter();
                fieldVal = dateFormatter.format(fieldVal);
            }

            parameters.put(fieldNm, fieldVal.toString());
        }

        return parameters;
    }

    @Override
    public Collection<? extends BusinessObject> performLookup(
            final LookupForm lookupForm, final Collection<ResultRow> resultTable,
            final boolean bounded) {
        final List<? extends BusinessObject> displayList = getSearchResults(lookupForm.getFieldsForLookup());

        // MJM get resultTable populated here
        if (bounded) {
            boolean hasReturnableRow = false;

            final Person user = GlobalVariables.getUserSession().getPerson();

            // iterate through result list and wrap rows with return url and action urls
            for (final BusinessObject element : displayList) {
                final BusinessObjectRestrictions businessObjectRestrictions = getBusinessObjectAuthorizationService()
                        .getLookupResultRestrictions(element, user);

                if (ObjectUtils.isNotNull(getColumns())) {
                    final List<Column> columns = getColumns();
                    populateCutoffdateLabels();
                    for (final Column col : columns) {
                        final Formatter formatter = col.getFormatter();

                        // pick off result column from result list, do formatting
                        final Object prop = ObjectUtils.getPropertyValue(element, col.getPropertyName());

                        String propValue = ObjectUtils.getFormattedPropertyValue(element, col.getPropertyName(),
                                col.getFormatter());
                        final Class propClass = getPropertyClass(element, col.getPropertyName());

                        // formatters
                        if (ObjectUtils.isNotNull(prop)) {
                            propValue = getContractsGrantsReportHelperService().formatByType(prop, formatter);
                        }

                        // comparator
                        col.setComparator(CellComparatorHelper.getAppropriateComparatorForPropertyClass(propClass));
                        col.setValueComparator(CellComparatorHelper.getAppropriateValueComparatorForPropertyClass(
                                propClass));

                        propValue = super.maskValueIfNecessary(element, col.getPropertyName(), propValue,
                                businessObjectRestrictions);
                        col.setPropertyValue(propValue);

                        // add correct label for sysparam
                        if (StringUtils.equals("unpaidBalance91toSYSPR", col.getPropertyName())) {
                            col.setColumnTitle(cutoffdate91toSYSPRlabel);
                        }
                        if (StringUtils.equals("unpaidBalanceSYSPRplus1orMore", col.getPropertyName())) {
                            col.setColumnTitle(cutoffdateSYSPRplus1orMorelabel);
                        }
                        if (StringUtils.equals("reportingName", col.getPropertyName())) {
                            col.setColumnTitle(agencyShortName);
                        }

                        if (StringUtils.isNotBlank(propValue)) {
                            // do not add link to the values in column "Customer Name"
                            if (StringUtils.equals(customerNameLabel, col.getColumnTitle())) {
                                col.setPropertyURL(getCustomerLookupUrl(element));
                            } else if (StringUtils.equals(customerNumberLabel, col.getColumnTitle())) {
                                col.setPropertyURL(getCustomerOpenInvoicesReportUrl(element, col.getColumnTitle(),
                                        lookupForm.getFieldsForLookup()));
                            } else if (StringUtils.equals(ArConstants.ContractsGrantsAgingReportFields.TOTAL_CREDITS,
                                    col.getColumnTitle())) {
                                col.setPropertyURL(getCreditMemoDocSearchUrl(element));
                            } else if (StringUtils.equals(ArConstants.ContractsGrantsAgingReportFields.TOTAL_WRITEOFF,
                                    col.getColumnTitle())) {
                                col.setPropertyURL(getCustomerWriteoffSearchUrl(element));
                            } else if (StringUtils.equals(
                                    ArConstants.ContractsGrantsAgingReportFields.AGENCY_SHORT_NAME,
                                    col.getColumnTitle())) {
                                col.setPropertyURL(getAgencyInquiryUrl(element));
                            } else {
                                col.setPropertyURL(getCustomerOpenInvoicesReportUrl(element, col.getColumnTitle(),
                                        lookupForm.getFieldsForLookup()));
                            }
                        }
                    }

                    final ResultRow row = new ResultRow(columns, KFSConstants.EMPTY_STRING, KFSConstants.EMPTY_STRING);
                    if (element instanceof PersistableBusinessObject) {
                        row.setObjectId(((PersistableBusinessObject) element).getObjectId());
                    }

                    final boolean rowReturnable = isResultReturnable(element);
                    row.setRowReturnable(rowReturnable);
                    if (rowReturnable) {
                        hasReturnableRow = true;
                    }

                    resultTable.add(row);
                }

                lookupForm.setHasReturnableRow(hasReturnableRow);
            }

            if (displayList.size() != 0) {
                ((ContractsGrantsAgingReportForm) lookupForm).setTotal0to30(total0to30.toString());
                ((ContractsGrantsAgingReportForm) lookupForm).setTotal31to60(total31to60.toString());
                ((ContractsGrantsAgingReportForm) lookupForm).setTotal61to90(total61to90.toString());
                ((ContractsGrantsAgingReportForm) lookupForm).setTotal91toSYSPR(total91toSYSPR.toString());
                ((ContractsGrantsAgingReportForm) lookupForm).setTotalSYSPRplus1orMore(totalSYSPRplus1orMore.toString());
                ((ContractsGrantsAgingReportForm) lookupForm).setTotalOpenInvoices(totalOpenInvoices.toString());
                ((ContractsGrantsAgingReportForm) lookupForm).setTotalCredits(totalCredits.toString());
                ((ContractsGrantsAgingReportForm) lookupForm).setTotalWriteOffs(totalWriteOffs.toString());
            }
        }
        return displayList;
    }

    protected String getCustomerOpenInvoicesReportUrl(
            final BusinessObject bo, String columnTitle,
            final Map<String, String> fieldsMap) {
        final Map<String, String> parameters = new HashMap<>();

        final ContractsAndGrantsAgingReport detail = (ContractsAndGrantsAgingReport) bo;

        parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, ContractsGrantsAgingOpenInvoicesReport.class
                .getName());
        parameters.put("lookupableImplementaionServiceName", "arContractsGrantsAgingOpenInvoicesReportLookupable");
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.SEARCH_METHOD);
        parameters.put("reportName", ArConstants.ContractsGrantsAgingReportFields.OPEN_INVOCE_REPORT_NAME);
        parameters.put(KFSConstants.DOC_FORM_KEY, "88888888");

        if (ObjectUtils.isNotNull(fieldsMap) && !fieldsMap.isEmpty()) {
            for (final String key : fieldsMap.keySet()) {
                final String val = fieldsMap.get(key);
                // put if val is not blank or null
                if (ObjectUtils.isNotNull(val) && StringUtils.isNotEmpty(val)) {
                    parameters.put(key, fieldsMap.get(key));
                }
            }
        }

        parameters.put(KFSPropertyConstants.CUSTOMER_NUMBER, detail.getCustomerNumber());
        parameters.put(KFSPropertyConstants.CUSTOMER_NAME, detail.getCustomerName());

        // Report Run Date
        parameters.put(ArPropertyConstants.ContractsGrantsAgingReportFields.REPORT_RUN_DATE, getDateTimeService()
                .toDateString(reportRunDate));

        // put bucket dates
        if (StringUtils.equals(columnTitle, customerNumberLabel)) {
            parameters.put(ArPropertyConstants.COLUMN_TITLE, KFSConstants.CustomerOpenItemReport.ALL_DAYS);
            parameters.put(ArPropertyConstants.START_DATE, KFSConstants.EMPTY_STRING);
            parameters.put(ArPropertyConstants.END_DATE, getDateTimeService().toDateString(reportRunDate));
        } else {
            if (StringUtils.equals(columnTitle, cutoffdate30Label)) {
                parameters.put(ArPropertyConstants.START_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -30)));
                parameters.put(ArPropertyConstants.END_DATE, getDateTimeService().toDateString(reportRunDate));
            } else if (StringUtils.equals(columnTitle, cutoffdate60Label)) {
                parameters.put(ArPropertyConstants.START_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -60)));
                parameters.put(ArPropertyConstants.END_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -31)));
            } else if (StringUtils.equals(columnTitle, cutoffdate90Label)) {
                parameters.put(ArPropertyConstants.START_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -90)));
                parameters.put(ArPropertyConstants.END_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -61)));
            } else if (StringUtils.equals(columnTitle, cutoffdate91toSYSPRlabel)) {
                parameters.put(ArPropertyConstants.START_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -120)));
                parameters.put(ArPropertyConstants.END_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -91)));
            } else if (StringUtils.equals(columnTitle, cutoffdateSYSPRplus1orMorelabel)) {
                parameters.put(ArPropertyConstants.START_DATE, KFSConstants.EMPTY_STRING);
                parameters.put(ArPropertyConstants.END_DATE, getDateTimeService().toDateString(DateUtils
                        .addDays(reportRunDate, -121)));
                columnTitle = (Integer.parseInt(nbrDaysForLastBucket) + 1) + " days and older";
            } else {
                parameters.put(ArPropertyConstants.START_DATE, KFSConstants.EMPTY_STRING);
                parameters.put(ArPropertyConstants.END_DATE, getDateTimeService().toDateString(reportRunDate));
            }
            parameters.put(ArPropertyConstants.COLUMN_TITLE, columnTitle);
        }
        return UrlFactory.parameterizeUrl("arContractsGrantsAgingOpenInvoicesReportLookup.do", parameters);
    }

    /**
     * @param bo
     * @return the url for the Payment Application search.
     */
    protected String getCreditMemoDocSearchUrl(final BusinessObject bo) {
        final Map<String, String> params = new HashMap<>();
        final ContractsAndGrantsAgingReport detail = (ContractsAndGrantsAgingReport) bo;
        params.put(KFSPropertyConstants.DOCUMENT_TYPE_NAME, ArConstants.ArDocumentTypeCodes.CUSTOMER_CREDIT_MEMO_DOCUMENT_TYPE_CODE);
        params.put(KewApiConstants.DOCUMENT_ATTRIBUTE_FIELD_PREFIX + ArPropertyConstants.CustomerFields.CUSTOMER_NUMBER, detail.getCustomerNumber());
        params.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.SEARCH_METHOD);
        return UrlFactory.parameterizeUrl(getKualiConfigurationService().getPropertyValueAsString(KRADConstants.WORKFLOW_DOCUMENTSEARCH_URL_KEY), params);
    }

    /**
     * @param bo
     * @return the Url for the customer write off doc search
     */
    protected String getCustomerWriteoffSearchUrl(final BusinessObject bo) {
        final Map<String, String> params = new HashMap<>();
        final ContractsAndGrantsAgingReport detail = (ContractsAndGrantsAgingReport) bo;
        params.put(KFSPropertyConstants.DOCUMENT_TYPE_NAME, ArConstants.ArDocumentTypeCodes.INVOICE_WRITEOFF_DOCUMENT_TYPE_CODE);
        params.put(KewApiConstants.DOCUMENT_ATTRIBUTE_FIELD_PREFIX + ArPropertyConstants.CustomerFields.CUSTOMER_NUMBER, detail.getCustomerNumber());
        params.put(KRADConstants.DISPATCH_REQUEST_PARAMETER, KRADConstants.SEARCH_METHOD);
        return UrlFactory.parameterizeUrl(getKualiConfigurationService().getPropertyValueAsString(KRADConstants.WORKFLOW_DOCUMENTSEARCH_URL_KEY), params);
    }

    /**
     * This method returns the customer lookup url
     *
     * @param bo business object
     * @return the url for the customer lookup
     */
    protected String getCustomerLookupUrl(final BusinessObject bo) {
        final Map<String, String> params = new HashMap<>();
        final ContractsAndGrantsAgingReport detail = (ContractsAndGrantsAgingReport) bo;
        params.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, Customer.class.getName());
        params.put(KFSConstants.RETURN_LOCATION_PARAMETER, "portal.do");
        params.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.START_METHOD);
        params.put(KFSConstants.DOC_FORM_KEY, "88888888");
        params.put(KFSConstants.HIDE_LOOKUP_RETURN_LINK, "true");
        params.put(ArPropertyConstants.CustomerFields.CUSTOMER_NUMBER, detail.getCustomerNumber());
        params.put(ArPropertyConstants.CustomerInvoiceWriteoffLookupResultFields.CUSTOMER_NAME,
                detail.getCustomerName());
        return UrlFactory.parameterizeUrl(KFSConstants.LOOKUP_ACTION, params);
    }

    protected String getAgencyInquiryUrl(final BusinessObject bo) {
        final Map<String, String> params = new HashMap<>();
        final ContractsAndGrantsAgingReport detail = (ContractsAndGrantsAgingReport) bo;
        params.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, ContractsAndGrantsBillingAgency.class.getName());
        params.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, "continueWithInquiry");
        params.put(KFSConstants.DOC_FORM_KEY, "88888888");
        params.put(KFSConstants.HIDE_LOOKUP_RETURN_LINK, "true");
        params.put(KFSPropertyConstants.AGENCY_NUMBER, detail.getAgencyNumber());
        return UrlFactory.parameterizeUrl(KFSConstants.INQUIRY_ACTION, params);
    }

    protected void populateCutoffdateLabels() {
        final String className = ContractsAndGrantsAgingReport.class.getName();
        customerNameLabel = getDataDictionaryService().getAttributeDefinition(className,
                KFSConstants.CustomerAgingReport.CUSTOMER_NAME).getLabel();
        customerNumberLabel = getDataDictionaryService().getAttributeDefinition(className,
                KFSConstants.CustomerOpenItemReport.CUSTOMER_NUMBER).getLabel();
        cutoffdate30Label = getDataDictionaryService().getAttributeDefinition(className,
                KFSConstants.CustomerAgingReport.UNPAID_BALANCE_0_TO_30).getLabel();
        cutoffdate60Label = getDataDictionaryService().getAttributeDefinition(className,
                KFSConstants.CustomerAgingReport.UNPAID_BALANCE_31_TO_60).getLabel();
        cutoffdate90Label = getDataDictionaryService().getAttributeDefinition(className,
                KFSConstants.CustomerAgingReport.UNPAID_BALANCE_61_TO_90).getLabel();
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public KualiDecimal getTotal0to30() {
        return total0to30;
    }

    public void setTotal0to30(final KualiDecimal total0to30) {
        this.total0to30 = total0to30;
    }

    public KualiDecimal getTotal31to60() {
        return total31to60;
    }

    public void setTotal31to60(final KualiDecimal total31to60) {
        this.total31to60 = total31to60;
    }

    public KualiDecimal getTotal61to90() {
        return total61to90;
    }

    public void setTotal61to90(final KualiDecimal total61to90) {
        this.total61to90 = total61to90;
    }

    public KualiDecimal getTotal91toSYSPR() {
        return total91toSYSPR;
    }

    public void setTotal91toSYSPR(final KualiDecimal total91toSYSPR) {
        this.total91toSYSPR = total91toSYSPR;
    }

    public KualiDecimal getTotalSYSPRplus1orMore() {
        return totalSYSPRplus1orMore;
    }

    public void setTotalSYSPRplus1orMore(final KualiDecimal totalSYSPRplus1orMore) {
        this.totalSYSPRplus1orMore = totalSYSPRplus1orMore;
    }

    /**
     * @param knownCustomers
     * @param customer
     * @return
     */
    protected ContractsAndGrantsAgingReport pickContractsGrantsAgingReportDetail(
            final Map<String, ContractsAndGrantsAgingReport> knownCustomers, final String customer) {
        ContractsAndGrantsAgingReport agingReportDetail = knownCustomers.get(customer);
        if (ObjectUtils.isNull(agingReportDetail)) {
            agingReportDetail = new ContractsAndGrantsAgingReport();
            agingReportDetail.setCustomerNumber(customer.substring(0, customer.indexOf('-')));
            agingReportDetail.setCustomerName(customer.substring(customer.indexOf('-') + 1));
            knownCustomers.put(customer, agingReportDetail);
        }
        return agingReportDetail;
    }

    /**
     * @param cgMapByCustomer
     * @param begin
     * @param end
     * @param knownCustomers
     */
    protected void computeFor0To30DaysByBillingChartAndOrg(
            final Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer, final java.sql.Date begin, final java.sql.Date end,
            final Map<String, ContractsAndGrantsAgingReport> knownCustomers) {
        for (final String customerId : cgMapByCustomer.keySet()) {
            final ContractsAndGrantsAgingReport agingReportDetail = pickContractsGrantsAgingReportDetail(knownCustomers,
                    customerId);
            final KualiDecimal amount = calculateInvoiceAmountForCustomer(cgMapByCustomer.get(customerId), begin, end);
            agingReportDetail.setUnpaidBalance0to30(amount);
            total0to30 = total0to30.add(amount);
        }
    }

    /**
     * @param cgMapByCustomer
     * @param begin
     * @param end
     * @param knownCustomers
     */
    protected void computeFor31To60DaysByBillingChartAndOrg(
            final Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer, final java.sql.Date begin, final java.sql.Date end,
            final Map<String, ContractsAndGrantsAgingReport> knownCustomers) {
        for (final String customerId : cgMapByCustomer.keySet()) {
            final ContractsAndGrantsAgingReport agingReportDetail = pickContractsGrantsAgingReportDetail(knownCustomers,
                    customerId);
            final KualiDecimal amount = calculateInvoiceAmountForCustomer(cgMapByCustomer.get(customerId), begin, end);
            agingReportDetail.setUnpaidBalance31to60(amount);
            total31to60 = total31to60.add(amount);
        }
    }

    /**
     * @param cgMapByCustomer
     * @param begin
     * @param end
     * @param knownCustomers
     */
    protected void computeFor61To90DaysByBillingChartAndOrg(
            final Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer, final java.sql.Date begin, final java.sql.Date end,
            final Map<String, ContractsAndGrantsAgingReport> knownCustomers) {
        for (final String customerId : cgMapByCustomer.keySet()) {
            final ContractsAndGrantsAgingReport agingReportDetail = pickContractsGrantsAgingReportDetail(knownCustomers,
                    customerId);
            final KualiDecimal amount = calculateInvoiceAmountForCustomer(cgMapByCustomer.get(customerId), begin, end);
            agingReportDetail.setUnpaidBalance61to90(amount);
            total61to90 = total61to90.add(amount);
        }
    }

    /**
     * @param cgMapByCustomer
     * @param begin
     * @param end
     * @param knownCustomers
     */
    protected void computeFor91ToSYSPRDaysByBillingChartAndOrg(
            final Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer, final java.sql.Date begin, final java.sql.Date end,
            final Map<String, ContractsAndGrantsAgingReport> knownCustomers) {
        for (final String customerId : cgMapByCustomer.keySet()) {
            final ContractsAndGrantsAgingReport agingReportDetail = pickContractsGrantsAgingReportDetail(knownCustomers,
                    customerId);
            final KualiDecimal amount = calculateInvoiceAmountForCustomer(cgMapByCustomer.get(customerId), begin, end);
            agingReportDetail.setUnpaidBalance91toSYSPR(amount);
            total91toSYSPR = total91toSYSPR.add(amount);
        }
    }

    /**
     * @param cgMapByCustomer
     * @param begin
     * @param end
     * @param knownCustomers
     */
    protected void computeForSYSPRplus1orMoreDaysByBillingChartAndOrg(
            final Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer, final java.sql.Date begin, final java.sql.Date end,
            final Map<String, ContractsAndGrantsAgingReport> knownCustomers) {
        for (final String customerId : cgMapByCustomer.keySet()) {
            final ContractsAndGrantsAgingReport agingReportDetail = pickContractsGrantsAgingReportDetail(knownCustomers,
                    customerId);
            final KualiDecimal amount = calculateInvoiceAmountForCustomer(cgMapByCustomer.get(customerId), begin, end);
            agingReportDetail.setUnpaidBalanceSYSPRplus1orMore(amount);
            totalSYSPRplus1orMore = totalSYSPRplus1orMore.add(amount);
        }
    }

    /**
     * @param cgDocs
     * @param begin
     * @param end
     * @return the total invoice amount for the customers.
     */
    protected KualiDecimal calculateInvoiceAmountForCustomer(
            final Collection<ContractsGrantsInvoiceDocument> cgDocs,
            final java.sql.Date begin, final java.sql.Date end) {
        KualiDecimal invoiceAmt = KualiDecimal.ZERO;
        if (!CollectionUtils.isEmpty(cgDocs)) {
            for (final ContractsGrantsInvoiceDocument cgDoc : cgDocs) {
                if (ObjectUtils.isNotNull(cgDoc.getBillingDate())) {
                    if (ObjectUtils.isNotNull(begin)) {
                        if (KfsDateUtils.isSameDayOrLater(cgDoc.getBillingDate(), begin)
                                && KfsDateUtils.isSameDayOrEarlier(cgDoc.getBillingDate(), end)) {
                            invoiceAmt = invoiceAmt.add(customerInvoiceDocumentService
                                    .getOpenAmountForCustomerInvoiceDocument(cgDoc));
                        }
                    } else {
                        if (KfsDateUtils.isSameDayOrEarlier(cgDoc.getBillingDate(), end)) {
                            invoiceAmt = invoiceAmt.add(customerInvoiceDocumentService
                                    .getOpenAmountForCustomerInvoiceDocument(cgDoc));
                        }
                    }
                }
            }
        }
        return invoiceAmt;
    }

    protected ContractsAndGrantsBillingAgency getAgencyByCustomer(final String customerNumber) {
        final Map<String, Object> args = new HashMap<>();
        args.put(KFSPropertyConstants.CUSTOMER_NUMBER, customerNumber);
        return getModuleService().getResponsibleModuleService(ContractsAndGrantsBillingAgency.class)
                .getExternalizableBusinessObject(ContractsAndGrantsBillingAgency.class, args);
    }

    /**
     * This method calculates the total credits for the customers.
     *
     * @param cgMapByCustomer
     * @param knownCustomers
     */
    protected void calculateTotalCreditsForCustomers(
            final Map<String, List<ContractsGrantsInvoiceDocument>> cgMapByCustomer,
            final Map<String, ContractsAndGrantsAgingReport> knownCustomers) {
        final Set<String> customerIds = cgMapByCustomer.keySet();
        KualiDecimal credits = KualiDecimal.ZERO;
        for (final String customer : customerIds) {
            final ContractsAndGrantsAgingReport agingReportDetail = pickContractsGrantsAgingReportDetail(knownCustomers,
                    customer);
            final List<ContractsGrantsInvoiceDocument> cgDocs = cgMapByCustomer.get(customer);
            if (!CollectionUtils.isEmpty(cgDocs)) {
                credits = KualiDecimal.ZERO;
                for (final ContractsGrantsInvoiceDocument cgDoc : cgDocs) {
                    final Collection<CustomerCreditMemoDocument> creditDocs = customerCreditMemoDocumentService
                            .getCustomerCreditMemoDocumentByInvoiceDocument(cgDoc.getDocumentNumber());
                    if (ObjectUtils.isNotNull(creditDocs) && !creditDocs.isEmpty()) {
                        for (final CustomerCreditMemoDocument cm : creditDocs) {
                            for (final CustomerCreditMemoDetail cmDetail : cm.getCreditMemoDetails()) {
                                if (ObjectUtils.isNotNull(cmDetail.getCreditMemoItemTotalAmount())) {
                                    credits = credits.add(cmDetail.getCreditMemoItemTotalAmount());
                                }
                            }
                        }
                    }
                }
            }
            agingReportDetail.setTotalCredits(credits);
            totalCredits = totalCredits.add(credits);
        }
    }

    /**
     * Some properties depend on parameters, so let's wait until the parameter service has been set
     *
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        nbrDaysForLastBucket = getParameterService().getParameterValueAsString(CustomerAgingReportDetail.class,
                ArParameterConstants.LAST_TWO_BUCKETS);
        // default is 120 days
        cutoffdate91toSYSPRlabel = "91-" + nbrDaysForLastBucket + " days";
        cutoffdateSYSPRplus1orMorelabel = (Integer.parseInt(nbrDaysForLastBucket) + 1) + "+ days";
    }

    public ContractsGrantsAgingReportService getContractsGrantsAgingReportService() {
        return contractsGrantsAgingReportService;
    }

    public void setContractsGrantsAgingReportService(
            final ContractsGrantsAgingReportService contractsGrantsAgingReportService) {
        this.contractsGrantsAgingReportService = contractsGrantsAgingReportService;
    }

    public ContractsGrantsReportHelperService getContractsGrantsReportHelperService() {
        return contractsGrantsReportHelperService;
    }

    public void setContractsGrantsReportHelperService(
            final ContractsGrantsReportHelperService contractsGrantsReportHelperService) {
        this.contractsGrantsReportHelperService = contractsGrantsReportHelperService;
    }

    public KualiModuleService getModuleService() {
        return moduleService;
    }

    public void setModuleService(final KualiModuleService moduleService) {
        this.moduleService = moduleService;
    }

    public CustomerCreditMemoDocumentService getCustomerCreditMemoDocumentService() {
        return customerCreditMemoDocumentService;
    }

    public void setCustomerCreditMemoDocumentService(
            final CustomerCreditMemoDocumentService customerCreditMemoDocumentService) {
        this.customerCreditMemoDocumentService = customerCreditMemoDocumentService;
    }

    public CustomerAgingReportService getCustomerAgingReportService() {
        return customerAgingReportService;
    }

    public void setCustomerAgingReportService(final CustomerAgingReportService customerAgingReportService) {
        this.customerAgingReportService = customerAgingReportService;
    }

    public void setCustomerInvoiceDocumentService(
            final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }
}
