/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.businessobject.lookup;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.businessobject.Chart;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.document.authorization.BusinessObjectRestrictions;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.web.comparator.CellComparatorHelper;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.krad.util.UrlFactory;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.Bill;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsPredeterminedBillingReport;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;

public class ContractsGrantsPredeterminedBillingReportLookupableHelperServiceImpl
        extends ContractsGrantsReportLookupableHelperServiceImplBase {

    @Override
    public Collection<? extends BusinessObject> performLookup(final LookupForm lookupForm, final Collection<ResultRow> resultTable,
            final boolean bounded) {
        final Map<String, String> lookupFormFields = lookupForm.getFieldsForLookup();

        setBackLocation(lookupForm.getFieldsForLookup().get(KRADConstants.BACK_LOCATION));
        setDocFormKey(lookupForm.getFieldsForLookup().get(KRADConstants.DOC_FORM_KEY));

        final Collection<Bill> bills = getLookupService().findCollectionBySearchHelper(Bill.class, lookupFormFields, true);

        final List<ContractsGrantsPredeterminedBillingReport> displayList = bills.stream().map(bill -> {
            final ContractsGrantsPredeterminedBillingReport pdbr = new ContractsGrantsPredeterminedBillingReport();
            pdbr.setProposalNumber(bill.getProposalNumber());
            pdbr.setAccountNumber(bill.getAccountNumber());
            pdbr.setChartOfAccountsCode(bill.getChartOfAccountsCode());
            pdbr.setActive(bill.isActive());
            pdbr.setBillNumber(bill.getBillNumber());
            pdbr.setBillDescription(bill.getBillDescription());
            pdbr.setBillDate(bill.getBillDate());
            pdbr.setBilled(bill.isBilled());
            pdbr.setEstimatedAmount(bill.getEstimatedAmount());
            if (bill.isBilled()) {
                pdbr.setInvoiceDocumentNumber(bill.getMostRecentInvoiceDocumentNumber());
            }
            return pdbr;
        }).collect(Collectors.toList());

        buildResultTable(lookupForm, displayList, resultTable);

        return displayList;
    }

    @Override
    protected void buildResultTable(final LookupForm lookupForm, final Collection displayList, final Collection resultTable) {
        final Person user = GlobalVariables.getUserSession().getPerson();
        boolean hasReturnableRow = false;
        // iterate through result list and wrap rows with return url and action url
        for (final Object entry : displayList) {
            final BusinessObject element = (BusinessObject) entry;

            final BusinessObjectRestrictions businessObjectRestrictions = getBusinessObjectAuthorizationService()
                    .getLookupResultRestrictions(element, user);

            final List<Column> columns = getColumns();
            for (final Column col : columns) {
                String propValue = ObjectUtils.getFormattedPropertyValue(element, col.getPropertyName(),
                        col.getFormatter());
                final Class propClass = getPropertyClass(element, col.getPropertyName());

                col.setComparator(CellComparatorHelper.getAppropriateComparatorForPropertyClass(propClass));
                col.setValueComparator(CellComparatorHelper.getAppropriateValueComparatorForPropertyClass(propClass));

                propValue = maskValueIfNecessary(element, col.getPropertyName(), propValue, businessObjectRestrictions);
                col.setPropertyValue(propValue);

                final String propertyName = col.getPropertyName();

                if (KFSPropertyConstants.PROPOSAL_NUMBER.equals(propertyName)) {
                    col.setColumnAnchor(generateInquiryUrl(ContractsAndGrantsBillingAward.class, element,
                            KFSPropertyConstants.PROPOSAL_NUMBER));
                }

                if (KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE.equals(propertyName)) {
                    col.setColumnAnchor(
                            generateInquiryUrl(Chart.class, element, KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE));
                }

                if (KFSPropertyConstants.ACCOUNT_NUMBER.equals(propertyName)) {
                    col.setColumnAnchor(generateInquiryUrl(Account.class, element,
                            KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, KFSPropertyConstants.ACCOUNT_NUMBER));
                }

                if (col.getPropertyName().equals(ArPropertyConstants.INVOICE_DOCUMENT_NUMBER)) {
                    final String url = contractsGrantsReportHelperService.getDocSearchUrl(propValue);

                    final Map<String, String> fieldList = new HashMap<>();
                    fieldList.put(KFSPropertyConstants.DOCUMENT_NUMBER, propValue);
                    final AnchorHtmlData a = new AnchorHtmlData(url, KFSConstants.EMPTY_STRING);
                    a.setTitle(HtmlData.getTitleText(
                            getContractsGrantsReportHelperService()
                                    .createTitleText(ContractsGrantsPredeterminedBillingReport.class),
                            ContractsGrantsPredeterminedBillingReport.class, fieldList));

                    col.setColumnAnchor(a);
                }
            }

            final ResultRow row = new ResultRow(columns, "", ACTION_URLS_EMPTY);

            if (getBusinessObjectDictionaryService().isExportable(getBusinessObjectClass())) {
                row.setBusinessObject(element);
            }
            final boolean isRowReturnable = isResultReturnable(element);
            row.setRowReturnable(isRowReturnable);
            if (isRowReturnable) {
                hasReturnableRow = true;
            }
            resultTable.add(row);
        }
        lookupForm.setHasReturnableRow(hasReturnableRow);
    }

    protected AnchorHtmlData generateInquiryUrl(final Class<?> boClass, final BusinessObject bo, final String... propertyNames) {
        final Map<String, String> parameters = new HashMap<>();
        parameters.put(KFSConstants.DISPATCH_REQUEST_PARAMETER, KFSConstants.START_METHOD);
        parameters.put(KFSConstants.BUSINESS_OBJECT_CLASS_ATTRIBUTE, boClass.getName());
        for (final String property : propertyNames) {
            parameters.put(property, ObjectUtils.getPropertyValue(bo, property).toString());
        }

        final AnchorHtmlData inquiryHref = new AnchorHtmlData(KFSConstants.EMPTY_STRING, KFSConstants.EMPTY_STRING);
        inquiryHref.setHref(UrlFactory.parameterizeUrl(KFSConstants.INQUIRY_ACTION, parameters));
        return inquiryHref;
    }

}
