/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document;

import java.time.LocalDate;

public final class CustomerInvoiceRecurrenceUtils {

    /**
     * Private Constructor since this is a util class that should never be instantiated.
     */
    private CustomerInvoiceRecurrenceUtils() {
    }

    /**
     * Determines whether a given number of recurrences is valid based on the start date, end date, and interval.
     * beginDate counts as recurrence #1.
     * @param beginDate
     * @param endDate
     * @param recurrenceIntervalCode How frequently we create recurrences. M - Monthly. Q - Quarterly
     * @param totalRecurrenceNumber
     * @return
     */
    public static boolean isRecurrenceTotalValid(
            final LocalDate beginDate,
            final LocalDate endDate,
            final String recurrenceIntervalCode,
            final int totalRecurrenceNumber) {
        final int totalRecurrences = calculateTotalRecurrences(beginDate, endDate, recurrenceIntervalCode);
        if (totalRecurrences != totalRecurrenceNumber) {
            return false;
        }
        return true;
    }

    /**
     * Calculates the number of recurrences between a start date and end date given a particular interval code.
     * beginDate counts as recurrence #1.
     * @param beginDate
     * @param endDate
     * @param recurrenceIntervalCode
     * @return
     */
    public static int calculateTotalRecurrences(LocalDate beginDate, final LocalDate endDate, final String recurrenceIntervalCode) {
        int totalRecurrences = 0;
        int addCounter = 0;
        if ("M".equals(recurrenceIntervalCode)) {
            addCounter = 1;
        }
        if ("Q".equals(recurrenceIntervalCode)) {
            addCounter = 3;
        }
        /* perform this loop while begin_date is less than or equal to end_date */
        while (!beginDate.isAfter(endDate)) {
            beginDate = beginDate.plusMonths(addCounter);
            totalRecurrences++;

            final LocalDate nextDate = beginDate.plusMonths(addCounter);
            if (endDate.isAfter(beginDate) && endDate.isBefore(nextDate)) {
                totalRecurrences++;
                break;
            }
        }
        return totalRecurrences;
    }

    /**
     * Determines the date of the final recurrence given a start date, number of recurrences, and interval.
     * beginDate counts as recurrence #1.
     * @param beginDate
     * @param intervalCode
     * @param totalRecurrences
     * @return
     */
    public static LocalDate calculateEndDateByTotalRecurrences(
            final LocalDate beginDate, final String intervalCode,
            final int totalRecurrences) {
        int addCounter = 0;
        if ("M".equals(intervalCode)) {
            addCounter = -1;
            addCounter += totalRecurrences;
        }
        if ("Q".equals(intervalCode)) {
            addCounter = -3;
            addCounter += totalRecurrences * 3;
        }
        return beginDate.plusMonths(addCounter);
    }
}
