/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service;

import org.apache.commons.collections4.CollectionUtils;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.FinalBilledIndicatorEntry;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.FinalBilledIndicatorDocument;

import java.util.List;

public final class FinalBilledIndicatorValidationService {

    private DocumentService documentService;

    /**
     * @param document doc to validate
     * @return true if the Document has invoices for reversal, and they are all valid, false otherwise
     */
    public boolean validateDocument(final Document document) {
        boolean valid = true;

        final List<FinalBilledIndicatorEntry> invoiceEntries = ((FinalBilledIndicatorDocument) document).getInvoiceEntries();
        if (CollectionUtils.isEmpty(invoiceEntries)) {
            GlobalVariables.getMessageMap().putError(ArPropertyConstants.FINAL_BILLED_INDICATOR_ENTRIES_PROPERTY_PATH,
                    ArKeyConstants.ERROR_FINAL_BILLED_INDICATOR_NO_INVOICE);
            valid = false;
        } else {
            for (final FinalBilledIndicatorEntry entry : invoiceEntries) {
                valid &= validateEntry(entry);
            }
        }
        return valid;
    }

    /**
     * @param entry FinalBilledIndicatorEntry to validate
     * @return true if the entry is not null, has an invoice doc # and is valid, false otherwise
     */
    public boolean validateEntry(final FinalBilledIndicatorEntry entry) {
        if (ObjectUtils.isNull(entry) || ObjectUtils.isNull(entry.getInvoiceDocumentNumber())) {
            GlobalVariables.getMessageMap().putError(ArPropertyConstants.FINAL_BILLED_INDICATOR_ENTRIES_PROPERTY_PATH,
                    ArKeyConstants.ERROR_FINAL_BILLED_INDICATOR_NO_INVOICE);
            return false;
        }

        return validContractsGrantsInvoiceValidation(entry);
    }

    /**
     * @param entry entry to validate
     * @return true if the entry is a valid Contracts & Grants Invoice document, false otherwise
     */
    private boolean validContractsGrantsInvoiceValidation(final FinalBilledIndicatorEntry entry) {
        final String docNumber = entry.getInvoiceDocumentNumber();

        final Document testDocument;
        if (documentService.documentExists(docNumber)) {
            testDocument = documentService.getByDocumentHeaderId(docNumber);
            if (!testDocument.getDocumentHeader().getWorkflowDocument().getDocumentTypeName()
                    .equals(ArConstants.ArDocumentTypeCodes.CONTRACTS_GRANTS_INVOICE)) {
                GlobalVariables.getMessageMap()
                        .putError(ArPropertyConstants.FINAL_BILLED_INDICATOR_ENTRIES_PROPERTY_PATH,
                                ArKeyConstants.ERROR_FINAL_BILLED_INDICATOR_INVALID_INVOICE);
                return false;
            }
            return entryValidations((ContractsGrantsInvoiceDocument) testDocument);
        } else {
            GlobalVariables.getMessageMap().putError(ArPropertyConstants.FINAL_BILLED_INDICATOR_ENTRIES_PROPERTY_PATH,
                    ArKeyConstants.ERROR_FINAL_BILLED_INDICATOR_INVALID_INVOICE);
            return false;
        }
    }

    /**
     * @param document invoice document to validate
     * @return true if each invoice is final Billed and in final Status, false otherwise
     */
    private boolean entryValidations(final ContractsGrantsInvoiceDocument document) {
        final WorkflowDocument workflowDocument = document.getDocumentHeader().getWorkflowDocument();
        if (!workflowDocument.isFinal() && !workflowDocument.isProcessed()) {
            GlobalVariables.getMessageMap().putError(ArPropertyConstants.FINAL_BILLED_INDICATOR_ENTRIES_PROPERTY_PATH,
                    ArKeyConstants.ERROR_FINAL_BILLED_INDICATOR_INVOICE_NOT_FINAL);
            return false;
        }
        if (!document.getInvoiceGeneralDetail().isFinalBillIndicator()) {
            GlobalVariables.getMessageMap().putError(ArPropertyConstants.FINAL_BILLED_INDICATOR_ENTRIES_PROPERTY_PATH,
                    ArKeyConstants.ERROR_FINAL_BILLED_INDICATOR_INVOICE_NOT_MARKED_FINAL_BILL);
            return false;
        }
        return true;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }
}
