/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service;

import org.kuali.kfs.module.ar.businessobject.AppliedPayment;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;

import java.util.Collection;
import java.util.List;

public interface InvoicePaidAppliedService<T extends AppliedPayment> {

    Integer getNumberOfInvoicePaidAppliedsForInvoiceDetail(String financialDocumentReferenceInvoiceNumber, Integer invoiceItemNumber);

    /**
     * Clears all the PaidApplieds for the given document number from the database.
     * This is typically done prior to loading a new set of lines in.
     * <p>
     * NOTE that this will delete the Paid Applieds that were generated BY the
     * documentNumber passed in, not that apply TO the documentNumber passed in.
     *
     * @param documentNumber
     */
    void clearDocumentPaidAppliedsFromDatabase(String documentNumber);

    /**
     * @param document
     * @return {@code true} if invoice has applied amounts (i.e. from application, credit memo, etc), not including
     *         discounts
     */
    boolean doesInvoiceHaveAppliedAmounts(CustomerInvoiceDocument document);

    /**
     * This method returns InvoicePaidApplied objects corresponding to an invoice by documentNumber.
     *
     * @param documentNumber used to find the invoice and get corresponding InvoicePaidApplied objects.
     * @return Collection<InvoicePaidApplied> invoice paid applieds for the invoice
     */
    Collection<InvoicePaidApplied> getInvoicePaidAppliedsForInvoice(String documentNumber);

    /**
     * This method returns InvoicePaidApplied objects corresponding to an invoice.
     *
     * @param invoice used get corresponding InvoicePaidApplied objects.
     * @return Collection<InvoicePaidApplied> invoice paid applied for the invoice
     */
    Collection<InvoicePaidApplied> getInvoicePaidAppliedsForInvoice(CustomerInvoiceDocument invoice);

    /**
     * @param invoice the invoice used to determine with InvoicePaidApplieds to collect
     * @return All InvoicePaidApplieds for a give invoice that have not been adjusted and refer to the provided invoice
     * document
     */
    List<InvoicePaidApplied> getActiveInvoicePaidAppliedsForInvoice(CustomerInvoiceDocument invoice);

    /**
     * This method takes into account the possibility of an Invoice Paid applied to be adjusted. If a given
     * InvoicePaidApplied has additional InvoicePaidApplied in adjuster documents, then we ignore the amount applied to
     * that invoice and use the last non-adjusted application
     *
     * @param invoicePaidApplieds A list of InvoicePaidApplieds to be filtered
     * @return A list of InvoicePaidApplied that do not have amounts that have been adjusted by other InvoicePaidApplieds
     */
    List<InvoicePaidApplied> filterInvoicePaidAppliedsToOnlyActive(List<InvoicePaidApplied> invoicePaidApplieds);
}
