/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAgency;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.businessobject.ContractsGrantsAgingOpenInvoicesReport;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsAgingOpenInvoicesReportService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.report.service.ContractsGrantsAgingReportService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.core.api.datetime.DateTimeService;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class is used to get the services for PDF generation and other services for Contracts & Grants Aging open
 * Invoices report
 */
public class ContractsGrantsAgingOpenInvoicesReportServiceImpl implements ContractsGrantsAgingOpenInvoicesReportService {

    protected ContractsGrantsAgingReportService contractsGrantsAgingReportService;
    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected DateTimeService dateTimeService;
    private static final Logger LOG = LogManager.getLogger();
    protected KualiModuleService kualiModuleService;

    public ContractsGrantsAgingReportService getContractsGrantsAgingReportService() {
        return contractsGrantsAgingReportService;
    }

    public void setContractsGrantsAgingReportService(
            final ContractsGrantsAgingReportService contractsGrantsAgingReportService) {
        this.contractsGrantsAgingReportService = contractsGrantsAgingReportService;
    }

    public CustomerInvoiceDocumentService getCustomerInvoiceDocumentService() {
        return customerInvoiceDocumentService;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    /**
     * This method populates ContractsGrantsAgingOpenInvoicesReportDetails (Contracts & Grants Open Invoices Report)
     *
     * @param urlParameters
     */
    @Override
    public List getPopulatedReportDetails(final Map urlParameters) {
        final List results = new ArrayList();
        final String customerNumber = ((String[]) urlParameters.get(KFSPropertyConstants.CUSTOMER_NUMBER))[0];
        final String customerName = ((String[]) urlParameters.get(KFSPropertyConstants.CUSTOMER_NAME))[0];

        final String strBeginDate = ObjectUtils.isNotNull(
                urlParameters.get(KFSConstants.CustomerOpenItemReport.REPORT_BEGIN_DATE)) ? ((String[]) urlParameters
                .get(KFSConstants.CustomerOpenItemReport.REPORT_BEGIN_DATE))[0] : null;
        final String strEndDate = ObjectUtils.isNotNull(
                urlParameters.get(KFSConstants.CustomerOpenItemReport.REPORT_END_DATE)) ? ((String[]) urlParameters
                .get(KFSConstants.CustomerOpenItemReport.REPORT_END_DATE))[0] : null;
        java.sql.Date startDate = null;
        java.sql.Date endDate = null;
        List<ContractsGrantsInvoiceDocument> selectedInvoices = new ArrayList<>();
        try {
            if (ObjectUtils.isNotNull(strBeginDate) && StringUtils.isNotEmpty(strBeginDate)) {
                startDate = getDateTimeService().convertToSqlDate(strBeginDate);
            }

            if (ObjectUtils.isNotNull(strEndDate) && StringUtils.isNotEmpty(strEndDate)) {
                endDate = getDateTimeService().convertToSqlDate(strEndDate);
            }

            final Map<String, String> fieldValueMap = new HashMap<>();
            for (final Object key : urlParameters.keySet()) {
                final String val = ((String[]) urlParameters.get(key))[0];
                fieldValueMap.put(key.toString(), val);
            }

            final Map<String, List<ContractsGrantsInvoiceDocument>> map = contractsGrantsAgingReportService
                    .filterContractsGrantsAgingReport(fieldValueMap, startDate, endDate);
            if (ObjectUtils.isNotNull(map) && !map.isEmpty()) {
                selectedInvoices = map.get(customerNumber + "-" + customerName);
            }

            if (selectedInvoices.size() == 0) {
                return results;
            }

        } catch (final ParseException ex) {
            LOG.error("problem during ContractsGrantsAgingOpenInvoicesReportServiceImpl.getPopulatedReportDetails",
                    ex);
            throw new RuntimeException("Couldn't parse a date", ex);
        }

        populateReportDetails(selectedInvoices, results);
        return results;
    }

    /**
     * This method prepare the report model object to display on jsp page.
     *
     * @param invoices
     * @param results
     */
    protected void populateReportDetails(final List<ContractsGrantsInvoiceDocument> invoices, final List results) {
        for (final ContractsGrantsInvoiceDocument invoice : invoices) {
            final ContractsGrantsAgingOpenInvoicesReport detail = new ContractsGrantsAgingOpenInvoicesReport();
            detail.setDocumentType(invoice.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
            detail.setDocumentNumber(invoice.getDocumentNumber());
            final String documentDescription = invoice.getDocumentHeader().getDocumentDescription();
            if (ObjectUtils.isNotNull(documentDescription)) {
                detail.setDocumentDescription(documentDescription);
            } else {
                detail.setDocumentDescription("");
            }
            detail.setBillingDate(invoice.getBillingDate());
            detail.setDueApprovedDate(invoice.getInvoiceDueDate());
            detail.setDocumentPaymentAmount(invoice.getDocumentHeader().getFinancialDocumentTotalAmount());
            detail.setUnpaidUnappliedAmount(customerInvoiceDocumentService.getOpenAmountForCustomerInvoiceDocument(invoice));
            detail.setFinalInvoice(ObjectUtils.isNotNull(invoice.getInvoiceGeneralDetail())
                    && invoice.getInvoiceGeneralDetail().isFinalBillIndicator() ?
                    KFSConstants.ParameterValues.STRING_YES : KFSConstants.ParameterValues.STRING_NO);
            // set agency number, proposal number, account number
            if (ObjectUtils.isNotNull(invoice.getInvoiceGeneralDetail())
                    && ObjectUtils.isNotNull(invoice.getInvoiceGeneralDetail().getProposalNumber())) {
                detail.setProposalNumber(invoice.getInvoiceGeneralDetail().getProposalNumber());
            }

            // Set Agency Number
            final ContractsAndGrantsBillingAgency cgAgency = getAgencyByCustomer(
                    invoice.getAccountsReceivableDocumentHeader().getCustomerNumber());
            if (ObjectUtils.isNotNull(cgAgency)) {
                detail.setAgencyNumber(cgAgency.getAgencyNumber());
            }

            // Set Account number
            final List<CustomerInvoiceDetail> details = invoice.getSourceAccountingLines();
            final String accountNum = CollectionUtils.isNotEmpty(details) && ObjectUtils.isNotNull(details.get(0)) ?
                    details.get(0).getAccountNumber() : "";
            detail.setAccountNumber(accountNum);
            results.add(detail);
        }
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    /**
     * This method retrieves the agency for particular customer
     *
     * @param customerNumber
     * @return Returns the agency for the customer
     */
    protected ContractsAndGrantsBillingAgency getAgencyByCustomer(final String customerNumber) {
        final Map<String, Object> args = new HashMap<>();
        args.put(KFSPropertyConstants.CUSTOMER_NUMBER, customerNumber);
        return kualiModuleService.getResponsibleModuleService(ContractsAndGrantsBillingAgency.class)
                .getExternalizableBusinessObject(ContractsAndGrantsBillingAgency.class, args);
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }
}
