/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.actiontaken.ActionTaken;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.action.WorkflowAction;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.dao.DocumentDao;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceRecurrenceDetails;
import org.kuali.kfs.module.ar.businessobject.InvoicePaidApplied;
import org.kuali.kfs.module.ar.businessobject.NonInvoicedDistribution;
import org.kuali.kfs.module.ar.businessobject.OrganizationOptions;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.dataaccess.CustomerInvoiceDocumentDao;
import org.kuali.kfs.module.ar.document.service.AccountsReceivableDocumentHeaderService;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.InvoicePaidAppliedService;
import org.kuali.kfs.module.ar.document.service.NonInvoicedDistributionService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.ChartOrgHolder;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.service.FinancialSystemUserService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.sys.util.KfsDateUtils;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Date;
import java.text.MessageFormat;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.Stream;

@Transactional
public class CustomerInvoiceDocumentServiceImpl implements CustomerInvoiceDocumentService {

    private static final Logger LOG = LogManager.getLogger();

    protected AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService;
    protected BusinessObjectService businessObjectService;
    protected CustomerAddressService customerAddressService;
    protected CustomerInvoiceDetailService customerInvoiceDetailService;
    protected CustomerInvoiceDocumentDao customerInvoiceDocumentDao;
    protected ConfigurationService configurationService;
    protected DateTimeService dateTimeService;
    protected DocumentService documentService;
    protected DocumentDao documentDao;
    protected FinancialSystemUserService financialSystemUserService;
    protected InvoicePaidAppliedService<CustomerInvoiceDetail> invoicePaidAppliedService;
    protected NonInvoicedDistributionService nonInvoicedDistributionService;
    protected PersonService personService;
    protected UniversityDateService universityDateService;
    protected NoteService noteService;

    @Override
    public void convertDiscountsToPaidApplieds(final CustomerInvoiceDocument invoice) {
        // this needs a little explanation. we have to calculate manually whether we've written off the whole thing,
        // because the regular code uses the invoice paid applieds to discount, but since those are added but not
        // committed in this transaction, they're also not visible in this transaction, so we do it manually.
        KualiDecimal openAmount = invoice.getOpenAmount();

        final String invoiceNumber = invoice.getDocumentNumber();
        final List<CustomerInvoiceDetail> discounts = invoice.getDiscounts();

        // retrieve the number of current paid applieds, so we dont have item number overlap
        Integer paidAppliedItemNumber = 0;

        for (final CustomerInvoiceDetail discount : discounts) {
            // if credit amount is zero, do nothing
            if (KualiDecimal.ZERO.equals(discount.getAmount())) {
                continue;
            }

            if (paidAppliedItemNumber == 0) {
                paidAppliedItemNumber = invoicePaidAppliedService.getNumberOfInvoicePaidAppliedsForInvoiceDetail(
                        invoiceNumber, discount.getInvoiceItemNumber());
            }

            // create and save the paidApplied
            final InvoicePaidApplied invoicePaidApplied = new InvoicePaidApplied();
            invoicePaidApplied.setDocumentNumber(invoiceNumber);
            invoicePaidApplied.setPaidAppliedItemNumber(paidAppliedItemNumber++);
            invoicePaidApplied.setFinancialDocumentReferenceInvoiceNumber(invoiceNumber);
            invoicePaidApplied.setInvoiceItemNumber(discount.getInvoiceItemNumber());
            invoicePaidApplied.setUniversityFiscalYear(universityDateService.getCurrentFiscalYear());
            invoicePaidApplied.setUniversityFiscalPeriodCode(universityDateService.getCurrentUniversityDate()
                    .getUniversityFiscalAccountingPeriod());
            invoicePaidApplied.setInvoiceItemAppliedAmount(discount.getAmount().abs());
            openAmount = openAmount.subtract(discount.getAmount().abs());
            businessObjectService.save(invoicePaidApplied);
        }

        // if its open, but now with a zero open amount, then close it
        if (KualiDecimal.ZERO.equals(openAmount)) {
            invoice.setOpenInvoiceIndicator(false);
            invoice.setClosedDate(dateTimeService.getCurrentSqlDate());
            documentService.updateDocument(invoice);
        }
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocuments() {
        return getAllOpenCustomerInvoiceDocuments(true);
    }

    /**
     * @param includeWorkflowHeaders
     * @return
     */
    public Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocuments(final boolean includeWorkflowHeaders) {
        // retrieve the set of documents without workflow headers
        final Collection<CustomerInvoiceDocument> invoices = customerInvoiceDocumentDao.getAllOpen();

        // if we dont need workflow headers, then we're done
        if (!includeWorkflowHeaders) {
            return invoices;
        }

        // make a list of necessary workflow docs to retrieve
        final List<String> documentHeaderIds = new ArrayList<>();
        for (final CustomerInvoiceDocument invoice : invoices) {
            documentHeaderIds.add(invoice.getDocumentNumber());
        }

        // get all of our docs with full workflow headers
        final List<CustomerInvoiceDocument> docs = new ArrayList<>();
        for (final Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(CustomerInvoiceDocument.class,
                documentHeaderIds)) {
            docs.add((CustomerInvoiceDocument) doc);
        }

        return docs;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllOpenCustomerInvoiceDocumentsWithoutWorkflow() {
        return getAllOpenCustomerInvoiceDocuments(false);
    }

    @Override
    public Collection<CustomerInvoiceDocument> attachWorkflowHeadersToTheInvoices(
            final Collection<CustomerInvoiceDocument> invoices) {
        final List<CustomerInvoiceDocument> docs = new ArrayList<>();
        if (invoices == null || invoices.isEmpty()) {
            return docs;
        }

        // make a list of necessary workflow docs to retrieve
        final List<String> documentHeaderIds = new ArrayList<>();
        for (final CustomerInvoiceDocument invoice : invoices) {
            documentHeaderIds.add(invoice.getDocumentNumber());
        }

        // get all of our docs with full workflow headers
        for (final Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(CustomerInvoiceDocument.class,
                documentHeaderIds)) {
            docs.add((CustomerInvoiceDocument) doc);
        }

        return docs;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerNumber(String customerNumber) {
        // customer number is not required to be populated, so we need to check that it's not null first
        if (StringUtils.isNotEmpty(customerNumber)) {
            // trim and force-caps the customer number
            customerNumber = customerNumber.trim().toUpperCase(Locale.US);
        }

        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerNumber(customerNumber));
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerNameByCustomerType(String customerName,
            String customerTypeCode) {
        // trim and force-caps the customer name
        customerName = StringUtils.replace(customerName, KFSConstants.WILDCARD_CHARACTER, KFSConstants.PERCENTAGE_SIGN);
        customerName = customerName.trim();
        if (!customerName.contains("%")) {
            customerName += "%";
        }

        // trim and force-caps
        customerTypeCode = customerTypeCode.trim().toUpperCase(Locale.US);

        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerNameByCustomerType(customerName,
                customerTypeCode));
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerName(String customerName) {
        // trim and force-caps the customer name
        customerName = StringUtils.replace(customerName, KFSConstants.WILDCARD_CHARACTER, KFSConstants.PERCENTAGE_SIGN);
        customerName = customerName.trim();
        if (!customerName.contains("%")) {
            customerName += "%";
        }

        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerName(customerName));
    }

    @Override
    public Collection<CustomerInvoiceDocument> getOpenInvoiceDocumentsByCustomerType(String customerTypeCode) {
        // trim and force-caps
        customerTypeCode = customerTypeCode.trim().toUpperCase(Locale.US);
        return new ArrayList<>(customerInvoiceDocumentDao.getOpenByCustomerType(customerTypeCode));
    }

    @Override
    public Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocument(
            final CustomerInvoiceDocument customerInvoiceDocument) {
        return getCustomerInvoiceDetailsForCustomerInvoiceDocument(customerInvoiceDocument.getDocumentNumber());
    }

    @Override
    public Collection<CustomerInvoiceDetail> getCustomerInvoiceDetailsForCustomerInvoiceDocument(
            final String customerInvoiceDocumentNumber) {
        return customerInvoiceDetailService.getCustomerInvoiceDetailsForInvoice(customerInvoiceDocumentNumber);
    }

    @Override
    public KualiDecimal getOpenAmountForCustomerInvoiceDocument(final String customerInvoiceDocumentNumber) {
        if (null == customerInvoiceDocumentNumber) {
            return null;
        }
        return getOpenAmountForCustomerInvoiceDocument(getInvoiceByInvoiceDocumentNumber(customerInvoiceDocumentNumber));
    }

    @Override
    public KualiDecimal getOpenAmountForCustomerInvoiceDocument(final CustomerInvoiceDocument customerInvoiceDocument) {
        KualiDecimal total = new KualiDecimal(0);
        if (customerInvoiceDocument.isOpenInvoiceIndicator()) {
            final Collection<CustomerInvoiceDetail> customerInvoiceDetails = customerInvoiceDocument
                    .getCustomerInvoiceDetailsWithoutDiscounts();
            for (final CustomerInvoiceDetail detail : customerInvoiceDetails) {
                // note that we're now dealing with conditionally applying discounts depending on whether the doc is
                // saved or approved one level down, in the CustomerInvoiceDetail.getAmountOpen()
                detail.setCustomerInvoiceDocument(customerInvoiceDocument);
                total = total.add(detail.getAmountOpen());
            }
        }
        return total;
    }

    @Override
    public KualiDecimal getOriginalTotalAmountForCustomerInvoiceDocument(
            final CustomerInvoiceDocument customerInvoiceDocument) {
        LOG.info(
                "\n\n\n\t\t invoice: {}\n\t\t 111111111 HEADER TOTAL AMOUNT (should be null): {}\n\n",
                customerInvoiceDocument::getDocumentNumber,
                () -> customerInvoiceDocument.getDocumentHeader().getFinancialDocumentTotalAmount()
        );
        final HashMap<String, String> criteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.DOCUMENT_NUMBER, customerInvoiceDocument.getDocumentHeader()
                .getDocumentTemplateNumber());
        final DocumentHeader documentHeader = businessObjectService.findByPrimaryKey(
                DocumentHeader.class, criteria);
        final KualiDecimal originalTotalAmount = documentHeader.getFinancialDocumentTotalAmount();

        LOG.info(
                "\n\n\n\t\t invoice: {}\n\t\t 333333333333 HEADER TOTAL AMOUNT (should be set now): {}\n\n",
                customerInvoiceDocument::getDocumentNumber,
                () -> customerInvoiceDocument.getDocumentHeader().getFinancialDocumentTotalAmount()
        );
        return originalTotalAmount;
    }

    @Override
    public Collection<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByCustomerNumber(final String customerNumber) {
        final Collection<CustomerInvoiceDocument> invoices = new ArrayList<>();

        final Map<String, String> fieldValues = new HashMap<>();
        fieldValues.put("customerNumber", customerNumber);

        final Collection<AccountsReceivableDocumentHeader> documentHeaders = businessObjectService.findMatching(
                AccountsReceivableDocumentHeader.class, fieldValues);

        final List<String> documentHeaderIds = new ArrayList<>();
        for (final AccountsReceivableDocumentHeader header : documentHeaders) {
            documentHeaderIds.add(header.getDocumentHeader().getDocumentNumber());
        }

        if (0 < documentHeaderIds.size()) {
            for (final Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                    CustomerInvoiceDocument.class, documentHeaderIds)) {
                invoices.add((CustomerInvoiceDocument) doc);
            }
        }
        return invoices;
    }

    @Override
    public Customer getCustomerByOrganizationInvoiceNumber(final String organizationInvoiceNumber) {
        final CustomerInvoiceDocument invoice = getInvoiceByOrganizationInvoiceNumber(organizationInvoiceNumber);
        return invoice.getAccountsReceivableDocumentHeader().getCustomer();
    }

    @Override
    public CustomerInvoiceDocument getInvoiceByOrganizationInvoiceNumber(final String organizationInvoiceNumber) {
        return customerInvoiceDocumentDao.getInvoiceByOrganizationInvoiceNumber(organizationInvoiceNumber);
    }

    @Override
    public Customer getCustomerByInvoiceDocumentNumber(final String invoiceDocumentNumber) {
        final CustomerInvoiceDocument invoice = getInvoiceByInvoiceDocumentNumber(invoiceDocumentNumber);
        return invoice.getAccountsReceivableDocumentHeader().getCustomer();
    }

    @Override
    public CustomerInvoiceDocument getInvoiceByInvoiceDocumentNumber(final String invoiceDocumentNumber) {
        return customerInvoiceDocumentDao.getInvoiceByInvoiceDocumentNumber(invoiceDocumentNumber);
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByInitiatorPrincipalName(
            final String initiatorPrincipalName) {
        if (StringUtils.isBlank(initiatorPrincipalName)) {
            throw new IllegalArgumentException("The parameter [initiatorPrincipalName] passed in was null or blank.");
        }

        // IMPORTANT NOTES ABOUT THIS METHOD
        //
        // This method behaves differently than the other invoice printing methods. This is because there's no way
        // from within KFS to do a direct DB call to get all the invoices you want. This is because workflow holds the
        // document initiator, and you can't guarantee that in a given implementation that you have access to that
        // other db. It could be on another box in another network, and you only have web-services access to the Rice
        // box.
        //
        // Given that, we try to minimize the resource hit of this call as much as possible. First we retrieve all
        // invoices that haven't been printed (ie, dont have a print date) and that are marked for the USER print
        // queue. At any given time that should be a manageable number of documents.
        //
        // Then we walk through them, retrieve the full workflow-populated version of it, and only return the ones
        // that match the initiator.
        //
        // This isn't as performant a solution as the other getPrintableCustomerInvoiceBy...
        // methods, but its the best we can do in this release, and it should be manageable.

        //
        // attempt to retrieve the initiator person specified, and puke if not found
        final Person person = personService.getPersonByPrincipalName(initiatorPrincipalName);
        if (person == null) {
            throw new IllegalArgumentException("The parameter value for initiatorPrincipalName [" +
                    initiatorPrincipalName + "] passed in doesnt map to a person.");
        }

        // retrieve all the ready-to-print docs in the user-queue for all users
        final List<String> printableUserQueueDocNumbers = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersFromUserQueue();

        // get all the documents that might be right, but this set includes documents generated by the wrong user
        List<CustomerInvoiceDocument> customerInvoiceDocumentsSuperSet = new ArrayList<>();
        if (printableUserQueueDocNumbers.size() > 0) {
            for (final Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                    CustomerInvoiceDocument.class, printableUserQueueDocNumbers)) {
                customerInvoiceDocumentsSuperSet.add((CustomerInvoiceDocument) doc);
            }
        } else {
            customerInvoiceDocumentsSuperSet = new ArrayList<>();
        }

        // filter only the ones initiated by the correct user
        final List<CustomerInvoiceDocument> customerInvoiceDocuments = new ArrayList<>();
        for (final CustomerInvoiceDocument superSetDocument : customerInvoiceDocumentsSuperSet) {
            if (StringUtils.equalsIgnoreCase(superSetDocument.getDocumentHeader().getWorkflowDocument()
                    .getInitiatorPrincipalId(), person.getPrincipalId())) {
                customerInvoiceDocuments.add(superSetDocument);
            }
        }
        return customerInvoiceDocuments;
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByBillingChartAndOrg(
            final String chartOfAccountsCode, final String organizationCode) {
        final List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersByBillingChartAndOrg(chartOfAccountsCode, organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    protected List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByDocumentNumbers(
            final List<String> documentHeaderIds) {
        final List<CustomerInvoiceDocument> customerInvoiceDocuments = new ArrayList<>(documentHeaderIds.size());
        if (documentHeaderIds != null && !documentHeaderIds.isEmpty()) {
            for (final Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                    CustomerInvoiceDocument.class, documentHeaderIds)) {
                customerInvoiceDocuments.add((CustomerInvoiceDocument) doc);
            }
        }
        return customerInvoiceDocuments;
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsForBillingStatementByBillingChartAndOrg(
            final String chartOfAccountsCode, final String organizationCode) {
        final List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersForBillingStatementByBillingChartAndOrg(
                        chartOfAccountsCode, organizationCode);

        final List<CustomerInvoiceDocument> customerInvoiceDocuments = new ArrayList<>();
        if (documentHeaderIds != null && !documentHeaderIds.isEmpty()) {
            for (final Document doc : documentService.getDocumentsByListOfDocumentHeaderIds(
                    CustomerInvoiceDocument.class, documentHeaderIds)) {
                customerInvoiceDocuments.add((CustomerInvoiceDocument) doc);
            }
        }
        return customerInvoiceDocuments;
    }

    @Override
    public List<CustomerInvoiceDocument> getPrintableCustomerInvoiceDocumentsByProcessingChartAndOrg(
            final String chartOfAccountsCode, final String organizationCode) {
        final List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getPrintableCustomerInvoiceDocumentNumbersByProcessingChartAndOrg(chartOfAccountsCode,
                        organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public Collection<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByAccountNumber(final String accountNumber) {
        final List<String> documentHeaderIds = customerInvoiceDetailService
                .getCustomerInvoiceDocumentNumbersByAccountNumber(accountNumber);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByBillingChartAndOrg(
            final String chartOfAccountsCode,
            final String organizationCode) {
        final List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getCustomerInvoiceDocumentNumbersByBillingChartAndOrg(chartOfAccountsCode, organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public List<CustomerInvoiceDocument> getCustomerInvoiceDocumentsByProcessingChartAndOrg(
            final String chartOfAccountsCode, final String organizationCode) {
        final List<String> documentHeaderIds = customerInvoiceDocumentDao
                .getCustomerInvoiceDocumentNumbersByProcessingChartAndOrg(chartOfAccountsCode, organizationCode);
        return getCustomerInvoiceDocumentsByDocumentNumbers(documentHeaderIds);
    }

    @Override
    public void setupDefaultValuesForNewCustomerInvoiceDocument(final CustomerInvoiceDocument document) {
        setupBasicDefaultValuesForCustomerInvoiceDocument(document);

        // set up the default values for the AR DOC Header

        final AccountsReceivableDocumentHeader accountsReceivableDocumentHeader =
                accountsReceivableDocumentHeaderService.getNewAccountsReceivableDocumentHeaderForCurrentUser();
        accountsReceivableDocumentHeader.setDocumentNumber(document.getDocumentNumber());
        document.setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);

        // set up the primary key for AR_INV_RCURRNC_DTL_T
        final CustomerInvoiceRecurrenceDetails recurrenceDetails = new CustomerInvoiceRecurrenceDetails();
        recurrenceDetails.setInvoiceNumber(document.getDocumentNumber());
        document.setCustomerInvoiceRecurrenceDetails(recurrenceDetails);

        final Map<String, String> criteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, document.getBillByChartOfAccountCode());
        criteria.put(KFSPropertyConstants.ORGANIZATION_CODE, document.getBilledByOrganizationCode());
        final OrganizationOptions organizationOptions = businessObjectService.findByPrimaryKey(OrganizationOptions.class,
                criteria);

        if (ObjectUtils.isNotNull(organizationOptions)) {
            document.setPrintInvoiceIndicator(organizationOptions.getPrintInvoiceIndicator());
            document.setInvoiceTermsText(organizationOptions.getOrganizationPaymentTermsText());
        }
    }

    @Override
    public void loadCustomerAddressesForCustomerInvoiceDocument(final CustomerInvoiceDocument customerInvoiceDocument) {
        // if address identifier is provided, try to refresh customer address data
        if (ObjectUtils.isNotNull(customerInvoiceDocument.getAccountsReceivableDocumentHeader())) {
            final CustomerAddress customerShipToAddress = customerAddressService.getByPrimaryKey(
                    customerInvoiceDocument.getAccountsReceivableDocumentHeader().getCustomerNumber(),
                    customerInvoiceDocument.getCustomerShipToAddressIdentifier());
            final CustomerAddress customerBillToAddress = customerAddressService.getByPrimaryKey(
                    customerInvoiceDocument.getAccountsReceivableDocumentHeader().getCustomerNumber(),
                    customerInvoiceDocument.getCustomerBillToAddressIdentifier());

            if (ObjectUtils.isNotNull(customerShipToAddress)) {
                customerInvoiceDocument.setCustomerShipToAddress(customerShipToAddress);
                customerInvoiceDocument.setCustomerShipToAddressOnInvoice(customerShipToAddress);
            }

            if (ObjectUtils.isNotNull(customerBillToAddress)) {
                customerInvoiceDocument.setCustomerBillToAddress(customerBillToAddress);
                customerInvoiceDocument.setCustomerBillToAddressOnInvoice(customerBillToAddress);
            }
        }
    }

    @Override
    public void setupDefaultValuesForCopiedCustomerInvoiceDocument(final CustomerInvoiceDocument document) {
        setupBasicDefaultValuesForCustomerInvoiceDocument(document);

        // Save customer number since it will get overwritten when we retrieve the accounts receivable document
        // header from service
        final String customerNumber = document.getAccountsReceivableDocumentHeader().getCustomerNumber();

        // Set up the default values for the AR DOC Header
        final AccountsReceivableDocumentHeader accountsReceivableDocumentHeader =
                accountsReceivableDocumentHeaderService.getNewAccountsReceivableDocumentHeaderForCurrentUser();
        accountsReceivableDocumentHeader.setDocumentNumber(document.getDocumentNumber());
        accountsReceivableDocumentHeader.setCustomerNumber(customerNumber);
        document.setAccountsReceivableDocumentHeader(accountsReceivableDocumentHeader);

        // set up the primary key for AR_INV_RCURRNC_DTL_T
        final CustomerInvoiceRecurrenceDetails recurrenceDetails = new CustomerInvoiceRecurrenceDetails();
        recurrenceDetails.setInvoiceNumber(document.getDocumentNumber());
        document.setCustomerInvoiceRecurrenceDetails(recurrenceDetails);

        // make open invoice indicator to true
        document.setOpenInvoiceIndicator(true);
        document.setPrintDate(null);
        document.setBillingDate(dateTimeService.getCurrentSqlDateMidnight());
    }

    @Override
    public Collection<NonInvoicedDistribution> getNonInvoicedDistributionsForInvoice(final String documentNumber) {
        return nonInvoicedDistributionService.getNonInvoicedDistributionsForInvoice(documentNumber);
    }

    @Override
    public KualiDecimal getNonInvoicedTotalForInvoice(final CustomerInvoiceDocument invoice) {
        final Collection<NonInvoicedDistribution> payments = nonInvoicedDistributionService
                .getNonInvoicedDistributionsForInvoice(invoice);
        KualiDecimal total = new KualiDecimal(0);
        for (final NonInvoicedDistribution payment : payments) {
            total = total.add(payment.getFinancialDocumentLineAmount());
        }
        return total;
    }

    @Override
    public KualiDecimal getNonInvoicedTotalForInvoice(final String documentNumber) {
        return getNonInvoicedTotalForInvoice(getInvoiceByInvoiceDocumentNumber(documentNumber));
    }

    @Override
    public KualiDecimal getPaidAppliedTotalForInvoice(final CustomerInvoiceDocument invoice) {
        final Collection<InvoicePaidApplied> payments = invoicePaidAppliedService.getActiveInvoicePaidAppliedsForInvoice(invoice);
        KualiDecimal total = new KualiDecimal(0);
        for (final InvoicePaidApplied payment : payments) {
            total = total.add(payment.getInvoiceItemAppliedAmount());
        }
        return total;
    }

    @Override
    public KualiDecimal getPaidAppliedTotalForInvoice(final String documentNumber) {
        return getPaidAppliedTotalForInvoice(getInvoiceByInvoiceDocumentNumber(documentNumber));
    }

    /**
     * @param document
     */
    protected void setupBasicDefaultValuesForCustomerInvoiceDocument(final CustomerInvoiceDocument document) {
        final ChartOrgHolder currentUser = financialSystemUserService.getPrimaryOrganization(
                GlobalVariables.getUserSession().getPerson(), ArConstants.AR_NAMESPACE_CODE);
        if (currentUser != null) {
            document.setBillByChartOfAccountCode(currentUser.getChartOfAccountsCode());
            document.setBilledByOrganizationCode(currentUser.getOrganizationCode());
        }
        document.setInvoiceDueDate(getDefaultInvoiceDueDate());
        document.setOpenInvoiceIndicator(true);
    }

    /**
     * This method sets due date equal to today's date +30 days by default
     */
    protected Date getDefaultInvoiceDueDate() {
        final LocalDate dueDate = LocalDate.now().plusDays(30);
        final Date sqlDueDate = Date.valueOf(dueDate);
        return sqlDueDate;
    }

    @Override
    public void closeCustomerInvoiceDocument(final CustomerInvoiceDocument customerInvoiceDocument) {
        customerInvoiceDocument.setOpenInvoiceIndicator(false);
        customerInvoiceDocument.setClosedDate(dateTimeService.getCurrentSqlDate());
        businessObjectService.save(customerInvoiceDocument);
    }

    @Override
    public boolean checkIfInvoiceNumberIsFinal(final String invDocumentNumber) {
        boolean isSuccess = true;
        if (StringUtils.isBlank(invDocumentNumber)) {
            isSuccess = false;
        } else {
            final CustomerInvoiceDocument customerInvoiceDocument = getInvoiceByInvoiceDocumentNumber(invDocumentNumber);

            if (ObjectUtils.isNull(customerInvoiceDocument)) {
                isSuccess = false;
            } else {
                final Document doc = documentService.getByDocumentHeaderId(invDocumentNumber);
                if (ObjectUtils.isNull(doc) || ObjectUtils.isNull(doc.getDocumentHeader())
                        || doc.getDocumentHeader().getWorkflowDocument() == null
                        || !(doc.getDocumentHeader().getWorkflowDocument().isApproved()
                        || doc.getDocumentHeader().getWorkflowDocument().isProcessed())) {
                    isSuccess = false;
                }
            }
        }
        return isSuccess;
    }

    /**
     * get the date before the given amount of days
     */
    protected Date getPastDate(final Integer amount) {
        final int pastDateAmount = -1 * amount;

        final java.util.Date today = getDateTimeService().getCurrentDate();
        final java.util.Date pastDate = DateUtils.addDays(today, pastDateAmount);

        return KfsDateUtils.convertToSqlDate(pastDate);
    }

    @Override
    public Collection<CustomerInvoiceDocument> getAllAgingInvoiceDocumentsByCustomerTypes(
            final List<String> customerTypes,
            final Integer invoiceAge, final Date invoiceDueDateFrom) {
        final Date pastDate = getPastDate(invoiceAge - 1);
        final Date invoiceDueDateTo = KfsDateUtils.convertToSqlDate(DateUtils.addDays(pastDate, 1));
        LOG.info("invoiceDueDateTo{}", invoiceDueDateTo);

        return customerInvoiceDocumentDao.getAllAgingInvoiceDocumentsByCustomerTypes(customerTypes,
                invoiceDueDateFrom, invoiceDueDateTo);
    }

    @Override
    public void addCloseNote(final CustomerInvoiceDocument documentToClose, final WorkflowDocument closingDocument) {
        if (!documentToClose.isOpenInvoiceIndicator()) {
            // If it already is closed, no need to add a note
            return;
        }

        String principalName = "Unknown";
        final List<ActionTaken> actionsTaken = closingDocument.getActionsTaken();
        if (ObjectUtils.isNotNull(actionsTaken)) {
            for (final ActionTaken action : actionsTaken) {
                // we're looking for the person who completed the closing document, so we want the COMPLETE action
                if (isActionClose(action.getActionTaken())) {
                    principalName = getPersonService().getPerson(action.getPrincipalId()).getName();
                    break;
                }
            }
        }

        final String noteTextPattern = getConfigurationService().getPropertyValueAsString(
                ArKeyConstants.INVOICE_CLOSE_NOTE_TEXT);
        final Object[] arguments = {principalName, closingDocument.getDocumentTypeName(), closingDocument.getDocumentId()};
        final String noteText = MessageFormat.format(noteTextPattern, arguments);

        final Note note = getDocumentService().createNoteFromDocument(documentToClose, noteText);
        note.setAuthorUniversalIdentifier(personService.getPersonByPrincipalName(
                KFSConstants.SYSTEM_USER).getPrincipalId());
        documentToClose.addNote(noteService.save(note));
    }

    /**
     * Looks up all invoice paid applieds applied to this invoice owned by payment applications or customer credit
     * memos, and sums the invoice item applied amount
     */
    @Override
    public KualiDecimal calculateAppliedPaymentAmount(final CustomerInvoiceDocument invoice) {
        final Map<String, Object> criteria = new HashMap<>();
        final KualiDecimal totalPayments = KualiDecimal.ZERO;
        criteria.put(ArPropertyConstants.CustomerInvoiceDocumentFields.FINANCIAL_DOCUMENT_REF_INVOICE_NUMBER,
                invoice.getDocumentNumber());
        final List<String> allowedOwningDocumentTypes = new ArrayList<>();
        allowedOwningDocumentTypes.add(ArConstants.ArDocumentTypeCodes.CUSTOMER_CREDIT_MEMO_DOCUMENT_TYPE_CODE);
        criteria.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME,
                allowedOwningDocumentTypes);

        final Collection<InvoicePaidApplied> creditMemoInvoicePaidApplieds = businessObjectService.findMatching(
                InvoicePaidApplied.class, criteria);
        final List<InvoicePaidApplied> paymentAdjustableInvoicePaidApplieds = invoicePaidAppliedService
                .getActiveInvoicePaidAppliedsForInvoice(invoice);

        return Stream.of(creditMemoInvoicePaidApplieds, paymentAdjustableInvoicePaidApplieds)
                .flatMap(Collection::stream)
                .map(InvoicePaidApplied::getInvoiceItemAppliedAmount)
                .reduce(KualiDecimal.ZERO, KualiDecimal::add);
    }

    private boolean isActionClose(final String actionTaken) {
        return actionTaken.equals(WorkflowAction.COMPLETE.getCode())
                || actionTaken.equals(WorkflowAction.SU_BLANKET_APPROVE.getCode())
                || actionTaken.equals(WorkflowAction.BLANKET_APPROVE.getCode())
                || actionTaken.equals(WorkflowAction.SU_COMPLETE.getCode());
    }

    public CustomerInvoiceDocumentDao getCustomerInvoiceDocumentDao() {
        return customerInvoiceDocumentDao;
    }

    public void setCustomerInvoiceDocumentDao(final CustomerInvoiceDocumentDao customerInvoiceDocumentDao) {
        this.customerInvoiceDocumentDao = customerInvoiceDocumentDao;
    }

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public AccountsReceivableDocumentHeaderService getAccountsReceivableDocumentHeaderService() {
        return accountsReceivableDocumentHeaderService;
    }

    public void setAccountsReceivableDocumentHeaderService(
            final AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService) {
        this.accountsReceivableDocumentHeaderService = accountsReceivableDocumentHeaderService;
    }

    public CustomerAddressService getCustomerAddressService() {
        return customerAddressService;
    }

    public void setCustomerAddressService(final CustomerAddressService customerAddressService) {
        this.customerAddressService = customerAddressService;
    }

    public void setDocumentDao(final DocumentDao documentDao) {
        this.documentDao = documentDao;
    }

    public void setInvoicePaidAppliedService(final InvoicePaidAppliedService invoicePaidAppliedService) {
        this.invoicePaidAppliedService = invoicePaidAppliedService;
    }

    public void setNonInvoicedDistributionService(final NonInvoicedDistributionService nonInvoicedDistributionService) {
        this.nonInvoicedDistributionService = nonInvoicedDistributionService;
    }

    public void setCustomerInvoiceDetailService(final CustomerInvoiceDetailService customerInvoiceDetailService) {
        this.customerInvoiceDetailService = customerInvoiceDetailService;
    }

    public void setUniversityDateService(final UniversityDateService universityDateService) {
        this.universityDateService = universityDateService;
    }

    public UniversityDateService getUniversityDateService() {
        return universityDateService;
    }

    public void setNoteService(final NoteService noteService) {
        this.noteService = noteService;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }

    public PersonService getPersonService() {
        return personService;
    }

    public ConfigurationService getConfigurationService() {
        return configurationService;
    }

    public void setConfigurationService(final ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public FinancialSystemUserService getFinancialSystemUserService() {
        return financialSystemUserService;
    }

    public void setFinancialSystemUserService(final FinancialSystemUserService financialSystemUserService) {
        this.financialSystemUserService = financialSystemUserService;
    }
}
