/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kew.api.WorkflowDocumentFactory;
import org.kuali.kfs.kns.rules.TransactionalDocumentRuleBase;
import org.kuali.kfs.krad.UserSession;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.document.TransactionalDocument;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.validation.ContinueCustomerCreditMemoDocumentRule;
import org.kuali.kfs.module.ar.document.validation.RecalculateCustomerCreditMemoDetailRule;
import org.kuali.kfs.module.ar.document.validation.RecalculateCustomerCreditMemoDocumentRule;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntry;
import org.kuali.kfs.sys.context.SpringContext;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * This class holds the business rules for the AR Credit Memo Document
 */
public class CustomerCreditMemoDocumentRule extends TransactionalDocumentRuleBase implements
        RecalculateCustomerCreditMemoDetailRule<TransactionalDocument>,
        RecalculateCustomerCreditMemoDocumentRule<TransactionalDocument>,
        ContinueCustomerCreditMemoDocumentRule<TransactionalDocument> {
    protected static final BigDecimal ALLOWED_QTY_DEVIATION = new BigDecimal("0.10");

    private BusinessObjectService businessObjectService;
    private CustomerInvoiceDocumentService customerInvoiceDocumentService;

    @Override
    protected boolean processCustomSaveDocumentBusinessRules(final Document document) {
        boolean isValid = super.processCustomSaveDocumentBusinessRules(document);

        GlobalVariables.getMessageMap().addToErrorPath(KRADConstants.DOCUMENT_PROPERTY_NAME);
        isValid &= processRecalculateCustomerCreditMemoDocumentRules((TransactionalDocument) document, true);
        GlobalVariables.getMessageMap().removeFromErrorPath(KRADConstants.DOCUMENT_PROPERTY_NAME);

        return isValid;
    }

    @Override
    public boolean processRecalculateCustomerCreditMemoDetailRules(
            final TransactionalDocument document,
            final CustomerCreditMemoDetail customerCreditMemoDetail) {
        boolean success;

        final CustomerCreditMemoDocument customerCreditMemoDocument = (CustomerCreditMemoDocument) document;
        customerCreditMemoDocument.refreshReferenceObject("invoice");
        final String inputKey = isQtyOrItemAmountEntered(customerCreditMemoDetail);

        // refresh InvoiceOpenItemAmount and InvoiceOpenAmountQuantity if changed by any other transaction
        customerCreditMemoDetail
                .setInvoiceOpenItemAmount(customerCreditMemoDetail.getCustomerInvoiceDetail().getAmountOpen());
        customerCreditMemoDetail
                .setInvoiceOpenItemQuantity(customerCreditMemoDocument.getInvoiceOpenItemQuantity(
                        customerCreditMemoDetail, customerCreditMemoDetail.getCustomerInvoiceDetail()));

        // 'Qty' was entered
        if (StringUtils.equals(ArConstants.CustomerCreditMemoConstants.CUSTOMER_CREDIT_MEMO_ITEM_QUANTITY, inputKey)) {
            success = isValueGreaterThanZero(customerCreditMemoDetail.getCreditMemoItemQuantity());
            success &= isCustomerCreditMemoQtyLessThanEqualToInvoiceOpenQty(customerCreditMemoDetail);
        } else if (StringUtils.equals(ArConstants.CustomerCreditMemoConstants.CUSTOMER_CREDIT_MEMO_ITEM_TOTAL_AMOUNT,
                inputKey)) {
            // 'Item Amount' was entered
            success = isValueGreaterThanZero(customerCreditMemoDetail.getCreditMemoItemTotalAmount());
            success &= isCustomerCreditMemoItemAmountLessThanEqualToInvoiceOpenItemAmount(
                    customerCreditMemoDetail);
        } else if (StringUtils.equals(ArConstants.CustomerCreditMemoConstants.BOTH_QUANTITY_AND_ITEM_TOTAL_AMOUNT_ENTERED,
                inputKey)) {
            // both 'Qty' and 'Item Amount' were entered -> validate
            success = isValueGreaterThanZero(customerCreditMemoDetail.getCreditMemoItemTotalAmount());
            success &= isCustomerCreditMemoItemAmountLessThanEqualToInvoiceOpenItemAmount(
                    customerCreditMemoDetail);
            success &= isValueGreaterThanZero(customerCreditMemoDetail.getCreditMemoItemQuantity());
            success &= isCustomerCreditMemoQtyLessThanEqualToInvoiceOpenQty(customerCreditMemoDetail);
            success &= checkIfCustomerCreditMemoQtyAndCustomerCreditMemoItemAmountValid(customerCreditMemoDetail,
                    customerCreditMemoDetail.getCustomerInvoiceDetail().getInvoiceItemUnitPrice());
        } else {
            // if there is no input -> wrong input
            success = false;
        }
        return success;
    }

    protected String isQtyOrItemAmountEntered(final CustomerCreditMemoDetail customerCreditMemoDetail) {

        final BigDecimal customerCreditMemoItemQty = customerCreditMemoDetail.getCreditMemoItemQuantity();
        final KualiDecimal customerCreditMemoItemAmount = customerCreditMemoDetail.getCreditMemoItemTotalAmount();
        String inputKey = "";

        if (ObjectUtils.isNotNull(customerCreditMemoItemQty) && ObjectUtils.isNotNull(customerCreditMemoItemAmount)) {
            inputKey = ArConstants.CustomerCreditMemoConstants.BOTH_QUANTITY_AND_ITEM_TOTAL_AMOUNT_ENTERED;
        } else if (ObjectUtils.isNotNull(customerCreditMemoItemQty)) {
            inputKey = ArConstants.CustomerCreditMemoConstants.CUSTOMER_CREDIT_MEMO_ITEM_QUANTITY;
        } else if (ObjectUtils.isNotNull(customerCreditMemoItemAmount)) {
            inputKey = ArConstants.CustomerCreditMemoConstants.CUSTOMER_CREDIT_MEMO_ITEM_TOTAL_AMOUNT;
        }

        return inputKey;
    }

    protected boolean isValueGreaterThanZero(final BigDecimal value) {
        final boolean validValue = value.compareTo(BigDecimal.ZERO) > 0;
        if (!validValue) {
            GlobalVariables.getMessageMap().putError(
                    ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_ITEM_QUANTITY,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DETAIL_ITEM_QUANTITY_LESS_THAN_OR_EQUAL_TO_ZERO);
        }
        return validValue;
    }

    protected boolean isValueGreaterThanZero(final KualiDecimal value) {
        final boolean validValue = value.isPositive();
        if (!validValue) {
            GlobalVariables.getMessageMap().putError(
                    ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_ITEM_TOTAL_AMOUNT,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DETAIL_ITEM_AMOUNT_LESS_THAN_OR_EQUAL_TO_ZERO);
        }
        return validValue;
    }

    protected boolean isCustomerCreditMemoItemAmountLessThanEqualToInvoiceOpenItemAmount(
            final CustomerCreditMemoDetail customerCreditMemoDetail) {
        final KualiDecimal invoiceOpenItemAmount = customerCreditMemoDetail.getInvoiceOpenItemAmount();
        final KualiDecimal creditMemoItemAmount = customerCreditMemoDetail.getCreditMemoItemTotalAmount();

        final boolean validItemAmount = creditMemoItemAmount.isLessEqual(invoiceOpenItemAmount);
        if (!validItemAmount) {
            GlobalVariables.getMessageMap()
                    .putError(ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_ITEM_TOTAL_AMOUNT,
                            ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DETAIL_ITEM_AMOUNT_GREATER_THAN_INVOICE_ITEM_AMOUNT);
        }

        return validItemAmount;
    }

    protected boolean isCustomerCreditMemoQtyLessThanEqualToInvoiceOpenQty(
            final CustomerCreditMemoDetail customerCreditMemoDetail) {
        final BigDecimal invoiceOpenItemQty = customerCreditMemoDetail.getInvoiceOpenItemQuantity();
        final BigDecimal customerCreditMemoItemQty = customerCreditMemoDetail.getCreditMemoItemQuantity();

        // customer credit memo quantity must not be greater than invoice open item quantity
        final boolean validQuantity = customerCreditMemoItemQty.compareTo(invoiceOpenItemQty) < 1;
        if (!validQuantity) {
            GlobalVariables.getMessageMap().putError(
                    ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_ITEM_QUANTITY,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DETAIL_ITEM_QUANTITY_GREATER_THAN_INVOICE_ITEM_QUANTITY);
        }

        return validQuantity;
    }

    protected boolean checkIfCustomerCreditMemoQtyAndCustomerCreditMemoItemAmountValid(
            final CustomerCreditMemoDetail customerCreditMemoDetail, final BigDecimal unitPrice) {
        final KualiDecimal creditAmount = customerCreditMemoDetail.getCreditMemoItemTotalAmount();
        final BigDecimal creditQuantity = customerCreditMemoDetail.getCreditMemoItemQuantity();

        // if unit price is zero, leave this validation, as it will cause an exception below by attempting to
        // divide by zero
        if (unitPrice.compareTo(BigDecimal.ZERO) == 0) {
            // no need to report error, because it is already recorded by another validation check.
            return false;
        }

        // determine the expected exact total credit memo quantity, based on actual credit amount entered
        final BigDecimal expectedCreditQuantity = creditAmount.bigDecimalValue().divide(unitPrice,
                ArConstants.ITEM_QUANTITY_SCALE, RoundingMode.HALF_UP);

        // return false if the expected quantity is 0 to avoid divide by zero exception
        if (expectedCreditQuantity.compareTo(BigDecimal.ZERO) == 0) {
            return false;
        }

        // determine the deviation percentage that the actual creditQuantity has from expectedCreditQuantity
        final BigDecimal deviationPercentage = creditQuantity.subtract(expectedCreditQuantity).divide(expectedCreditQuantity,
                ArConstants.ITEM_QUANTITY_SCALE, RoundingMode.HALF_UP).abs();

        // only allow a certain deviation of creditQuantity from the expectedCreditQuantity
        final boolean validFlag = deviationPercentage.compareTo(ALLOWED_QTY_DEVIATION) < 1;

        if (!validFlag) {
            GlobalVariables.getMessageMap()
                    .putError(ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_ITEM_QUANTITY,
                            ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DETAIL_INVALID_DATA_INPUT);
            GlobalVariables.getMessageMap()
                    .putError(ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_ITEM_TOTAL_AMOUNT,
                            ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DETAIL_INVALID_DATA_INPUT);
        }
        return validFlag;
    }

    @Override
    public boolean processRecalculateCustomerCreditMemoDocumentRules(
            final TransactionalDocument document,
            final boolean printErrMsgFlag) {
        boolean success = true;
        boolean crmDataEnteredFlag = false;
        final CustomerCreditMemoDocument customerCreditMemoDocument = (CustomerCreditMemoDocument) document;
        final List<CustomerCreditMemoDetail> customerCreditMemoDetails = customerCreditMemoDocument.getCreditMemoDetails();
        int i = 0;
        String propertyName;

        for (final CustomerCreditMemoDetail customerCreditMemoDetail : customerCreditMemoDetails) {
            propertyName = KFSConstants.CUSTOMER_CREDIT_MEMO_DETAIL_PROPERTY_NAME + "[" + i + "]";
            GlobalVariables.getMessageMap().addToErrorPath(propertyName);

            // validate only if there is input data
            if (!isQtyOrItemAmountEntered(customerCreditMemoDetail).equals(StringUtils.EMPTY)) {
                crmDataEnteredFlag = true;
                success &= processRecalculateCustomerCreditMemoDetailRules(customerCreditMemoDocument,
                        customerCreditMemoDetail);
            }
            GlobalVariables.getMessageMap().removeFromErrorPath(propertyName);
            i++;
        }

        success &= crmDataEnteredFlag;

        // print error message if 'Submit'/'Save'/'Blanket Approved' button is pressed and there is no CRM data entered
        if (!crmDataEnteredFlag && printErrMsgFlag) {
            GlobalVariables.getMessageMap().putError(KFSConstants.DOCUMENT_PROPERTY_NAME,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DOCUMENT_NO_DATA_TO_SUBMIT);
        }

        return success;
    }

    @Override
    public boolean processContinueCustomerCreditMemoDocumentRules(final TransactionalDocument document) {
        boolean success;
        final CustomerCreditMemoDocument customerCreditMemoDocument = (CustomerCreditMemoDocument) document;

        success = checkIfInvoiceNumberIsFinal(customerCreditMemoDocument.getFinancialDocumentReferenceInvoiceNumber());
        if (success) {
            success = checkIfThereIsNoAnotherCRMInRouteForTheInvoice(
                    customerCreditMemoDocument.getFinancialDocumentReferenceInvoiceNumber());
        }
        if (success) {
            success = checkInvoiceForErrorCorrection(
                    customerCreditMemoDocument.getFinancialDocumentReferenceInvoiceNumber());
        }

        return success;
    }

    protected boolean checkIfInvoiceNumberIsFinal(final String invDocumentNumber) {
        boolean success = true;

        if (StringUtils.isBlank(invDocumentNumber)) {
            success = false;
            GlobalVariables.getMessageMap().putError(
                    ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_DOCUMENT_REF_INVOICE_NUMBER,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DOCUMENT__INVOICE_DOCUMENT_NUMBER_IS_REQUIRED);
        } else {
            final CustomerInvoiceDocument customerInvoiceDocument = getCustomerInvoiceDocumentService()
                    .getInvoiceByInvoiceDocumentNumber(invDocumentNumber);
            if (ObjectUtils.isNull(customerInvoiceDocument)) {
                success = false;
                GlobalVariables.getMessageMap().putError(
                        ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_DOCUMENT_REF_INVOICE_NUMBER,
                        ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DOCUMENT_INVALID_INVOICE_DOCUMENT_NUMBER);
            } else if (!getCustomerInvoiceDocumentService().checkIfInvoiceNumberIsFinal(invDocumentNumber)) {
                GlobalVariables.getMessageMap().putError(
                        ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_DOCUMENT_REF_INVOICE_NUMBER,
                        ArKeyConstants.ERROR_CUSTOMER_INVOICE_DOCUMENT_NOT_FINAL);
                success = false;
            }
        }
        return success;
    }

    /**
     * This method checks if there is another CRM in route for the invoice
     *
     * @param invoiceDocumentNumber used to find the invoice document to check for
     * @return true if there is not another CRM in route for the invoice, false otherwise
     */
    protected boolean checkIfThereIsNoAnotherCRMInRouteForTheInvoice(final String invoiceDocumentNumber) {
        WorkflowDocument workflowDocument;
        boolean success = true;

        final Map<String, String> fieldValues = new HashMap<>();
        fieldValues.put("financialDocumentReferenceInvoiceNumber", invoiceDocumentNumber);

        final Collection<CustomerCreditMemoDocument> customerCreditMemoDocuments = getBusinessObjectService()
            .findMatching(CustomerCreditMemoDocument.class, fieldValues);

        // no CRMs associated with the invoice are found
        if (customerCreditMemoDocuments.isEmpty()) {
            return true;
        }

        final String userId = getUserSession().getPrincipalId();
        for (final CustomerCreditMemoDocument customerCreditMemoDocument : customerCreditMemoDocuments) {
            workflowDocument = loadWorkflowDocument(userId, customerCreditMemoDocument.getDocumentNumber());
            if (!(workflowDocument.isApproved() || workflowDocument.isProcessed() ||
                    workflowDocument.isCanceled() || workflowDocument.isDisapproved())) {
                GlobalVariables.getMessageMap().putError(
                    ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_DOCUMENT_REF_INVOICE_NUMBER,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DOCUMENT_ONE_CRM_IN_ROUTE_PER_INVOICE);
                success = false;
                break;
            }
        }
        return success;
    }

    /**
     * This method checks if the Invoice has been error corrected or is an error correcting invoice
     *
     * @param invoiceDocumentNumber
     * @return
     */
    protected boolean checkInvoiceForErrorCorrection(final String invoiceDocumentNumber) {
        final CustomerInvoiceDocumentService service = getCustomerInvoiceDocumentService();
        final CustomerInvoiceDocument customerInvoiceDocument =
                service.getInvoiceByInvoiceDocumentNumber(invoiceDocumentNumber);

        // invoice has been corrected
        if (customerInvoiceDocument.hasInvoiceBeenCorrected()) {
            GlobalVariables.getMessageMap().putError(
                    ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_DOCUMENT_REF_INVOICE_NUMBER,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DOCUMENT_CORRECTED_INVOICE);
            return false;
        }
        // this is a correcting invoice
        if (customerInvoiceDocument.isInvoiceReversal()) {
            GlobalVariables.getMessageMap().putError(
                    ArPropertyConstants.CustomerCreditMemoDocumentFields.CREDIT_MEMO_DOCUMENT_REF_INVOICE_NUMBER,
                    ArKeyConstants.ERROR_CUSTOMER_CREDIT_MEMO_DOCUMENT_CORRECTING_INVOICE);
            return false;
        }
        return true;
    }

    @Override
    public boolean isDocumentAttributesValid(final Document document, final boolean validateRequired) {
        //refresh GLPE nonupdateable business object references....
        final CustomerCreditMemoDocument customerCreditMemoDocument = (CustomerCreditMemoDocument) document;

        for (final CustomerCreditMemoDetail customerDetail : customerCreditMemoDocument.getCreditMemoDetails()) {
            customerDetail.getCustomerInvoiceDetail().refreshNonUpdateableReferences();
        }

        for (final GeneralLedgerPendingEntry glpe : customerCreditMemoDocument.getGeneralLedgerPendingEntries()) {
            glpe.refreshNonUpdateableReferences();
        }

        return super.isDocumentAttributesValid(document, validateRequired);
    }

    private BusinessObjectService getBusinessObjectService() {
        if (businessObjectService == null) {
            businessObjectService = SpringContext.getBean(BusinessObjectService.class);
        }
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    private CustomerInvoiceDocumentService getCustomerInvoiceDocumentService() {
        if (customerInvoiceDocumentService == null) {
            customerInvoiceDocumentService = SpringContext.getBean(CustomerInvoiceDocumentService.class);
        }
        return customerInvoiceDocumentService;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    /*
     * Wrapping static utility class in a method so tests can use a spy to mock this call; this way,
     * static mocking is not necessary.
     */
    UserSession getUserSession() {
        return GlobalVariables.getUserSession();
    }

    /*
     * Wrapping static utility class in a method so tests can use a spy to mock this call; this way,
     * static mocking is not necessary.
     */
    WorkflowDocument loadWorkflowDocument(final String userId, final String documentId) {
        return WorkflowDocumentFactory.loadDocument(userId, documentId);
    }

}
