/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.validation.impl;

import org.kuali.kfs.coa.businessobject.AccountingPeriod;
import org.kuali.kfs.coa.service.AccountingPeriodService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.batch.service.VerifyBillingFrequencyService;
import org.kuali.kfs.module.ar.businessobject.InvoiceGeneralDetail;
import org.kuali.kfs.module.ar.businessobject.BillingPeriod;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.validation.SuspensionCategoryBase;

import java.util.List;

public class InvoiceBillingPeriodSuspensionCategory extends SuspensionCategoryBase {

    private AccountingPeriodService accountingPeriodService;
    private DateTimeService dateTimeService;
    private VerifyBillingFrequencyService verifyBillingFrequencyService;

    @Override
    public boolean shouldSuspend(final ContractsGrantsInvoiceDocument contractsGrantsInvoiceDocument) {
        final InvoiceGeneralDetail invoiceGeneralDetail = contractsGrantsInvoiceDocument.getInvoiceGeneralDetail();
        final ContractsAndGrantsBillingAward award = contractsGrantsInvoiceDocument.getInvoiceGeneralDetail().getAward();

        if (isBillingFrequencyQuaterlySemiAnnuallyOrAnnually(invoiceGeneralDetail.getBillingFrequencyCode())) {
            final AccountingPeriod currentPeriod = accountingPeriodService
                    .getByDate(dateTimeService.getCurrentSqlDate());
            final AccountingPeriod previousPeriod = accountingPeriodService.getPreviousAccountingPeriod(currentPeriod);
            final BillingPeriod billingPeriod = verifyBillingFrequencyService
                    .getStartDateAndEndDateOfPreviousBillingPeriod(award, currentPeriod);

            // Check against previous period to allow 1 billing month; if previous accounting period's end date is after the
            // billing period end date of the invoice, this means system may potentially retrieve balance from the month outside of the
            // billing period - then it should suspend the invoice
            return previousPeriod.getUniversityFiscalPeriodEndDate().after(billingPeriod.getEndDate());
        }
        return false;
    }

    protected boolean isBillingFrequencyQuaterlySemiAnnuallyOrAnnually(final String billingFrequency) {
        return List.of(ArConstants.BillingFrequencyValues.QUARTERLY.getCode(),
                       ArConstants.BillingFrequencyValues.SEMI_ANNUALLY.getCode(),
                       ArConstants.BillingFrequencyValues.ANNUALLY.getCode()).contains(billingFrequency);
    }

    public void setAccountingPeriodService(final AccountingPeriodService accountingPeriodService) {
        this.accountingPeriodService = accountingPeriodService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setVerifyBillingFrequencyService(final VerifyBillingFrequencyService verifyBillingFrequencyService) {
        this.verifyBillingFrequencyService = verifyBillingFrequencyService;
    }
}
