/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.document.web.struts;

import org.kuali.kfs.core.web.format.CurrencyFormatter;
import org.kuali.kfs.kew.api.WorkflowDocument;
import org.kuali.kfs.kns.service.DocumentHelperService;
import org.kuali.kfs.kns.web.ui.ExtraButton;
import org.kuali.kfs.kns.web.ui.HeaderField;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.module.ar.ArAuthorizationConstants;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceItemCode;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.authorization.CustomerInvoiceDocumentPresentationController;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.businessobject.UnitOfMeasure;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.AccountingDocument;
import org.kuali.kfs.sys.web.struts.KualiAccountingDocumentFormBase;

import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class CustomerInvoiceForm extends KualiAccountingDocumentFormBase {

    protected CustomerInvoiceDetail newCustomerInvoiceDetail;

    /**
     * Constructs a CustomerInvoiceForm.java. Also sets new customer invoice document detail to a newly constructed
     * customer invoice detail.
     */
    public CustomerInvoiceForm() {
        super();
    }

    @Override
    protected String getDefaultDocumentTypeName() {
        return ArConstants.ArDocumentTypeCodes.INV_DOCUMENT_TYPE;
    }

    public CustomerInvoiceDocument getCustomerInvoiceDocument() {
        return (CustomerInvoiceDocument) getDocument();
    }

    @Override
    public void populate(final HttpServletRequest request) {
        super.populate(request);
        SpringContext.getBean(CustomerInvoiceDocumentService.class)
                .loadCustomerAddressesForCustomerInvoiceDocument(getCustomerInvoiceDocument());
    }

    /**
     * Reused to create new source accounting line (i.e customer invoice detail line) with defaulted values.
     */
    @Override
    protected SourceAccountingLine createNewSourceAccountingLine(final AccountingDocument financialDocument) {
        if (financialDocument == null) {
            throw new IllegalArgumentException("invalid (null) document");
        }
        return SpringContext.getBean(CustomerInvoiceDetailService.class)
                .getCustomerInvoiceDetailFromOrganizationAccountingDefaultForCurrentYear();
    }

    /**
     * By overriding this method, we can add the invoice total and open amount to the document header.
     *
     * @see org.kuali.kfs.kns.web.struts.form.KualiForm#getDocInfo()
     */
    @Override
    public void populateHeaderFields(final WorkflowDocument workflowDocument) {
        super.populateHeaderFields(workflowDocument);
        getDocInfo().add(new HeaderField("DataDictionary.CustomerInvoiceDocument.attributes.sourceTotal",
                (String) new CurrencyFormatter().format(getCustomerInvoiceDocument().getSourceTotal())));
        getDocInfo().add(new HeaderField("DataDictionary.CustomerInvoiceDocument.attributes.openAmount",
                (String) new CurrencyFormatter().format(getCustomerInvoiceDocument().getOpenAmount())));
    }

    /**
     * Configure lookup for Invoice Item Code source accounting line
     */
    @Override
    public Map getForcedLookupOptionalFields() {
        final Map forcedLookupOptionalFields = super.getForcedLookupOptionalFields();

        forcedLookupOptionalFields.put(ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_ITEM_CODE,
                ArPropertyConstants.CustomerInvoiceDocumentFields.INVOICE_ITEM_CODE + ";" +
                        CustomerInvoiceItemCode.class.getName());
        forcedLookupOptionalFields.put(ArPropertyConstants.CustomerInvoiceDocumentFields.UNIT_OF_MEASURE_CODE,
                ArPropertyConstants.CustomerInvoiceDocumentFields.UNIT_OF_MEASURE_CODE + ";" +
                        UnitOfMeasure.class.getName());

        return forcedLookupOptionalFields;
    }

    /**
     * Make amount and sales tax read only
     *
     * @see org.kuali.kfs.kns.web.struts.form.KualiTransactionalDocumentFormBase#getForcedReadOnlyFields()
     */
    @Override
    public Map getForcedReadOnlyFields() {
        final Map map = super.getForcedReadOnlyFields();
        map.put(KFSPropertyConstants.AMOUNT, Boolean.TRUE);
        map.put("invoiceItemTaxAmount", Boolean.TRUE);
        map.put("openAmount", Boolean.TRUE);
        return map;
    }

    /**
     * Build additional customer invoice specific buttons and set extraButtons list.
     *
     * @return list of extra buttons to be displayed to the user
     */
    @Override
    public List<ExtraButton> getExtraButtons() {
        // clear out the extra buttons array
        extraButtons.clear();

        //  get the edit modes from the preso controller
        final CustomerInvoiceDocument invoiceDocument = (CustomerInvoiceDocument) getDocument();
        final DocumentHelperService docHelperService = SpringContext.getBean(DocumentHelperService.class);
        final CustomerInvoiceDocumentPresentationController presoController =
            (CustomerInvoiceDocumentPresentationController) docHelperService.getDocumentPresentationController(
                    invoiceDocument);
        final Set<String> editModes = presoController.getEditModes(invoiceDocument);

        //  draw the Print File button if appropriate
        if (editModes.contains(ArAuthorizationConstants.CustomerInvoiceDocumentEditMode.DISPLAY_PRINT_BUTTON)) {
            final String printButtonURL = getConfigurationService().getPropertyValueAsString(
                    KFSConstants.EXTERNALIZABLE_IMAGES_URL_KEY);

            final ExtraButton newButton = new ExtraButton();

            newButton.setExtraButtonProperty("methodToCall.print");
            newButton.setExtraButtonSource(printButtonURL + "buttonsmall_genprintfile.gif");
            newButton.setExtraButtonAltText("Print");
            newButton.setExtraButtonOnclick("excludeSubmitRestriction=true");

            extraButtons.add(newButton);
        }

        //  draw the Error Correction button if appropriate
        if (presoController.canErrorCorrect(invoiceDocument)) {
            extraButtons.add(generateErrorCorrectionButton());
        }

        return extraButtons;
    }

    /**
     * Adds a new button to the extra buttons collection.
     *
     * @param property property for button
     * @param source   location of image
     * @param altText  alternate text for button if images don't appear
     */
    protected void addExtraButton(final String property, final String source, final String altText) {
        final ExtraButton newButton = new ExtraButton();

        newButton.setExtraButtonProperty(property);
        newButton.setExtraButtonSource(source);
        newButton.setExtraButtonAltText(altText);

        extraButtons.add(newButton);
    }

    @Override
    public boolean shouldMethodToCallParameterBeUsed(
            final String methodToCallParameterName,
            final String methodToCallParameterValue, final HttpServletRequest request) {
        if (KRADConstants.DISPATCH_REQUEST_PARAMETER.equals(methodToCallParameterName)
                && ArConstants.PRINT_INVOICE_PDF_METHOD.equals(methodToCallParameterValue)) {
            return true;
        }
        return super.shouldMethodToCallParameterBeUsed(methodToCallParameterName, methodToCallParameterValue, request);
    }
}
