/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.identity;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.integration.cg.ContractsAndGrantsFundManager;
import org.kuali.kfs.kns.kim.role.DerivedRoleTypeServiceBase;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.core.api.membership.MemberType;
import org.kuali.kfs.kim.api.KimConstants;
import org.kuali.kfs.kim.api.role.RoleMembership;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Contains methods to support the Funds Manager derived role: to return the active fund managers
 * of a given award and determine if a given principal has the Funds Manager role.
 */
public class FundsManagerDerivedRoleTypeServiceImpl extends DerivedRoleTypeServiceBase {

    protected KualiModuleService kualiModuleService;

    @Override
    public List<RoleMembership> getRoleMembersFromDerivedRole(
            final String namespaceCode, final String roleName, final Map<String,
            String> qualification) {
        final List<RoleMembership> roleMembers = new ArrayList<>();

        if (ObjectUtils.isNotNull(qualification) && !qualification.isEmpty()) {
            final String principalId = qualification.get(KimConstants.AttributeConstants.PRINCIPAL_ID);
            final String proposalNumber = qualification.get(KFSPropertyConstants.PROPOSAL_NUMBER);
            if (StringUtils.isNotBlank(proposalNumber)) {
                if (StringUtils.isNotBlank(principalId)) {
                    return getRoleMembersForPrincipalAndAward(principalId, proposalNumber);
                }
                return getRoleMembersForAward(proposalNumber);
            } else if (StringUtils.isNotBlank(principalId)) {
                return getRoleMembersForPrincipal(principalId);
            }
        }

        return roleMembers;
    }

    @Override
    public boolean hasDerivedRole(
            final String principalId, final List<String> groupIds, final String namespaceCode, final String roleName,
            final Map<String, String> qualification) {
        if (ObjectUtils.isNotNull(qualification) && !qualification.isEmpty()) {
            final String proposalNumber = qualification.get(KFSPropertyConstants.PROPOSAL_NUMBER);
            if (StringUtils.isNotBlank(proposalNumber)) {
                return CollectionUtils.isNotEmpty(getRoleMembersForPrincipalAndAward(principalId, proposalNumber));
            }
        }
        return CollectionUtils.isNotEmpty(getRoleMembersForPrincipal(principalId));
    }

    private List<RoleMembership> getRoleMembersForAward(final String proposalNumber) {
        final Map<String, Object> map = new HashMap<>();
        map.put(KFSPropertyConstants.PROPOSAL_NUMBER, proposalNumber);
        return getRoleMembers(map);
    }

    private List<RoleMembership> getRoleMembersForPrincipal(final String principalId) {
        final Map<String, Object> map = new HashMap<>();
        map.put(KimConstants.AttributeConstants.PRINCIPAL_ID, principalId);
        return getRoleMembers(map);
    }

    private List<RoleMembership> getRoleMembersForPrincipalAndAward(final String principalId, final String proposalNumber) {
        final Map<String, Object> map = new HashMap<>();
        map.put(KimConstants.AttributeConstants.PRINCIPAL_ID, principalId);
        map.put(KFSPropertyConstants.PROPOSAL_NUMBER, proposalNumber);
        return getRoleMembers(map);
    }

    private List<RoleMembership> getRoleMembers(final Map<String, Object> criteria) {
        final List<RoleMembership> roleMembers = new ArrayList<>();

        criteria.put(KFSPropertyConstants.ACTIVE, true);

        final List<ContractsAndGrantsFundManager> awardFundManagers =
                kualiModuleService.getResponsibleModuleService(ContractsAndGrantsFundManager.class)
                        .getExternalizableBusinessObjectsList(ContractsAndGrantsFundManager.class, criteria);
        for (final ContractsAndGrantsFundManager awardFundManager : awardFundManagers) {
            roleMembers.add(RoleMembership.Builder.create(null, null, awardFundManager.getPrincipalId(),
                    MemberType.PRINCIPAL, null).build());
        }

        return roleMembers;
    }

    public KualiModuleService getKualiModuleService() {
        return kualiModuleService;
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }
}
