/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.report.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.cg.ContractsAndGrantsAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAward;
import org.kuali.kfs.integration.cg.ContractsAndGrantsModuleBillingService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.CollectionActivityReport;
import org.kuali.kfs.module.ar.businessobject.CollectionActivityType;
import org.kuali.kfs.module.ar.businessobject.CollectionEvent;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.ContractsGrantsInvoiceDocumentService;
import org.kuali.kfs.module.ar.report.service.CollectionActivityReportService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.document.service.FinancialSystemDocumentService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This class is used to get the services for PDF generation and other services for Collection Activity Report
 */
public class CollectionActivityReportServiceImpl implements CollectionActivityReportService {

    protected ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService;
    protected ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService;
    protected BusinessObjectService businessObjectService;
    protected DateTimeService dateTimeService;
    protected FinancialSystemDocumentService financialSystemDocumentService;
    protected PersonService personService;
    private static final Logger LOG = LogManager.getLogger();

    @Override
    @Transactional
    public List<CollectionActivityReport> filterEventsForCollectionActivity(final Map lookupFormFields) {
        final List<CollectionActivityReport> displayList = new ArrayList<>();

        final Map<String, String> fieldValues = new HashMap<>();

        final String collectorPrincName = (String) lookupFormFields.get(ArPropertyConstants.COLLECTOR_PRINC_NAME);
        final String proposalNumber = (String) lookupFormFields.get(KFSPropertyConstants.PROPOSAL_NUMBER);
        final String agencyNumber = (String) lookupFormFields.get(ArPropertyConstants.CollectionActivityReportFields.AGENCY_NUMBER);
        final String invoiceNumber = (String) lookupFormFields.get(ArPropertyConstants.INVOICE_NUMBER);
        final String accountNumber = (String) lookupFormFields.get(KFSPropertyConstants.ACCOUNT_NUMBER);
        final String activityType = (String) lookupFormFields.get(ArPropertyConstants.CollectionActivityReportFields.ACTIVITY_TYPE);

        // Getting final docs
        fieldValues.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_DOCUMENT_STATUS_CODE,
                StringUtils.join(getFinancialSystemDocumentService().getSuccessfulDocumentStatuses(), "|"));
        fieldValues.put(KFSPropertyConstants.DOCUMENT_HEADER + "." + KFSPropertyConstants.WORKFLOW_DOCUMENT_TYPE_NAME,
                ArConstants.ArDocumentTypeCodes.CONTRACTS_GRANTS_INVOICE);

        if (StringUtils.isNotBlank(proposalNumber)) {
            fieldValues.put(ArPropertyConstants.ContractsGrantsInvoiceDocumentFields.PROPOSAL_NUMBER, proposalNumber);
        }

        if (StringUtils.isNotBlank(invoiceNumber)) {
            fieldValues.put(KFSPropertyConstants.DOCUMENT_NUMBER, invoiceNumber);
        }

        if (StringUtils.isNotBlank(accountNumber)) {
            fieldValues.put(ArPropertyConstants.CustomerInvoiceDocumentFields.ACCOUNT_NUMBER, accountNumber);
        }

        // Filter Invoice docs according to criteria.
        final Collection<ContractsGrantsInvoiceDocument> contractsGrantsInvoiceDocs =
                contractsGrantsInvoiceDocumentService.retrieveAllCGInvoicesByCriteria(fieldValues);

        String collectorPrincipalId = null;
        if (!CollectionUtils.isEmpty(contractsGrantsInvoiceDocs)) {
            if (StringUtils.isNotEmpty(collectorPrincName.trim())) {
                final Person collUser = personService.getPersonByPrincipalName(collectorPrincName);
                if (ObjectUtils.isNotNull(collUser)) {
                    collectorPrincipalId = collUser.getPrincipalId();
                }
            }

            Set<String> agencyNumbers = null;
            if (StringUtils.isNotBlank(agencyNumber)) {
                agencyNumbers = getMatchingAgencyNumbers(agencyNumber);
            }

            for (final Iterator<ContractsGrantsInvoiceDocument> iter = contractsGrantsInvoiceDocs.iterator(); iter.hasNext(); ) {
                final ContractsGrantsInvoiceDocument document = iter.next();

                if (!canDocumentBeViewed(document, collectorPrincipalId)) {
                    iter.remove();
                } else if (!CollectionUtils.isEmpty(agencyNumbers) && !matchesAgencyNumber(document, agencyNumbers)) {
                    iter.remove();
                }
            }
        }

        if (!CollectionUtils.isEmpty(contractsGrantsInvoiceDocs)) {
            for (final ContractsGrantsInvoiceDocument cgInvoiceDocument : contractsGrantsInvoiceDocs) {
                final List<CollectionEvent> events = cgInvoiceDocument.getCollectionEvents();
                final List<CustomerInvoiceDetail> details = cgInvoiceDocument.getSourceAccountingLines();
                final String accountNum = !CollectionUtils.isEmpty(details)
                        && ObjectUtils.isNotNull(details.get(0)) ? details.get(0).getAccountNumber() : "";
                if (CollectionUtils.isNotEmpty(events)) {
                    for (final CollectionEvent event : events) {
                        if (eventMatchesSearchCriteria(activityType, event, collectorPrincName, collectorPrincipalId)) {
                            final CollectionActivityReport collectionActivityReport = new CollectionActivityReport();
                            collectionActivityReport.setAccountNumber(accountNum);
                            convertEventToCollectionActivityReport(collectionActivityReport, event);
                            displayList.add(collectionActivityReport);
                        }
                    }
                }
            }
        }

        return displayList;
    }

    private boolean eventMatchesSearchCriteria(
            final String activityType, final CollectionEvent event, final String collectorPrincName,
            final String collectorPrincipalId) {
        final boolean activityCodeMatches = StringUtils.isBlank(activityType) ||
                StringUtils.equals(event.getActivityCode(), activityType);
        final boolean collectorMatches = StringUtils.isBlank(collectorPrincName) ||
                StringUtils.equals(event.getUserPrincipalId(), collectorPrincipalId);
        return activityCodeMatches && collectorMatches;
    }

    /**
     * Determines if the given CINV can be viewed for the given collector and by the current user
     *
     * @param document             the CINV document to test
     * @param collectorPrincipalId the collector principal id to test if it exists
     * @return true if the document can be viewed, false otherwise
     */
    protected boolean canDocumentBeViewed(final ContractsGrantsInvoiceDocument document, final String collectorPrincipalId) {
        final Person user = GlobalVariables.getUserSession().getPerson();
        return (StringUtils.isBlank(collectorPrincipalId)
                || contractsGrantsInvoiceDocumentService.canViewInvoice(document, collectorPrincipalId))
            && contractsGrantsInvoiceDocumentService.canViewInvoice(document, user.getPrincipalId());
    }

    /**
     * Assuming that the given agencyNumberLookup may have wildcard characters, attempts to look up all matching agency numbers
     *
     * @param agencyNumberLookup the agency number from the lookup to find agency numbers on actual awards for
     * @return any matching agency numbers from matching awards
     */
    protected Set<String> getMatchingAgencyNumbers(final String agencyNumberLookup) {
        final Set<String> matchingAgencyNumbers = new HashSet<>();

        final Map<String, String> agencyLookupFields = new HashMap<>();
        agencyLookupFields.put(KFSPropertyConstants.AGENCY_NUMBER, agencyNumberLookup);
        final List<? extends ContractsAndGrantsAward> awards = getContractsAndGrantsModuleBillingService().lookupAwards(agencyLookupFields, true);
        if (!CollectionUtils.isEmpty(awards)) {
            for (final ContractsAndGrantsAward award : awards) {
                if (award instanceof ContractsAndGrantsBillingAward) {
                    matchingAgencyNumbers.add(((ContractsAndGrantsBillingAward) award).getAgencyNumber());
                }
            }
        }
        if (matchingAgencyNumbers.isEmpty()) {
            return null;
        }

        return matchingAgencyNumbers;
    }

    /**
     * Determines if the given document matches the passed in agency number
     *
     * @param document     the document to check
     * @param agencyNumbers the agency numbers to verify against
     * @return true if the document matches the given agency number, false otherwise
     */
    protected boolean matchesAgencyNumber(final ContractsGrantsInvoiceDocument document, final Set<String> agencyNumbers) {
        if (ObjectUtils.isNotNull(document) && ObjectUtils.isNotNull(document.getInvoiceGeneralDetail().getAward())
                && StringUtils.isNotBlank(document.getInvoiceGeneralDetail().getAward().getAgencyNumber())) {
            final String documentAgencyNumber = document.getInvoiceGeneralDetail().getAward().getAgencyNumber();
            return agencyNumbers.contains(documentAgencyNumber);
        }
        return false;
    }

    /**
     * This method is used to convert the CollectionEvent Object into collection activity report.
     *
     * @param collectionActivityReport
     * @param collectionEvent
     * @return Returns the Object of CollectionActivityReport class.
     */
    protected CollectionActivityReport convertEventToCollectionActivityReport(
            final CollectionActivityReport collectionActivityReport, final CollectionEvent collectionEvent) {
        if (ObjectUtils.isNull(collectionEvent)) {
            LOG.error("an invalid(null) argument was given");
            throw new IllegalArgumentException("an invalid(null) argument was given");
        }

        collectionActivityReport.setEventId(collectionEvent.getId());
        collectionActivityReport.setCollectionEventCode(collectionEvent.getCollectionEventCode());

        // account no
        collectionActivityReport.setInvoiceNumber(collectionEvent.getInvoiceNumber());
        collectionActivityReport.setActivityDate(collectionEvent.getActivityDate());

        // Activity Type
        final CollectionActivityType collectionActivityType = collectionEvent.getCollectionActivityType();

        if (ObjectUtils.isNotNull(collectionActivityType)) {
            collectionActivityReport.setActivityType(collectionActivityType.getActivityDescription());
        }

        collectionActivityReport.setActivityComment(collectionEvent.getActivityText());
        collectionActivityReport.setFollowupDate(collectionEvent.getFollowupDate());
        collectionActivityReport.setProposalNumber(collectionEvent.getInvoiceDocument().getInvoiceGeneralDetail().getProposalNumber());
        collectionActivityReport.setCompletedDate(collectionEvent.getCompletedDate());

        final String userPrincId = collectionEvent.getUserPrincipalId();
        final Person person = personService.getPerson(userPrincId);

        if (ObjectUtils.isNotNull(person)) {
            collectionActivityReport.setUserName(person.getName());
        }

        if (ObjectUtils.isNotNull(collectionEvent.getInvoiceDocument())) {
            collectionActivityReport.setChartOfAccountsCode(collectionEvent.getInvoiceDocument().getBillByChartOfAccountCode());
        }
        if (ObjectUtils.isNotNull(collectionEvent.getInvoiceDocument().getInvoiceGeneralDetail())
                && ObjectUtils.isNotNull(collectionEvent.getInvoiceDocument().getInvoiceGeneralDetail().getAward())
                && ObjectUtils.isNotNull(collectionEvent.getInvoiceDocument().getInvoiceGeneralDetail().getAward().getAgency())) {
            collectionActivityReport.setAgencyNumber(collectionEvent.getInvoiceDocument().getInvoiceGeneralDetail().getAward().getAgency().getAgencyNumber());
            collectionActivityReport.setAgencyName(collectionEvent.getInvoiceDocument().getInvoiceGeneralDetail().getAward().getAgency().getFullName());
        }
        return collectionActivityReport;
    }

    public PersonService getPersonService() {
        return personService;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }

    public FinancialSystemDocumentService getFinancialSystemDocumentService() {
        return financialSystemDocumentService;
    }

    public void setFinancialSystemDocumentService(final FinancialSystemDocumentService financialSystemDocumentService) {
        this.financialSystemDocumentService = financialSystemDocumentService;
    }

    public ContractsAndGrantsModuleBillingService getContractsAndGrantsModuleBillingService() {
        return contractsAndGrantsModuleBillingService;
    }

    public void setContractsAndGrantsModuleBillingService(final ContractsAndGrantsModuleBillingService contractsAndGrantsModuleBillingService) {
        this.contractsAndGrantsModuleBillingService = contractsAndGrantsModuleBillingService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public ContractsGrantsInvoiceDocumentService getContractsGrantsInvoiceDocumentService() {
        return contractsGrantsInvoiceDocumentService;
    }

    public void setContractsGrantsInvoiceDocumentService(final ContractsGrantsInvoiceDocumentService contractsGrantsInvoiceDocumentService) {
        this.contractsGrantsInvoiceDocumentService = contractsGrantsInvoiceDocumentService;
    }
}
