/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.service.impl;

import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomer;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerAddress;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerAddressEmail;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerCreditMemo;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerInvoice;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerInvoiceDetail;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerInvoiceRecurrenceDetails;
import org.kuali.kfs.integration.ar.AccountsReceivableCustomerType;
import org.kuali.kfs.integration.ar.AccountsReceivableDocumentHeader;
import org.kuali.kfs.integration.ar.AccountsReceivableModuleService;
import org.kuali.kfs.integration.ar.AccountsReceivableOrganizationOptions;
import org.kuali.kfs.integration.ar.AccountsReceivableSystemInformation;
import org.kuali.kfs.integration.cg.ContractsAndGrantsBillingAgency;
import org.kuali.kfs.kns.lookup.Lookupable;
import org.kuali.kfs.krad.bo.BusinessObject;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiModuleService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArPropertyConstants.CustomerTypeFields;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.CustomerAddressEmail;
import org.kuali.kfs.module.ar.businessobject.CustomerCreditMemoDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceDetail;
import org.kuali.kfs.module.ar.businessobject.CustomerInvoiceRecurrenceDetails;
import org.kuali.kfs.module.ar.businessobject.CustomerType;
import org.kuali.kfs.module.ar.businessobject.OrganizationOptions;
import org.kuali.kfs.module.ar.document.CustomerCreditMemoDocument;
import org.kuali.kfs.module.ar.document.CustomerInvoiceDocument;
import org.kuali.kfs.module.ar.document.service.AccountsReceivableDocumentHeaderService;
import org.kuali.kfs.module.ar.document.service.AccountsReceivablePendingEntryService;
import org.kuali.kfs.module.ar.document.service.CustomerAddressService;
import org.kuali.kfs.module.ar.document.service.CustomerCreditMemoDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDetailService;
import org.kuali.kfs.module.ar.document.service.CustomerInvoiceDocumentService;
import org.kuali.kfs.module.ar.document.service.CustomerService;
import org.kuali.kfs.module.ar.document.service.SystemInformationService;
import org.kuali.kfs.module.ar.service.CustomerDocumentService;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.service.ElectronicPaymentClaimingDocumentGenerationStrategy;
import org.kuali.kfs.sys.service.FinancialSystemUserService;

import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The KFS AR module implementation of the AccountsReceivableModuleService
 */
public class AccountsReceivableModuleServiceImpl implements AccountsReceivableModuleService {

    protected AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService;
    protected AccountsReceivablePendingEntryService accountsReceivablePendingEntryService;
    protected BusinessObjectService businessObjectService;
    protected CustomerCreditMemoDetailService customerCreditMemoDetailService;
    protected CustomerDocumentService customerDocumentService;
    protected CustomerInvoiceDetailService customerInvoiceDetailService;
    protected CustomerInvoiceDocumentService customerInvoiceDocumentService;
    protected Lookupable customerLookupable;
    protected CustomerService customerService;
    protected CustomerAddressService customerAddressService;
    protected DocumentService documentService;
    protected ElectronicPaymentClaimingDocumentGenerationStrategy electronicPaymentClaimingDocumentGenerationStrategy;
    protected FinancialSystemUserService financialSystemUserService;
    protected KualiModuleService kualiModuleService;
    protected ParameterService parameterService;
    protected SystemInformationService systemInformationService;

    public CustomerAddressService getCustomerAddressService() {
        return customerAddressService;
    }

    public void setCustomerAddressService(final CustomerAddressService customerAddressService) {
        this.customerAddressService = customerAddressService;
    }

    public CustomerCreditMemoDetailService getCustomerCreditMemoDetailService() {
        return customerCreditMemoDetailService;
    }

    public void setCustomerCreditMemoDetailService(final CustomerCreditMemoDetailService customerCreditMemoDetailService) {
        this.customerCreditMemoDetailService = customerCreditMemoDetailService;
    }

    public SystemInformationService getSystemInformationService() {
        return systemInformationService;
    }

    public void setSystemInformationService(final SystemInformationService systemInformationService) {
        this.systemInformationService = systemInformationService;
    }

    public CustomerInvoiceDetailService getCustomerInvoiceDetailService() {
        return customerInvoiceDetailService;
    }

    public void setCustomerInvoiceDetailService(final CustomerInvoiceDetailService customerInvoiceDetailService) {
        this.customerInvoiceDetailService = customerInvoiceDetailService;
    }

    public CustomerService getCustomerService() {
        return customerService;
    }

    public void setCustomerService(final CustomerService customerService) {
        this.customerService = customerService;
    }

    public void setCustomerInvoiceDocumentService(final CustomerInvoiceDocumentService customerInvoiceDocumentService) {
        this.customerInvoiceDocumentService = customerInvoiceDocumentService;
    }

    public void setAccountsReceivableDocumentHeaderService(
            final AccountsReceivableDocumentHeaderService accountsReceivableDocumentHeaderService) {
        this.accountsReceivableDocumentHeaderService = accountsReceivableDocumentHeaderService;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public CustomerDocumentService getCustomerDocumentService() {
        return customerDocumentService;
    }

    public void setCustomerDocumentService(final CustomerDocumentService customerDocumentService) {
        this.customerDocumentService = customerDocumentService;
    }

    public void setCustomerLookupable(final Lookupable customerLookupable) {
        this.customerLookupable = customerLookupable;
    }

    public KualiModuleService getKualiModuleService() {
        return kualiModuleService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public DocumentService getDocumentService() {
        return documentService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    @Override
    public ElectronicPaymentClaimingDocumentGenerationStrategy getAccountsReceivablePaymentClaimingStrategy() {
        return electronicPaymentClaimingDocumentGenerationStrategy;
    }

    @Override
    public Collection<AccountsReceivableCustomer> searchForCustomers(final Map<String, String> fieldValues) {
        customerLookupable.setBusinessObjectClass(Customer.class);
        final List<? extends BusinessObject> results = customerLookupable.getSearchResults(fieldValues);

        final Collection<AccountsReceivableCustomer> customers = new ArrayList<>();
        for (final BusinessObject result : results) {
            customers.add((AccountsReceivableCustomer) result);
        }

        return customers;
    }

    @Override
    public AccountsReceivableCustomer findCustomer(final String customerNumber) {
        final Map<String, Object> primaryKey = new HashMap<>();
        primaryKey.put(KFSPropertyConstants.CUSTOMER_NUMBER, customerNumber);
        return getKualiModuleService().getResponsibleModuleService(Customer.class)
                .getExternalizableBusinessObject(Customer.class, primaryKey);
    }

    @Override
    public Collection<AccountsReceivableCustomerAddress> searchForCustomerAddresses(final Map<String, String> fieldValues) {
        return new ArrayList<>(getBusinessObjectService().findMatching(CustomerAddress.class,
                fieldValues));
    }

    @Override
    public AccountsReceivableCustomerAddress findCustomerAddress(
            final String customerNumber,
            final String customerAddressIdentifier) {
        final Map<String, String> addressKey = new HashMap<>();
        addressKey.put(KFSPropertyConstants.CUSTOMER_NUMBER, customerNumber);
        addressKey.put(KFSPropertyConstants.CUSTOMER_ADDRESS_IDENTIFIER, customerAddressIdentifier);

        return getBusinessObjectService().findByPrimaryKey(CustomerAddress.class, addressKey);
    }

    @Override
    public AccountsReceivableCustomerInvoice getOpenCustomerInvoice(final String customerInvoiceDocumentNumber) {
        return getBusinessObjectService().findBySinglePrimaryKey(CustomerInvoiceDocument.class,
                customerInvoiceDocumentNumber);
    }

    @Override
    public Map<String, KualiDecimal> getCustomerInvoiceOpenAmount(
            final List<String> customerTypeCodes,
            final Integer customerInvoiceAge, final Date invoiceDueDateFrom) {
        final Map<String, KualiDecimal> customerInvoiceOpenAmountMap = new HashMap<>();

        final Collection<? extends AccountsReceivableCustomerInvoice> customerInvoiceDocuments =
                getOpenCustomerInvoices(customerTypeCodes, customerInvoiceAge, invoiceDueDateFrom);
        if (ObjectUtils.isNull(customerInvoiceDocuments)) {
            return customerInvoiceOpenAmountMap;
        }

        for (final AccountsReceivableCustomerInvoice invoiceDocument : customerInvoiceDocuments) {
            final KualiDecimal openAmount = invoiceDocument.getOpenAmount();

            if (ObjectUtils.isNotNull(openAmount) && openAmount.isPositive()) {
                customerInvoiceOpenAmountMap.put(invoiceDocument.getDocumentNumber(), openAmount);
            }
        }

        return customerInvoiceOpenAmountMap;
    }

    private Collection<? extends AccountsReceivableCustomerInvoice> getOpenCustomerInvoices(
            final List<String> customerTypeCodes, final Integer customerInvoiceAge, final Date invoiceDueDateFrom) {
        return customerInvoiceDocumentService.getAllAgingInvoiceDocumentsByCustomerTypes(customerTypeCodes,
                customerInvoiceAge, invoiceDueDateFrom);
    }

    @Override
    public AccountsReceivableCustomer createCustomer() {
        return new Customer();
    }

    @Override
    public AccountsReceivableCustomerAddress createCustomerAddress() {
        return new CustomerAddress();
    }

    @Override
    public AccountsReceivableCustomerAddressEmail createCustomerAddressEmail() {
        return new CustomerAddressEmail();
    }

    @Override
    public String getNextCustomerNumber(final AccountsReceivableCustomer customer) {
        return customerService.getNextCustomerNumber((Customer) customer);
    }

    @Override
    public String createAndSaveCustomer(final String description, final ContractsAndGrantsBillingAgency agency) {
        return customerDocumentService.createAndSaveCustomer(description, agency);
    }

    @Override
    public void saveCustomer(final AccountsReceivableCustomer customer) {
        getBusinessObjectService().save((Customer) customer);
    }

    @Override
    public List<AccountsReceivableCustomerType> findByCustomerTypeDescription(final String customerTypeDescription) {
        final Map<String, String> fieldMap = new HashMap<>();
        fieldMap.put(CustomerTypeFields.CUSTOMER_TYPE_DESC, customerTypeDescription);
        return new ArrayList<>(getBusinessObjectService().findMatching(CustomerType.class, fieldMap));
    }

    @Override
    public AccountsReceivableOrganizationOptions getOrgOptionsIfExists(
            final String chartOfAccountsCode,
            final String organizationCode) {
        final Map<String, String> criteria = new HashMap<>();
        criteria.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        criteria.put(KFSPropertyConstants.ORGANIZATION_CODE, organizationCode);
        return getBusinessObjectService().findByPrimaryKey(OrganizationOptions.class, criteria);
    }

    @Override
    public void saveCustomerInvoiceDocument(final AccountsReceivableCustomerInvoice customerInvoiceDocument) {
        getDocumentService().saveDocument((CustomerInvoiceDocument) customerInvoiceDocument);
    }

    @Override
    public Document blanketApproveCustomerInvoiceDocument(final AccountsReceivableCustomerInvoice customerInvoiceDocument) {
        return getDocumentService().blanketApproveDocument((CustomerInvoiceDocument) customerInvoiceDocument,
                null, null);
    }

    @Override
    public AccountsReceivableCustomerInvoiceRecurrenceDetails createCustomerInvoiceRecurrenceDetails() {
        return new CustomerInvoiceRecurrenceDetails();
    }

    @Override
    public AccountsReceivableDocumentHeader createAccountsReceivableDocumentHeader() {
        return new org.kuali.kfs.module.ar.businessobject.AccountsReceivableDocumentHeader();
    }

    @Override
    public AccountsReceivableSystemInformation getSystemInformationByProcessingChartOrgAndFiscalYear(
            final String chartOfAccountsCode, final String organizationCode, final Integer currentFiscalYear) {
        return systemInformationService
                .getByProcessingChartOrgAndFiscalYear(chartOfAccountsCode, organizationCode, currentFiscalYear);
    }

    @Override
    public AccountsReceivableCustomerInvoiceDetail getCustomerInvoiceDetailFromCustomerInvoiceItemCode(
            final String invoiceItemCode, final String processingChartCode, final String processingOrgCode) {
        return customerInvoiceDetailService
                .getCustomerInvoiceDetailFromCustomerInvoiceItemCode(invoiceItemCode, processingChartCode,
                        processingOrgCode);
    }

    @Override
    public String getAccountsReceivableObjectCodeBasedOnReceivableParameter(
            final AccountsReceivableCustomerInvoiceDetail
            customerInvoiceDetail) {
        return getAccountsReceivablePendingEntryService()
                .getAccountsReceivableObjectCode((CustomerInvoiceDetail) customerInvoiceDetail);
    }

    @Override
    public void recalculateCustomerInvoiceDetail(
            final AccountsReceivableCustomerInvoice customerInvoiceDocument,
            final AccountsReceivableCustomerInvoiceDetail detail) {
        customerInvoiceDetailService.recalculateCustomerInvoiceDetail((CustomerInvoiceDocument) customerInvoiceDocument,
                (CustomerInvoiceDetail) detail);
    }

    @Override
    public void prepareCustomerInvoiceDetailForAdd(
            final AccountsReceivableCustomerInvoiceDetail detail,
            final AccountsReceivableCustomerInvoice customerInvoiceDocument) {
        customerInvoiceDetailService.prepareCustomerInvoiceDetailForAdd((CustomerInvoiceDetail) detail,
                (CustomerInvoiceDocument) customerInvoiceDocument);
    }

    @Override
    public KualiDecimal getOpenAmountForCustomerInvoiceDocument(final AccountsReceivableCustomerInvoice invoice) {
        return customerInvoiceDocumentService.getOpenAmountForCustomerInvoiceDocument((CustomerInvoiceDocument) invoice);
    }

    @Override
    public AccountsReceivableDocumentHeader getNewAccountsReceivableDocumentHeader(
            final String processingChart,
            final String processingOrg) {
        return accountsReceivableDocumentHeaderService.getNewAccountsReceivableDocumentHeader(processingChart,
                processingOrg);
    }

    @Override
    public AccountsReceivableCustomerInvoice createCustomerInvoiceDocument() {
        return (CustomerInvoiceDocument) getDocumentService().getNewDocument(CustomerInvoiceDocument.class);
    }

    @Override
    public AccountsReceivableCustomerCreditMemo createCustomerCreditMemoDocument() {
        final CustomerCreditMemoDocument crmDocument;
        crmDocument = (CustomerCreditMemoDocument) getDocumentService().getNewDocument(
                CustomerCreditMemoDocument.class);
        return crmDocument;
    }

    @Override
    public AccountsReceivableCustomerCreditMemo populateCustomerCreditMemoDocumentDetails(
            final AccountsReceivableCustomerCreditMemo arCrmDocument, final String invoiceNumber, final KualiDecimal creditAmount) {
        final CustomerCreditMemoDocument crmDocument = (CustomerCreditMemoDocument) arCrmDocument;

        crmDocument.setFinancialDocumentReferenceInvoiceNumber(invoiceNumber);
        //force to refresh invoice;
        crmDocument.getInvoice();
        crmDocument.populateCustomerCreditMemoDetails();

        //update the amount on the credit memo detail, there should only be one item for Travel item so we will
        // update that amount
        final CustomerCreditMemoDetail detail = crmDocument.getCreditMemoDetails().get(0);
        detail.setCreditMemoItemTotalAmount(creditAmount);

        customerCreditMemoDetailService.recalculateCustomerCreditMemoDetail(detail, crmDocument);

        return crmDocument;
    }

    @Override
    public Document blanketApproveCustomerCreditMemoDocument(
            final AccountsReceivableCustomerCreditMemo creditMemoDocument,
            final String annotation) {
        return getDocumentService().blanketApproveDocument((CustomerCreditMemoDocument) creditMemoDocument, annotation,
                null);
    }

    public void setKualiModuleService(final KualiModuleService kualiModuleService) {
        this.kualiModuleService = kualiModuleService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public FinancialSystemUserService getFinancialSystemUserService() {
        return financialSystemUserService;
    }

    public void setFinancialSystemUserService(final FinancialSystemUserService financialSystemUserService) {
        this.financialSystemUserService = financialSystemUserService;
    }

    public ElectronicPaymentClaimingDocumentGenerationStrategy getElectronicPaymentClaimingDocumentGenerationStrategy() {
        return electronicPaymentClaimingDocumentGenerationStrategy;
    }

    public void setElectronicPaymentClaimingDocumentGenerationStrategy(
            final ElectronicPaymentClaimingDocumentGenerationStrategy electronicPaymentClaimingDocumentGenerationStrategy) {
        this.electronicPaymentClaimingDocumentGenerationStrategy = electronicPaymentClaimingDocumentGenerationStrategy;
    }

    public AccountsReceivablePendingEntryService getAccountsReceivablePendingEntryService() {
        return accountsReceivablePendingEntryService;
    }

    public void setAccountsReceivablePendingEntryService(
            final AccountsReceivablePendingEntryService accountsReceivablePendingEntryService) {
        this.accountsReceivablePendingEntryService = accountsReceivablePendingEntryService;
    }
}
