/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.web.struts;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.kuali.kfs.kns.util.WebUtils;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.TransmitContractsAndGrantsInvoicesLookupDataHolder;
import org.kuali.kfs.module.ar.document.ContractsGrantsInvoiceDocument;
import org.kuali.kfs.module.ar.report.service.TransmitContractsAndGrantsInvoicesService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.krad.bo.BusinessObject;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.ByteArrayOutputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Action class for Transmit Contracts & Grants Invoices.
 */
public class TransmitContractsAndGrantsInvoicesLookupAction extends ContractsGrantsReportLookupAction {

    protected static volatile TransmitContractsAndGrantsInvoicesService invoiceReportDeliveryService;
    protected static volatile DateTimeService dateTimeService;

    /**
     * This method forwards the print request according to the selections.
     */
    @Override
    public ActionForward print(
            final ActionMapping mapping, final ActionForm form, final HttpServletRequest request,
            final HttpServletResponse response) throws Exception {
        final TransmitContractsAndGrantsInvoicesLookupForm lookupForm = (TransmitContractsAndGrantsInvoicesLookupForm) form;
        final Map<String, String> fieldValues = lookupForm.getFieldsForLookup();

        final String invoiceTransmissionMethodCode = fieldValues.get(ArPropertyConstants.INVOICE_TRANSMISSION_METHOD_CODE);

        getTransmitContractsAndGrantsInvoicesService().validateSearchParameters(fieldValues);

        // Fetch the invoices with the input parameters
        final Collection<ContractsGrantsInvoiceDocument> list = getTransmitContractsAndGrantsInvoicesService()
                .getInvoicesByParametersFromRequest(fieldValues);

        if (CollectionUtils.isNotEmpty(list)) {
            // mapping to return to once delivery method is processed
            final ActionForward forward = mapping.findForward(KFSConstants.MAPPING_BASIC);
            final Set<ContractsGrantsInvoiceDocument> emailSet = new HashSet<>();
            final Set<ContractsGrantsInvoiceDocument> mailSet = new HashSet<>();

            // Check delivery type for EMAIL
            if (ArConstants.InvoiceTransmissionMethod.EMAIL.equalsIgnoreCase(invoiceTransmissionMethodCode)) {
                // Get all email-able invoices
                for (final ContractsGrantsInvoiceDocument invoice : list) {
                    if (getTransmitContractsAndGrantsInvoicesService().isInvoiceValidToEmail(invoice)) {
                        emailSet.add(invoice);
                    }
                }
                if (CollectionUtils.isNotEmpty(emailSet)) {
                    if (getTransmitContractsAndGrantsInvoicesService().sendEmailForListofInvoicesToAgency(emailSet)) {
                        GlobalVariables.getMessageMap().putInfoForSectionId(ArPropertyConstants.LOOKUP_SECTION_ID,
                                ArKeyConstants.INVOICE_EMAILS_SENT);
                    } else {
                        GlobalVariables.getMessageMap().putInfoForSectionId(ArPropertyConstants.LOOKUP_SECTION_ID,
                                ArKeyConstants.ERROR_SENDING_INVOICE_EMAILS);
                    }
                } else {
                    // this is so the message about no search results shows up in the same place as when doing a search
                    request.setAttribute(KFSPropertyConstants.REQUEST_SEARCH_RESULTS_ACTUAL_SIZE, 0);
                }
            } else {
                // Get all mail-able invoices
                for (final ContractsGrantsInvoiceDocument invoice : list) {
                    if (getTransmitContractsAndGrantsInvoicesService().isInvoiceValidToMail(invoice)) {
                        mailSet.add(invoice);
                    }
                }

                // Process mailable invoices found
                if (CollectionUtils.isNotEmpty(mailSet)) {
                    final ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    if (getTransmitContractsAndGrantsInvoicesService().printInvoicesAndEnvelopesZip(mailSet, baos)) {
                        final String fileName = ArConstants.INVOICE_ZIP_FILE_PREFIX +
                                                getDateTimeService().toDateStringForFilename(getDateTimeService().getCurrentDate()) +
                                                KFSConstants.ReportGeneration.ZIP_FILE_EXTENSION;
                        WebUtils.saveMimeOutputStreamAsFile(response, KFSConstants.ReportGeneration.ZIP_MIME_TYPE,
                                baos, fileName);
                        GlobalVariables.getMessageMap().putInfoForSectionId(ArPropertyConstants.LOOKUP_SECTION_ID,
                                ArKeyConstants.INVOICES_PRINT_SUCCESSFULL);
                        request.setAttribute(KFSPropertyConstants.REQUEST_SEARCH_RESULTS_ACTUAL_SIZE, mailSet.size());

                        return null;
                    } else {
                        GlobalVariables.getMessageMap().putInfoForSectionId(ArPropertyConstants.LOOKUP_SECTION_ID,
                                ArKeyConstants.INVOICES_PRINT_UNSUCCESSFULL);
                    }
                } else {
                    // if no invoices to be mailed were found add status message this is so the message about no
                    // search results shows up in the same place as when doing a search
                    request.setAttribute(KFSPropertyConstants.REQUEST_SEARCH_RESULTS_ACTUAL_SIZE, 0);
                }
            }
            return forward;
        }
        // Catch all
        // this is so the message about no search results shows up in the same place as when doing a search
        request.setAttribute(KFSPropertyConstants.REQUEST_SEARCH_RESULTS_ACTUAL_SIZE, 0);

        return mapping.findForward(KFSConstants.MAPPING_BASIC);
    }

    protected TransmitContractsAndGrantsInvoicesService getTransmitContractsAndGrantsInvoicesService() {
        if (invoiceReportDeliveryService == null) {
            invoiceReportDeliveryService = SpringContext.getBean(TransmitContractsAndGrantsInvoicesService.class);
        }
        return invoiceReportDeliveryService;
    }

    @Override
    protected String getSortFieldName() {
        return null;
    }

    @Override
    public String getReportBuilderServiceBeanName() {
        return null;
    }

    @Override
    public Class<? extends BusinessObject> getPrintSearchCriteriaClass() {
        return TransmitContractsAndGrantsInvoicesLookupDataHolder.class;
    }

    @Override
    public String generateReportTitle(final LookupForm lookupForm) {
        return null;
    }

    protected DateTimeService getDateTimeService() {
        if (dateTimeService == null) {
            dateTimeService = SpringContext.getBean(DateTimeService.class);
        }
        return dateTimeService;
    }
}
