/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.batch.service.CustomerLoadService;
import org.kuali.kfs.module.ar.batch.vo.CustomerCsvHeaders;
import org.kuali.kfs.module.ar.batch.vo.CustomerVO;
import org.kuali.kfs.sys.batch.CsvBatchInputFileTypeBase;

import java.util.List;
import java.util.Map;

public class CustomerLoadCSVInputFileType extends CsvBatchInputFileTypeBase<CustomerCsvHeaders> {

    private static final Logger LOG = LogManager.getLogger();

    private static final String FILE_NAME_PREFIX = "customer_load";
    private static final String FILE_NAME_DELIM = "_";

    private CustomerLoadService customerLoadService;

    @Override
    public String getFileName(
            final String principalName,
            final Object parsedFileContents,
            final String fileUserIdentifier
    ) {
        return customerLoadService.getFileName(principalName, fileUserIdentifier, FILE_NAME_PREFIX, FILE_NAME_DELIM);
    }

    @Override
    public String getFileTypeIdentifier() {
        return ArConstants.CustomerLoad.CUSTOMER_CSV_LOAD_FILE_TYPE_IDENTIFIER;
    }

    @Override
    public boolean validate(final Object parsedFileContents) {
        final List<CustomerVO> customerVOs = (List<CustomerVO>) parsedFileContents;
        return customerLoadService.validate(customerVOs);
    }

    @Override
    public String getTitleKey() {
        return ArKeyConstants.CustomerLoad.MESSAGE_BATCH_UPLOAD_CSV_TITLE_CUSTOMER;
    }

    public void setCustomerLoadService(final CustomerLoadService customerLoadService) {
        this.customerLoadService = customerLoadService;
    }

    /**
     * Convert the parsedFileContents object into CustomerVO for validation
     */
    @Override
    protected Object convertParsedObjectToVO(final Object parsedContent) {
        final List<CustomerVO> customerVOs;
        try {
            final List<Map<String, String>> parseDataList = (List<Map<String, String>>) parsedContent;
            customerVOs = CustomerLoadCSVBuilder.buildCustomerDigestVO(parseDataList);
        } catch (final Exception e) {
            LOG.error(e);
            throw new RuntimeException(e.getMessage(), e);
        }
        return customerVOs;
    }
}
