/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import org.apache.commons.beanutils.converters.BooleanConverter;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.kns.document.MaintenanceDocument;
import org.kuali.kfs.kns.util.KNSGlobalVariables;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.util.ErrorMessage;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.ArKeyConstants;
import org.kuali.kfs.module.ar.ArPropertyConstants;
import org.kuali.kfs.module.ar.businessobject.Bill;
import org.kuali.kfs.module.ar.businessobject.PredeterminedBillingSchedule;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSKeyConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.springframework.util.AutoPopulatingList;

import java.sql.Date;
import java.text.ParseException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class PredeterminedBillingScheduleCsvInputFileType extends ScheduleCsvInputFileType {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    protected Object convertParsedObjectToVO(final Object parsedContent) {
        final List<PredeterminedBillingSchedule> predeterminedBillingSchedules = new LinkedList<>();
        final List<Map<String, String>> parseDataList = (List<Map<String, String>>) parsedContent;
        for (final Map<String, String> row : parseDataList) {
            final String proposalNumber = row.get(KFSPropertyConstants.PROPOSAL_NUMBER);
            String chartOfAccountsCode = row.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
            final String accountNumber = row.get(KFSPropertyConstants.ACCOUNT_NUMBER);

            if (StringUtils.isEmpty(chartOfAccountsCode)) {
                chartOfAccountsCode = deriveChartOfAccountsCodeFromAccount(accountNumber);
            }

            final PredeterminedBillingSchedule currentSchedule;

            try {
                currentSchedule = new PredeterminedBillingSchedule.PredeterminedBillingScheduleBuilder()
                        .setProposalNumber(proposalNumber)
                        .setChartOfAccountsCode(chartOfAccountsCode)
                        .setAccountNumber(accountNumber)
                        .addBill(createBill(proposalNumber, chartOfAccountsCode, accountNumber, row))
                        .build();
            } catch (IllegalStateException | ParseException e) {
                GlobalVariables.getMessageMap().putError(KFSConstants.GLOBAL_ERRORS,
                        KFSKeyConstants.ERROR_BATCH_UPLOAD_PARSING, e.getMessage());
                return null;
            }

            if (predeterminedBillingSchedules.contains(currentSchedule)) {
                final PredeterminedBillingSchedule existingSchedule = predeterminedBillingSchedules
                        .get(predeterminedBillingSchedules.indexOf(currentSchedule));
                existingSchedule.addBill(currentSchedule.getBills().get(0));
            } else {
                predeterminedBillingSchedules.add(currentSchedule);
            }
        }

        return predeterminedBillingSchedules;
    }

    private Bill createBill(
            final String proposalNumber, final String chartOfAccountsCode, final String accountNumber,
            final Map<String, String> row) throws ParseException {
        final String billNumber = row.get(ArPropertyConstants.BillFields.BILL_NUMBER);
        final String billDescription = row.get(ArPropertyConstants.BillFields.BILL_DESCRIPTION);

        final String estimatedAmountString = row.get(ArPropertyConstants.BillFields.ESTIMATED_AMOUNT);
        KualiDecimal estimatedAmount = null;
        if (StringUtils.isNotBlank(estimatedAmountString)) {
            estimatedAmount = new KualiDecimal(estimatedAmountString);
        }

        final String billDateString = row.get(ArPropertyConstants.BillFields.BILL_DATE);
        Date billDate = null;
        if (billDateString.length() > 0) {
            billDate = dateTimeService.convertToSqlDate(billDateString);
        }

        final BooleanConverter converter = new BooleanConverter(Boolean.FALSE);
        final boolean active = converter.convert(Boolean.TYPE, row.get(KFSPropertyConstants.ACTIVE));

        final Bill bill = new Bill.BillBuilder()
                .setProposalNumber(proposalNumber)
                .setChartOfAccountsCode(chartOfAccountsCode)
                .setAccountNumber(accountNumber)
                .setActive(active)
                .setBillNumber(billNumber)
                .setBillDescription(billDescription)
                .setBillDate(billDate)
                .setEstimatedAmount(estimatedAmount)
                .build();

        bill.setNewCollectionRecord(true);

        return bill;
    }

    @Override
    public String getFileTypeIdentifier() {
        return ArConstants.PredeterminedBillingScheduleImport.FILE_TYPE_IDENTIFIER;
    }

    @Override
    public void process(final String fileName, final Object parsedFileContents) {
        final List<PredeterminedBillingSchedule> predeterminedBillingSchedules =
                (List<PredeterminedBillingSchedule>) parsedFileContents;

        for (final PredeterminedBillingSchedule predeterminedBillingSchedule: predeterminedBillingSchedules) {
            final MaintenanceDocument predeterminedBillingScheduleDocument;
            predeterminedBillingScheduleDocument = (MaintenanceDocument) documentService.getNewDocument(
                    maintenanceDocumentDictionaryService.getDocumentTypeName(PredeterminedBillingSchedule.class));

            predeterminedBillingScheduleDocument.getDocumentHeader()
                    .setDocumentDescription("Predetermined Billing Schedule Import");

            final PredeterminedBillingSchedule existingPredeterminedBillingSchedule =
                    getPredeterminedBillingSchedule(predeterminedBillingSchedule);
            if (ObjectUtils.isNull(existingPredeterminedBillingSchedule)) {
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setMaintenanceAction(KRADConstants.MAINTENANCE_NEW_ACTION);
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setBusinessObject(predeterminedBillingSchedule);
                predeterminedBillingScheduleDocument.getOldMaintainableObject()
                        .setBusinessObject(new PredeterminedBillingSchedule());
            } else {
                final PredeterminedBillingSchedule copyOfExistingPredeterminedBillingSchedule =
                        (PredeterminedBillingSchedule) ObjectUtils.deepCopy(existingPredeterminedBillingSchedule);
                predeterminedBillingSchedule.getBills().forEach(bill -> {
                    copyOfExistingPredeterminedBillingSchedule.addBill(bill);
                    // add empty bill to collection so that bill collection sizes on old and new
                    // will match. yes this is dumb. see FieldUtils.meshFields (src of stupid)
                    existingPredeterminedBillingSchedule.addBill(new Bill());
                });
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setBusinessObject(copyOfExistingPredeterminedBillingSchedule);
                predeterminedBillingScheduleDocument.getNewMaintainableObject()
                        .setMaintenanceAction(KRADConstants.MAINTENANCE_EDIT_ACTION);
                predeterminedBillingScheduleDocument.getOldMaintainableObject()
                        .setBusinessObject(existingPredeterminedBillingSchedule);
            }

            try {
                documentService.saveDocument(predeterminedBillingScheduleDocument);
                documentService.routeDocument(predeterminedBillingScheduleDocument,
                        "Routed New/Edit Predetermined Billing Schedule Maintenance Document from " +
                                "Predetermined Billing Schedule Import Process",
                        null);
            } catch (final ValidationException ve) {
                // ValidationException(s) that are mapped to GLOBAL_ERRORS may prevent the document from being saved
                // (e.g. when there is another maintenance document that has a lock on the same key as the given
                // document). In those cases we want to display an appropriate message to the user.
                final Map<String, AutoPopulatingList<ErrorMessage>> errorMessages = GlobalVariables.getMessageMap()
                        .getErrorMessages();
                errorMessages.keySet().stream()
                        .filter(key -> key.equals(KFSConstants.GLOBAL_ERRORS))
                        .forEach(key -> KNSGlobalVariables.getMessageList()
                                .add(ArKeyConstants.ERROR_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_SAVE_FAILURE,
                                predeterminedBillingSchedule.getProposalNumber(),
                                        predeterminedBillingSchedule.getChartOfAccountsCode(),
                                        predeterminedBillingSchedule.getAccountNumber(),
                                        flattenErrorMessages(errorMessages.get(key))));
                // We have elevated all the error messages we need to relay to the user, so we can clear them out.
                GlobalVariables.getMessageMap().clearErrorMessages();
                // no additional action is necessary if routing caused a ValidationException; in this case we want to
                // continue potentially processing additionally predetermined billing schedules and leave the in error
                // document in saved state
            }
        }

        // used this method so that messaging would go to same place as the messaging coming from batch save process
        KNSGlobalVariables.getMessageList().add(
                ArKeyConstants.MESSAGE_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_SUCCESSFUL);
        removeFiles(fileName);
    }

    private PredeterminedBillingSchedule getPredeterminedBillingSchedule(
            final PredeterminedBillingSchedule predeterminedBillingSchedule) {
        final Map<String, Object> primaryKeys = new HashMap<>();
        primaryKeys.put(KFSPropertyConstants.PROPOSAL_NUMBER, predeterminedBillingSchedule.getProposalNumber());
        primaryKeys.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE,
                predeterminedBillingSchedule.getChartOfAccountsCode());
        primaryKeys.put(KFSPropertyConstants.ACCOUNT_NUMBER, predeterminedBillingSchedule.getAccountNumber());
        return businessObjectService.findByPrimaryKey(PredeterminedBillingSchedule.class, primaryKeys);
    }

    @Override
    public String getTitleKey() {
        return ArKeyConstants.MESSAGE_AR_PREDETERMINED_BILLING_SCHEDULE_IMPORT_TITLE;
    }

}
