/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch;

import com.opencsv.CSVReader;
import com.opencsv.exceptions.CsvException;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.datadictionary.legacy.MaintenanceDocumentDictionaryService;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.util.ErrorMessage;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ar.batch.vo.PredeterminedBillingScheduleCsvHeaders;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.batch.CsvBatchInputFileTypeBase;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.springframework.util.AutoPopulatingList;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;

public abstract class ScheduleCsvInputFileType extends
        CsvBatchInputFileTypeBase<PredeterminedBillingScheduleCsvHeaders> {
    private static final Logger LOG = LogManager.getLogger();

    protected BusinessObjectService businessObjectService;
    protected DocumentService documentService;
    protected DateTimeService dateTimeService;
    MaintenanceDocumentDictionaryService maintenanceDocumentDictionaryService;
    private AccountService accountService;
    private ConfigurationService configurationService;

    /**
     * Overridden to remove chart of accounts code from the expected header list if accounts cannot cross charts and
     * the chart of accounts code column hasn't been included in the file to import.
     */
    @Override
    protected void validateCSVFileInput(final List<String> expectedHeaderList, final InputStream fileContents) throws
            CsvException, IOException {
        final List<String> inputHeaderList;
        try (CSVReader csvReader = new CSVReader(new InputStreamReader(fileContents, StandardCharsets.UTF_8))) {
            inputHeaderList = Arrays.asList(csvReader.readNext());
        }
        fileContents.reset();

        if (!accountService.accountsCanCrossCharts()
                && !inputHeaderList.contains(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE)) {
            expectedHeaderList.remove(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        }

        super.validateCSVFileInput(expectedHeaderList, fileContents);
    }

    @Override
    public String getFileName(final String principalName, final Object parsedFileContents, final String fileUserIdentifier) {
        final StringBuilder fileName = new StringBuilder("_");

        fileName.append("_").append(principalName);

        if (StringUtils.isNotBlank(fileUserIdentifier)) {
            fileName.append("_").append(fileUserIdentifier);
        }

        final DateFormat dateFormat = new SimpleDateFormat("yyyyMMdd_HHmmss", Locale.US);
        dateFormat.format(Timestamp.valueOf(LocalDateTime.now()));
        fileName.append("_").append(dateFormat.format(Timestamp.valueOf(LocalDateTime.now())));

        return fileName.toString();
    }

    protected String flattenErrorMessages(final AutoPopulatingList<ErrorMessage> errorMessages) {
        final StringBuilder flattenedErrorMessage = new StringBuilder();

        errorMessages.forEach(errorMessage -> flattenedErrorMessage.append(MessageFormat.format(
                        configurationService.getPropertyValueAsString(errorMessage.getErrorKey()),
                        errorMessage.getMessageParameters())).append("<br/>"));

        return flattenedErrorMessage.toString();
    }

    void removeFiles(final String filename) {
        final String doneFileName = StringUtils.substringBeforeLast(filename, ".") + ".done";
        removeFile(doneFileName);
        removeFile(filename);
    }

    private void removeFile(final String qualifiedFilePath) {
        final File fileToDelete = new File(qualifiedFilePath);
        if (fileToDelete.exists()) {
            if (!fileToDelete.delete()) {
                LOG.warn("{} could not be successfully deleted.", fileToDelete::getName);
            }
        }
    }

    @Override
    public boolean validate(final Object parsedFileContents) {
        // validation is performed by convertParsedObjectToVO when the file is parsed
        return true;
    }

    String deriveChartOfAccountsCodeFromAccount(final String accountNumber) {
        if (!accountService.accountsCanCrossCharts()) {
            final Account account = accountService.getUniqueAccountForAccountNumber(accountNumber);
            if (ObjectUtils.isNotNull(account)) {
                return account.getChartOfAccountsCode();
            }
        }
        return null;
    }

    public void setAccountService(final AccountService accountService) {
        this.accountService = accountService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setConfigurationService(final ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setMaintenanceDocumentDictionaryService(
            final MaintenanceDocumentDictionaryService maintenanceDocumentDictionaryService) {
        this.maintenanceDocumentDictionaryService = maintenanceDocumentDictionaryService;
    }

    public DateTimeService getDateTimeService() {
        return dateTimeService;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }
}
