/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ar.batch.vo;

import org.apache.commons.beanutils.ConversionException;
import org.apache.commons.beanutils.converters.SqlDateConverter;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.datadictionary.legacy.MaintenanceDocumentDictionaryService;
import org.kuali.kfs.krad.service.SequenceAccessorService;
import org.kuali.kfs.module.ar.ArConstants;
import org.kuali.kfs.module.ar.batch.report.CustomerLoadBatchErrors;
import org.kuali.kfs.module.ar.businessobject.Customer;
import org.kuali.kfs.module.ar.businessobject.CustomerAddress;
import org.kuali.kfs.module.ar.businessobject.CustomerAddressEmail;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.sql.Date;

/**
 * This class converts a CustomerVO object to a standard Customer object.
 */
public class CustomerVOAdapter {

    private static final Logger LOG = LogManager.getLogger();

    private static final Class<Customer> BO_CLASS = Customer.class;

    private DateTimeService dateTimeService;
    private MaintenanceDocumentDictionaryService maintDocDDService;
    private SequenceAccessorService sequenceAccessorService;

    private Customer customer;
    private String customerName;
    private CustomerLoadBatchErrors errors;
    private CustomerVO customerVO;

    /**
     * Converts a CustomerVO to a real Customer BO.  Tries to do intelligent type conversions where the types aren't
     * Strings.
     * <p>
     * NOTE that conversion exceptions will be swallowed! and converted to errors in the parameter errorMap.
     *
     * @param customerVO The VO full of String values to convert from.
     * @param errors     An empty MessageMap collection to add errors to.  Only new errors will be added.
     * @return A populated Customer object, from the VO.
     */
    public Customer convert(final CustomerVO customerVO, final CustomerLoadBatchErrors errors) {
        if (customerVO == null) {
            throw new IllegalArgumentException("Parameter customerVO may not be null.");
        }
        this.customerVO = customerVO;

        //  the whole error system is keyed of customerName, so if we dont get one of those, we cant even proceed.
        if (StringUtils.isBlank(customerVO.getCustomerName())) {
            LOG.error("CustomerName can never be empty-string or null.");
            addError("customerName",
                    String.class,
                    customerVO.getCustomerName(),
                    "CustomerName can never be empty-string or null."
            );
            return null;
        }

        customer = new Customer();
        customerName = this.customerVO.getCustomerName();

        if (errors == null) {
            LOG.error("Passed in CustomerLoadBatchErrors must not be null.");
            throw new IllegalArgumentException("Passed in CustomerLoadBatchErrors must not be null.");
        }
        this.errors = errors;

        convertCustomerStringProperties();
        convertCustomerDateProperties();
        convertCustomerKualiDecimalProperties();
        convertCustomerBooleanProperties();
        convertCustomerAddresses();

        return customer;
    }

    private void convertCustomerStringProperties() {
        //  these are String to String conversions, so they will always work
        customer.setCustomerNumber(applyDefaultValue("customerNumber", customerVO.getCustomerNumber()));
        customer.setCustomerName(applyDefaultValue("customerName", customerVO.getCustomerName()));
        customer.setCustomerParentCompanyNumber(applyDefaultValue("customerParentCompanyNumber",
                customerVO.getCustomerParentCompanyNumber()
        ));
        customer.setCustomerTypeCode(applyDefaultValue("customerTypeCode", customerVO.getCustomerTypeCode()));
        customer.setCustomerTaxTypeCode(applyDefaultValue("customerTaxTypeCode", customerVO.getCustomerTaxTypeCode()));
        customer.setCustomerTaxNbr(applyDefaultValue("customerTaxNbr", customerVO.getCustomerTaxNbr()));
        customer.setCustomerPhoneNumber(applyDefaultValue("customerPhoneNumber", customerVO.getCustomerPhoneNumber()));
        customer.setCustomer800PhoneNumber(applyDefaultValue("customer800PhoneNumber",
                customerVO.getCustomer800PhoneNumber()
        ));
        customer.setCustomerContactName(applyDefaultValue("customerContactName", customerVO.getCustomerContactName()));
        customer.setCustomerContactPhoneNumber(applyDefaultValue("customerContactPhoneNumber",
                customerVO.getCustomerContactPhoneNumber()
        ));
        customer.setCustomerFaxNumber(applyDefaultValue("customerFaxNumber", customerVO.getCustomerFaxNumber()));
        customer.setCustomerCreditApprovedByName(applyDefaultValue("customerCreditApprovedByName",
                customerVO.getCustomerCreditApprovedByName()
        ));
        customer.setCustomerEmailAddress(applyDefaultValue("customerEmailAddress",
                customerVO.getCustomerEmailAddress()
        ));
    }

    private void convertCustomerDateProperties() {
        final Date todayDate = dateTimeService.getCurrentSqlDate();

        customer.setCustomerAddressChangeDate(todayDate);
        customer.setCustomerRecordAddDate(todayDate);
        customer.setCustomerLastActivityDate(todayDate);

        customer.setCustomerBirthDate(convertToJavaSqlDate("customerBirthDate",
                applyDefaultValue("customerBirthDate", customerVO.getCustomerBirthDate())
        ));
    }

    private void convertCustomerKualiDecimalProperties() {
        customer.setCustomerCreditLimitAmount(convertToKualiDecimal("customerCreditLimitAmount",
                applyDefaultValue("customerCreditLimitAmount", customerVO.getCustomerCreditLimitAmount())
        ));
    }

    private void convertCustomerBooleanProperties() {
        customer.setActive(convertToLittleBoolean(applyDefaultValue("customerActiveIndicator",
                customerVO.getCustomerActiveIndicator()
        )));
        customer.setCustomerTaxExemptIndicator(convertToLittleBoolean(applyDefaultValue("customerTaxExemptIndicator",
                customerVO.getCustomerTaxExemptIndicator()
        )));
    }

    private void convertCustomerAddresses() {
        CustomerAddress customerAddress;
        for (final CustomerAddressVO addressVO : customerVO.getCustomerAddresses()) {
            customerAddress = convertCustomerAddress(addressVO, customer.getCustomerNumber());
            customer.getCustomerAddresses().add(customerAddress);
        }
    }

    private CustomerAddress convertCustomerAddress(
            final CustomerAddressVO customerAddressVO, final String customerNumber
    ) {
        final CustomerAddress customerAddress = new CustomerAddress();

        //  link the customerAddress to the parent customer
        customerAddress.setCustomerNumber(customerNumber);

        customerAddress.setCustomerAddressName(applyDefaultValue("customerAddressName",
                customerAddressVO.getCustomerAddressName()
        ));
        customerAddress.setCustomerLine1StreetAddress(applyDefaultValue("customerLine1StreetAddress",
                customerAddressVO.getCustomerLine1StreetAddress()
        ));
        customerAddress.setCustomerLine2StreetAddress(applyDefaultValue("customerLine2StreetAddress",
                customerAddressVO.getCustomerLine2StreetAddress()
        ));
        customerAddress.setCustomerCityName(applyDefaultValue("customerCityName",
                customerAddressVO.getCustomerCityName()
        ));
        customerAddress.setCustomerStateCode(applyDefaultValue("customerStateCode",
                customerAddressVO.getCustomerStateCode()
        ));
        customerAddress.setCustomerZipCode(applyDefaultValue("customerZipCode",
                customerAddressVO.getCustomerZipCode()
        ));
        customerAddress.setCustomerCountryCode(applyDefaultValue("customerCountryCode",
                customerAddressVO.getCustomerCountryCode()
        ));
        customerAddress.setCustomerAddressInternationalProvinceName(applyDefaultValue(
                "customerAddressInternationalProvinceName",
                customerAddressVO.getCustomerAddressInternationalProvinceName()
        ));
        customerAddress.setCustomerInternationalMailCode(applyDefaultValue("customerInternationalMailCode",
                customerAddressVO.getCustomerInternationalMailCode()
        ));

        final String customerEmailAddress = customerAddressVO.getCustomerEmailAddress();
        if (StringUtils.isNotBlank(customerEmailAddress)) {
            final CustomerAddressEmail customerAddressEmail = new CustomerAddressEmail();
            customerAddressEmail.setCustomerNumber(customerNumber);
            customerAddressEmail.setCustomerAddressIdentifier(Math.toIntExact(sequenceAccessorService.getNextAvailableSequenceNumber(
                    ArConstants.CUST_ADDR_ID_SEQ,
                    CustomerAddress.class
            )));
            customerAddressEmail.setCustomerAddressIdentifier(null);
            customerAddressEmail.setCustomerEmailAddress(applyDefaultValue("customerEmailAddress",
                    customerEmailAddress
            ));
            customerAddressEmail.setActive(true);
            customerAddress.getCustomerAddressEmails().add(customerAddressEmail);
        }

        customerAddress.setCustomerAddressTypeCode(applyDefaultValue("customerAddressTypeCode",
                customerAddressVO.getCustomerAddressTypeCode()
        ));

        customerAddress.setCustomerAddressEndDate(convertToJavaSqlDate("customerAddressEndDate",
                applyDefaultValue("customerAddressEndDate", customerAddressVO.getCustomerAddressEndDate())
        ));

        return customerAddress;
    }

    /**
     * This method converts a string value that may represent a date into a Date. If the value is blank (whitespace,
     * empty, null) then a null Date object is returned. If the value cannot be converted to a java.sql.Date, then a
     * RuntimException or ConversionException will be thrown.
     *
     * @param propertyName Name of the field whose value is being converted.
     * @param dateValue    The value being converted.
     * @return A valid Date with the converted value, if possible.
     */
    private Date convertToJavaSqlDate(final String propertyName, final String dateValue) {
        if (StringUtils.isBlank(dateValue)) {
            return null;
        }

        final Date date;
        final SqlDateConverter converter = new SqlDateConverter();
        final Object obj;
        try {
            obj = converter.convert(Date.class, dateValue);
        } catch (final ConversionException e) {
            LOG.error("Failed to convert the value [{}] from field [{}] to a Date.", dateValue, propertyName);
            addError(propertyName, Date.class, dateValue, "Could not convert value to target type.");
            return null;
        }
        try {
            date = (Date) obj;
        } catch (final Exception e) {
            LOG.error("Failed to cast the converters results to a Date.");
            addError(propertyName, Date.class, dateValue, "Could not convert value to target type.");
            return null;
        }

        if (!(obj instanceof Date)) {
            LOG.error("Failed to convert the value [{}] from field [{}] to a Date.", dateValue, propertyName);
            addError(propertyName, Date.class, dateValue, "Could not convert value to target type.");
            return null;
        }
        return date;
    }

    /**
     * This method converts a string, which may be blank, null or whitespace, into a KualiDecimal, if possible.
     * <p>
     * A null, blank, or whitespace value passed in will result in a Null KualiDecimal value returned.  A value passed
     * in which is not blank, but cannot otherwise be converted to a KualiDecimal, will throw a
     * ValueObjectConverterException. Otherwise, the value will be converted to a KualiDecimal and returned.
     *
     * @param propertyName  The name of the property being converted (used for exception handling).
     * @param stringDecimal The value being passed in, which will be converted to a KualiDecimal.
     * @return A valid KualiDecimal value.  If the method returns a value, then it will be a legitimate value.
     */
    private KualiDecimal convertToKualiDecimal(final String propertyName, final String stringDecimal) {
        if (StringUtils.isBlank(stringDecimal)) {
            return null;
        }
        final KualiDecimal kualiDecimal;
        try {
            kualiDecimal = new KualiDecimal(stringDecimal);
        } catch (final NumberFormatException e) {
            LOG.error("Failed to convert the value [{}] from field [{}] to a KualiDecimal.",
                    stringDecimal,
                    propertyName
            );
            addError(propertyName, KualiDecimal.class, stringDecimal, "Could not convert value to target type.");
            return null;
        }
        return kualiDecimal;
    }

    /**
     * This method converts a String into a boolean.
     *
     * @param stringBoolean
     * @return
     */
    private static boolean convertToLittleBoolean(final String stringBoolean) {
        if (StringUtils.isBlank(stringBoolean)) {
            return false;
        }
        if ("Y".equalsIgnoreCase(stringBoolean)) {
            return true;
        }
        if ("YES".equalsIgnoreCase(stringBoolean)) {
            return true;
        }
        if ("TRUE".equalsIgnoreCase(stringBoolean)) {
            return true;
        }
        if ("T".equalsIgnoreCase(stringBoolean)) {
            return true;
        }
        return "1".equalsIgnoreCase(stringBoolean);
    }

    /**
     * This method is used to apply any DataDictionary default value rules, if appropriate.
     * <p>
     * If the incoming value isn't blank, empty, or null, then nothing happens, and the method returns the incoming
     * value.
     * <p>
     * If the value is empty/blank/null, then the MaintenanceDocumentDictionaryService is consulted, and if there is a
     * defaultValue applied to this field, then its used in place of the empty/blank/null.
     *
     * @param propertyName The propertyName of the field (must match case exactly to work).
     * @param batchValue   The incoming value from the batch file, which at this point is always still a string.
     * @return If the original value is null/blank/empty, then if a default value is configured, that default value is
     * returned.  If no default value is configured, then the original value (trimmed) is returned.  Otherwise, if the
     * original incoming value is not empty/null/blank, then the original value is immediately returned.
     */
    private String applyDefaultValue(final String propertyName, final String batchValue) {
        //  short-circuit if value isn't empty/blank/null, as default wouldn't apply anyway
        if (StringUtils.isNotBlank(batchValue)) {
            return batchValue;
        }

        //  if its a string, we try to normalize it to null if empty/blank
        final String incomingValue;
        if (StringUtils.isBlank(batchValue)) {
            incomingValue = null;
        } else {
            incomingValue = StringUtils.trimToNull(batchValue);
        }

        //  apply the default value from DD if exists
        final String defaultValue = maintDocDDService.getFieldDefaultValue(BO_CLASS, propertyName);
        if (incomingValue == null && StringUtils.isNotBlank(defaultValue)) {
            LOG.info("Applied DD default value of '{}' to field [{}].", defaultValue, propertyName);
            return defaultValue;
        } else {
            return incomingValue == null ? "" : incomingValue;
        }
    }

    private void addError(
            final String propertyName,
            final Class<?> propertyClass,
            final String origValue,
            final String description
    ) {
        LOG.error("Failed conversion on field [{}] with value: '{}': {}", propertyName, origValue, description);
        errors.addError(customerName, propertyName, propertyClass, origValue, description);
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setMaintDocDDService(final MaintenanceDocumentDictionaryService maintDocDDService) {
        this.maintDocDDService = maintDocDDService;
    }

    public void setSequenceAccessorService(final SequenceAccessorService sequenceAccessorService) {
        this.sequenceAccessorService = sequenceAccessorService;
    }
}
